<?php

namespace BuddyBossApp\Integrations\WishlistMember;

use BuddyBossApp\InAppPurchases\Controller;
use BuddyBossApp\InAppPurchases\IntegrationAbstract;
use BuddyBossApp\InAppPurchases\Orders;

final class IAP extends IntegrationAbstract {

	private static $instance;

	/**
	 * @var array Wishlist LearnDash Setting
	 */
	private static $settings;

	/**
	 * MemberPress constructor.
	 */
	public function __construct() {
		//Using Singleton, see instance()
	}

	/**
	 * @return Main
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$className      = __CLASS__;
			self::$instance = new $className;
		}

		return self::$instance;
	}

	/**
	 * Overriding the parent(from base-class) function
	 *
	 * @param $integration_type
	 * @param $integration_label
	 */
	public function set_up( $integration_type, $integration_label ) {
		global $WishListMemberInstance;

		$this->integration_slug = Controller::$wishlist_membership_slug;

		parent::set_up( $integration_type, $integration_label );

		$this->item_label = __( 'Membership', 'buddyboss-app' );

		// Register Instance
		bbapp_iap()->integration[ $integration_type ] = $this::instance();

		$data                            = $WishListMemberInstance->get_option( 'learndash_settings' );
		self::$settings                  = is_array( $data ) ? $data : [];
		self::$settings['group_default'] = $WishListMemberInstance->get_option( 'wlm_ld_group_default' );
	}

	/**
	 * Trigger following code when BuddyBoss App order completed
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return false|mixed
	 */
	public function on_order_completed( $item_ids, $order ) {

		if ( ! class_exists( 'WishListMember' ) ) {
			Orders::instance()->add_history( $order["id"],'iap-error', __( "WishList Member plugin is not activated.", "buddyboss-app" ) );

			return false;
		}

		$trans_ids = array();
		foreach ( $item_ids as $item_identifier ) {
			$split    = explode( ':', $item_identifier );
			$level_id = $split[0];

			/**
			 * Grant the wishlist member level access
			 * Need to pass Cancelled:False to make exiting Cancelled level [ if already has ] active otherwise it'll create new.
			 */
			wlmapi_add_member_to_level( $level_id, array(
				'Users'     => array( $order->user_id ),
				'Cancelled' => false
			) );
			$txn_id = $this->get_member_level_txnid( $level_id, $order->user_id );
			if ( ! empty( $txn_id ) ) {
				$trans_ids[] = $txn_id;
				Orders::instance()->add_history( $order->id, 'info', __( "Created new transaction on Wishlist Member. Transaction ID is #$txn_id", 'buddyboss-app' ) );
			}
		}
		Orders::instance()->update_meta( $order->id, "_wishlist_member_trans_ids", serialize( $trans_ids ) );
	}

	/**
	 * Trigger following code when BuddyBoss App order Activated
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return false|mixed
	 */
	public function on_order_activate( $item_ids, $order ) {
		return $this->on_order_completed( $item_ids, $order );
	}

	/**
	 * Trigger following code when BuddyBoss App order cancelled
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return false
	 */
	public function on_order_cancelled( $item_ids, $order ) {

		if ( ! class_exists( 'WishListMember' ) ) {
			Orders::instance()->add_history( $order["id"],'iap-error', __( "Wishlist Member plugin is not activated.", "buddyboss-app" ) );

			return false;
		}
		$trans_ids = unserialize( Orders::instance()->get_meta( $order->id, "_wishlist_member_trans_ids" ) );

		foreach ( $item_ids as $item_identifier ) {
			$split    = explode( ':', $item_identifier );
			$level_id = $split[0];
			$txn_id   = $this->get_member_level_txnid( $level_id, $order->user_id );
			if ( ! empty( $txn_id ) && in_array( $txn_id, $trans_ids, true ) ) {
				$member = wlmapi_update_level_member_data( $level_id, $order->user_id, array( 'Cancelled' => true ) );
				if ( isset( $member['success'] ) && $member['success'] ) {
					Orders::instance()->add_history( $order->id, 'info', __( "Access revoked for Wishlist Member. Transaction ID is #$txn_id", 'buddyboss-app' ) );
				}
			}
		}
	}

	/**
	 * Trigger following code when BuddyBoss App order expired
	 *
	 * @param $item_ids
	 * @param $order
	 */
	public function on_order_expired( $item_ids, $order ) {
		$this->on_order_cancelled( $item_ids, $order );
	}

	/**
	 * Handle bbapp ajax iap_linking_options for this integration
	 *
	 * @param $results levels data
	 * @param $search : Search term
	 *
	 * @return mixed
	 */
	public function iap_linking_options( $results ) {
		$levels = class_exists( '\WishListMember_Level' ) ? \WishListMember_Level::GetAllLevels(true) : \WishListMember\Level::GetAllLevels( true );
		foreach ( $levels as $level ) {
			$level = (object) $level;
			$results[] = [
				'id'   => $level->ID . ':' . $level->name,
				'text' => $level->name . ':' . $level->ID,
			];
		}

		return $results;
	}

	/**
	 * Handle bbapp ajax iap_integration_ids for this integration
	 * it's return items label with id.
	 *
	 * @param $results levels data
	 * @param $integration_ids selected integration id
	 *
	 * @return mixed
	 */
	public function iap_integration_ids( $results, $integration_ids ) {
		$levels = class_exists( '\WishListMember_Level' ) ? \WishListMember_Level::GetAllLevels(true) : \WishListMember\Level::GetAllLevels( true );

		foreach ( $integration_ids as $key => $integration_id ) {
			$results[ $key ]['id']   = $integration_id;
			$results[ $key ]['text'] = "Not found: $integration_id";

			foreach ( $levels as $level ) {
				$level = (object) $level;

				/**
				 * Split[ level id, level name ];
				 */
				$split = explode( ':', $integration_id );
				if ( $level->ID === (int) $split[0] ) {
					$results[ $key ]['text'] = $level->name;
					break;
				}
			}
		}

		return $results;
	}

	/**
	 * Get item edit link to show on order page.
	 *
	 * @param $link
	 * @param $item_id
	 *
	 * @return string
	 */
	public function item_id_permalink( $link, $item_id ) {
		return 'admin.php?page=WishListMember&wl=setup%2Flevels#levels_access-' . $item_id;
	}

	/**
	 * Check given post has given membership configured or not
	 *
	 * @param $has_access
	 * @param $forItemId
	 * @param $integration_item_id
	 *
	 * @return bool
	 */
	public function is_purchase_available( $has_access, $forItemId, $integration_item_id ) {
		$post_levels = self::get_level_ids_for_post( $forItemId, [ $integration_item_id ] );

		return in_array( $integration_item_id, $post_levels );
	}

	/**
	 * Check given integration item has access.
	 *
	 * @param $item_ids
	 * @param $user_id
	 *
	 * @return false
	 */
	function has_access( $item_ids, $user_id ) {
		if ( is_admin() ) {
			return true;
		}
		$has_access = false;
		foreach ( $item_ids as $item_identifier ) {
			$split      = explode( ':', $item_identifier );
			$level_id   = $split[0];
			$has_access = wlmapi_is_user_a_member( $level_id, $user_id );
			if ( $has_access ) {
				break;
			}
		}

		return $has_access;
	}

	/**
	 * Get txnid for user level
	 *
	 * @param $level_id
	 * @param $user_id
	 *
	 * @return string
	 */
	private function get_member_level_txnid( $level_id, $user_id ) {
		$txn_id = '';
		$member = wlmapi_get_level_member_data( $level_id, $user_id );
		if ( isset( $member['success'] ) && $member['success'] ) {
			$txn_id = ( isset( $member['member']['level'] ) ) ? $member['member']['level']->TxnID : '';
		}

		return $txn_id;
	}

	/**
	 * get levels available for post
	 * Wishlist doesn't have function to get levels id for post.
	 *
	 * @param int $post_id
	 * @param array $level_ids
	 *
	 * @return array
	 */
	static public function get_level_ids_for_post( $post_id, $level_ids = [] ) {

		$transient_key  = 'wl_post_level_' . $post_id . '_' . implode( ',', $level_ids );
		$post_level_ids = get_transient( $transient_key );
		if ( false === $post_level_ids ) {
			$post_level_ids = [];

			if ( empty( $level_ids ) ) {
				$levels = wlmapi_the_levels();
				if ( ! empty( $levels['success'] ) && isset( $levels['levels']['level'] ) ) {
					$level_ids = wp_list_pluck( $levels['levels']['level'], 'id' );
				}
			}


			if ( ! empty( $level_ids ) ) {
				foreach ( $level_ids as $level_id ) {
					if ( ! empty( self::$settings['level'][ $level_id ] ) && isset( self::$settings['level'][ $level_id ]['add'] ) ) {
						$setting = self::$settings['level'][ $level_id ]['add'];
						if ( in_array( $post_id, $setting['apply_course'] ) ) {
							$post_level_ids[] = $level_id;
							break;
						}

						$groups_ids = learndash_get_course_groups( $post_id );
						if ( ! empty( $groups_ids ) && ! empty( $setting['apply_group'] ) ) {
							$diff = array_intersect( $groups_ids, $setting['apply_group'] );
							if ( ! empty( $diff ) ) {
								$post_level_ids[] = $level_id;
								break;
							}
						}
					}
				}
			}

			set_transient( $transient_key, $post_level_ids, 5 * MINUTE_IN_SECONDS );
		}

		return ( ! empty( $post_level_ids ) ) ? $post_level_ids : [];
	}
}
