<?php
/**
 * Holds WishList Member functionality related the Access Controls.
 *
 * @package BuddyBossApp\Integrations\WishlistMember
 */

namespace BuddyBossApp\Integrations\WishlistMember;

use BuddyBossApp\AccessControls\TmpItems;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * WishList Member class for Access Controls.
 */
class AccessControls extends \BuddyBossApp\AccessControls\Integration_Abstract {

	/**
	 * Condition name.
	 *
	 * @var string $condition_name condition name.
	 */
	private $condition_name = 'wishlist-member';

	/**
	 * Function to set up the conditions.
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed|void
	 */
	public function setup() {
		$this->register_condition(
			array(
				'condition'              => $this->condition_name,
				'items_callback'         => array( $this, 'wishlist_member_items_callback' ),
				'item_callback'          => array( $this, 'wishlist_member_item_callback' ),
				'users_callback'         => array( $this, 'wishlist_member_users_callback' ),
				'labels'                 => array(
					'condition_name'          => __( 'WishList Member', 'buddyboss-app' ),
					'item_singular'           => __( 'Membership', 'buddyboss-app' ),
					'member_of_specific_item' => __( 'Has specific membership', 'buddyboss-app' ),
					'member_of_any_items'     => __( 'Has any membership', 'buddyboss-app' ),
				),
				'support_any_items'      => true,
				'has_any_items_callback' => array( $this, 'has_any_items_callback' ),
			)
		);

		$this->load_hooks();
	}

	/**
	 * Function to load all hooks of this condition.
	 *
	 * @since 1.5.2.1
	 */
	public function load_hooks() {
		add_action( 'wishlistmember_add_user_levels', array( $this, 'bbapp_wlm_add_user_levels' ), 10, 3 );
		add_action( 'wishlistmember_remove_user_levels', array( $this, 'bbapp_wlm_remove_user_levels' ), 10, 3 );
		add_action( 'wishlistmember_uncancel_user_levels', array( $this, 'bbapp_wlm_uncancel_user_levels' ), 10, 2 );
		add_action( 'wishlistmember_cancel_user_levels', array( $this, 'bbapp_wlm_cancel_user_levels' ), 10, 2 );
		add_action( 'wishlistmember_update_option_wpm_levels', array( $this, 'bbapp_wlm_remove_levels' ), 10, 2 );
	}

	/**
	 * Items callback method.
	 *
	 * @param string $search Search the condition.
	 * @param int    $page   Page number.
	 * @param int    $limit  Limit the items to be fetched.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function wishlist_member_items_callback( $search = '', $page = 1, $limit = 20 ) {
		$list   = array();
		$levels = class_exists( '\WishListMember_Level' ) ? \WishListMember_Level::GetAllLevels( true ) : \WishListMember\Level::GetAllLevels( true );

		if ( ! empty( $levels ) ) {
			foreach ( $levels as $level ) {
				$level              = (object) $level;
				$list[ $level->id ] = array(
					'id'   => $level->id,
					'name' => $level->name,
					'link' => admin_url( "admin.php?page=WishListMember&wl=members/manage&level={$level->id}" ),
				);
			}
		}

		return $this->paginate_items_list( $list, $page, $limit, $search );
	}

	/**
	 * Item callback method.
	 *
	 * @param int $item_value Item value of condition.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array|false
	 */
	public function wishlist_member_item_callback( $item_value ) {
		$membership = wlmapi_get_level( $item_value );

		if ( empty( $membership ) || empty( $membership['level']['id'] ) || empty( $membership['level']['name'] ) ) {
			return false;
		}

		return array(
			'id'   => $membership['level']['id'],
			'name' => $membership['level']['name'],
			'link' => admin_url( "admin.php?page=WishListMember&wl=members/manage&level={$membership['level']['id']}" ),
		);
	}

	/**
	 * Users callback method.
	 *
	 * @param array $data     condition data.
	 * @param int   $page     current page number.
	 * @param int   $per_page limit.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function wishlist_member_users_callback( $data, $page = 1, $per_page = 10 ) {
		global $WishListMemberInstance; //phpcs:ignore WordPress.NamingConventions.ValidVariableName.VariableNotSnakeCase

		$item_value    = ( ! empty( $data['item_value'] ) ) ? $data['item_value'] : '';
		$group_id      = ( ! empty( $data['group_id'] ) ) ? $data['group_id'] : 0;
		$rounds_count  = ( ! empty( $data['rounds_count'] ) ) ? $data['rounds_count'] : 0;
		$transient_key = "access_ctrls_wlm_level_condition_jb_data_{$group_id}";

		if ( 1 === $rounds_count ) {
			$condition_data = array(
				'prepare_page'     => 1,
				'is_data_prepared' => false,
			);
		} else {
			$condition_data = get_transient( $transient_key );
		}

		if ( ! $condition_data['is_data_prepared'] ) {

			if ( empty( $item_value ) ) {
				$item_value = null;
			}

			if ( method_exists( $WishListMemberInstance, 'ActiveMemberIDs' ) ) { //phpcs:ignore WordPress.NamingConventions.ValidVariableName.VariableNotSnakeCase
				$wlm_user_ids = $WishListMemberInstance->ActiveMemberIDs( $item_value, false, false ); //phpcs:ignore WordPress.NamingConventions.ValidVariableName.VariableNotSnakeCase
			} else {
				$wlm_user_ids = $WishListMemberInstance->active_member_ids( $item_value, false, false ); //phpcs:ignore WordPress.NamingConventions.ValidVariableName.VariableNotSnakeCase
			}

			if ( ! empty( $wlm_user_ids ) ) {
				$user_ids = ! empty( $wlm_user_ids ) ? $this->paginate_users_list( array_unique( $wlm_user_ids ), $condition_data['prepare_page'], 500 ) : array();

				if ( ! empty( $user_ids ) ) {
					$items = array();

					foreach ( $user_ids as $user_id ) {
						$items[] = array(
							'item_type'         => "wlm_levels_{$group_id}",
							'item_id'           => $user_id,
							'secondary_item_id' => $item_value,
						);
					}

					$insert = TmpItems::instance()->insert_items( $items );

					$condition_data['prepare_page'] ++;

					if ( empty( $insert ) ) {
						$this->return_error( __( 'Error while preparing members data for WishList Member.', 'buddyboss-app' ) );
					}
				} else {
					$condition_data['is_data_prepared'] = true;
				}

				set_transient( $transient_key, $condition_data, DAY_IN_SECONDS );

				return $this->return_wait( __( 'Preparing WishList Member Data', 'buddyboss-app' ) );
			}
		}

		/**
		 * Start Sending Users Data Once Tmp Collection Preparations has done.
		 * Once the Preparations for any of specific memberships are done above the execution will reach here.
		 */

		$items = TmpItems::instance()->get_items(
			array(
				'include_item_types' => array( "wlm_levels_{$group_id}" ),
				'per_page'           => $per_page,
				'page'               => $page,
			)
		);

		if ( ! empty( $items ) ) {
			$user_ids = array();
			foreach ( $items['results'] as $item ) {
				$user_ids[] = $item->item_id;
			}

			return $this->return_users( $user_ids );
		}

		return $this->return_error( __( 'Invalid sub condition.', 'buddyboss-app' ) );
	}

	/**
	 * Function to check if user has any condition.
	 *
	 * @param int $user_id User id to check.
	 *
	 * @since 1.5.2.1
	 *
	 * @return bool
	 */
	public function has_any_items_callback( $user_id ) {
		$levels  = wlmapi_get_member_levels( $user_id );
		$has_any = false;

		if ( ! empty( $levels ) ) {
			foreach ( $levels as $level ) {
				if ( is_array( $level->Status ) && in_array( 'Active', $level->Status, true ) ) { //phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
					$has_any = true;
					break;
				}
			}
		}

		return $has_any;
	}

	/**
	 * This function add user to access control.
	 *
	 * @param int   $id             user id.
	 * @param array $new_levels     levels ids.
	 * @param array $removed_levels levels ids.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_wlm_add_user_levels( $id, $new_levels, $removed_levels ) {
		if ( empty( $id ) || empty( $new_levels ) ) {
			return;
		}

		foreach ( $new_levels as $new_level ) {
			$this->condition_add_user( $id, $this->condition_name, $new_level );
		}
	}

	/**
	 * This function remove user from access control.
	 *
	 * @param int   $id             user id.
	 * @param array $removed_levels levels ids.
	 * @param array $new_levels     levels ids.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_wlm_remove_user_levels( $id, $removed_levels, $new_levels ) {
		if ( empty( $id ) || empty( $removed_levels ) ) {
			return;
		}
		foreach ( $removed_levels as $removed_level ) {
			$this->condition_remove_user( $id, $this->condition_name, $removed_level );
		}
	}

	/**
	 * This function add user to access control.
	 *
	 * @param int   $id     user id.
	 * @param array $levels levels ids.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_wlm_uncancel_user_levels( $id, $levels ) {
		if ( empty( $id ) || empty( $levels ) ) {
			return;
		}
		foreach ( $levels as $level ) {
			$this->condition_add_user( $id, $this->condition_name, $level );
		}
	}

	/**
	 * This function add user to access control.
	 *
	 * @param int   $id     user id.
	 * @param array $levels levels ids.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_wlm_cancel_user_levels( $id, $levels ) {
		if ( empty( $id ) || empty( $levels ) ) {
			return;
		}
		foreach ( $levels as $level ) {
			$this->condition_remove_user( $id, $this->condition_name, $level );
		}
	}

	/**
	 * Function will remove level from access groups when remove level from wishlist admin.
	 *
	 * @param array $value     New value which will update in DB.
	 * @param array $old_value Old value which will fetch from DB.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_wlm_remove_levels( $value, $old_value ) {
		if ( ! empty( $old_value ) ) {
			if ( ! empty( $value ) ) {
				$deleted_level = array_diff( array_keys( $old_value ), array_keys( $value ) );
			} else {
				$deleted_level = array_keys( $old_value );
			}

			if ( ! empty( $deleted_level ) ) {
				foreach ( $deleted_level as $deleted_level_id ) {
					$this->condition_item_deleted( $this->condition_name, $deleted_level_id );
					$this->condition_remove_all_users( $this->condition_name, $deleted_level_id );
					bb_access_recalculate_member_for_has_any_membership_group( $this->condition_name );
				}
			}
		}
	}
}
