<?php

namespace BuddyBossApp\Integrations\WcMembership;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\InAppPurchases\Controller;
use BuddyBossApp\InAppPurchases\IntegrationAbstract;
use BuddyBossApp\InAppPurchases\Orders;

//Woo Membership Integration for InAppPurchases.
final class IAP extends IntegrationAbstract {

	private static $instance = null;

	private function __construct() {
		// ... leave empty, see Singleton below
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return IAP|null
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$className      = __CLASS__;
			self::$instance = new $className;
		}

		return self::$instance;
	}

	/**
	 * Overriding the parent(from base-class) function
	 *
	 * @param $integration_type
	 * @param $integration_label
	 */
	public function set_up( $integration_type, $integration_label ) {

		$this->integration_slug = Controller::$wc_membership_slug;

		parent::set_up( $integration_type, $integration_label );

		$this->item_label = __( 'Membership', 'buddyboss-app' );

		// Register Instance
		bbapp_iap()->integration[ $integration_type ] = $this::instance();
	}

	/**
	 * Below function get triggers when(hook) order is completed.
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return mixed
	 */
	public function on_order_completed( $item_ids, $order ) {

		if ( ! function_exists( 'wc_memberships' ) ) {
			Orders::instance()->add_history( $order["id"], 'iap-error', __( "WooCommerce Memberships plugin is not activated.", "buddyboss-app" ) );

			return false;
		}

		$membership_ids = array();
		foreach ( $item_ids as $item_id ) {
			$item_id       = $this->get_membership_id( $item_id );
			$membership_id = $this->update_membership_status( $order->user_id, $item_id, "subscribed", $order );

			$membership_ids[] = $membership_id;

			Orders::instance()->add_history( $order->id, 'info', sprintf( __( "Created new membership for WooCommerce Memberships. Membership ID is #%s", 'buddyboss-app' ), $membership_id ) );

		}
		Orders::instance()->update_meta( $order->id, "_wc_user_membership_ids", serialize( $membership_ids ) );
	}

	/**
	 * Below function get triggers when(hook) order is activated.
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return mixed
	 */
	public function on_order_activate( $item_ids, $order ) {
		return $this->on_order_completed( $item_ids, $order );
	}

	/**
	 * Below function get triggers when(hook) order is cancelled.
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return mixed
	 */
	public function on_order_cancelled( $item_ids, $order ) {
		foreach ( $item_ids as $item_id ) {
			$item_id       = $this->get_membership_id( $item_id );
			$membership_id = $this->update_membership_status( $order->user_id, $item_id, "expired", $order );
			Orders::instance()->add_history( $order->id, 'info', sprintf( __( "Access revoked for WooCommerce Memberships. Membership ID is #%s", 'buddyboss-app' ), $membership_id ) );
		}
	}

	/**
	 * Below function get triggers when(hook) order is expired.
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return mixed
	 */
	public function on_order_expired( $item_ids, $order ) {
		// NOTE : Similar to onOrderCancelled($order) until something needs to be changed?
		$this->on_order_cancelled( $item_ids, $order );
	}

	/**
	 * Updates membership status smartly.
	 *
	 * @param $user_id
	 * @param $membership_plan_id
	 * @param $status
	 * @param $order
	 *
	 * @return boolean
	 */
	public function update_membership_status( $user_id, $membership_plan_id, $status, $order ) {

		$user_memberships = wc_memberships_get_user_memberships( $user_id );

		$user_membership = false;

		foreach ( $user_memberships as $membership ) {
			$plan_id = $membership->plan_id;
			if ( $plan_id == $membership_plan_id ) {
				$user_membership = $membership;
			}
		}

		// if no exiting membership found create new.
		if ( empty( $user_membership ) ) {
			$user_membership = wc_memberships_create_user_membership( array(
				'user_id' => $user_id,
				'plan_id' => $membership_plan_id,
			), 'renew' );
		}

		$message = sprintf( __( "Membership get %s by BuddyBoss App In-App Purchase. Purchase ID #%s", 'buddyboss-app' ), $status, $order->id );

		// update the status.
		if ( 'subscribed' === $status ) {
			$user_membership->update_status( 'pending' );
			$user_membership->activate_membership( $message );
			$user_membership->set_end_date( false ); // remove expire & end date. as it's all handle by inapp purchases.
		}

		if ( 'expired' === $status ) {
			wc_memberships()->get_user_memberships_instance()->set_membership_status_transition_note( $message );
			$user_membership->expire_membership();
		} else {
			$user_membership->update_status( $status, $message );
		}

		return $user_membership->id;

	}

	function iap_linking_options( $results ) {
		return $results;
	}

	function iap_integration_ids( $results, $integration_ids ) {
		return $results;
	}

	function item_id_permalink( $link, $item_id ) {
		return "post.php?post=$item_id&action=edit";
	}

	function is_purchase_available( $is_available, $item_id, $integration_item_id ) {
		return $is_available;
	}

	/**
	 * Check given integration item has access.
	 *
	 * @param $item_ids
	 * @param $user_id
	 *
	 * @return false
	 */
	function has_access( $item_ids, $user_id ) {
		if ( is_admin() ) {
			return true;
		}
		$has_access = false;

		foreach ( $item_ids as $item_identifier ) {
			$level_id = $this->get_membership_id( $item_identifier );
			$plan     = wc_memberships_is_user_active_or_delayed_member( $user_id, $level_id );
			if ( $plan ) {
				$has_access = true;
				break;
			}
		}
		return $has_access;
	}

	/**
	 * Function to get membership id.
	 *
	 * @param string|int $item_id Item id.
	 *
	 * @since 1.6.8
	 *
	 * @return string
	 */
	private function get_membership_id( $item_id ) {
		$split = explode( ':', $item_id );

		return ! empty( $split[0] ) ? $split[0] : $item_id;
	}
}
