<?php
/**
 * Holds WC Membership functionality related to Access Controls.
 *
 * @package BuddyBossApp\Integrations\WcMembership
 */

namespace BuddyBossApp\Integrations\WcMembership;

use BuddyBossApp\AccessControls\TmpItems;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 *  WC Membership class for the Access Controls.
 */
class AccessControls extends \BuddyBossApp\AccessControls\Integration_Abstract {

	/**
	 * Condition name.
	 *
	 * @var string $condition_name condition name.
	 */
	private $condition_name = 'wc-membership';

	/**
	 * Function to set up the conditions.
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed|void
	 */
	public function setup() {
		$this->register_condition(
			array(
				'condition'              => $this->condition_name,
				'items_callback'         => array( $this, 'woocommerce_membership_items_callback' ),
				'item_callback'          => array( $this, 'woocommerce_membership_item_callback' ),
				'users_callback'         => array( $this, 'woocommerce_membership_users_callback' ),
				'labels'                 => array(
					'condition_name'          => __( 'WooCommerce Memberships', 'buddyboss-app' ),
					'item_singular'           => __( 'Membership', 'buddyboss-app' ),
					'member_of_specific_item' => __( 'Has specific membership', 'buddyboss-app' ),
					'member_of_any_items'     => __( 'Has any membership', 'buddyboss-app' ),
				),
				'support_any_items'      => true,
				'has_any_items_callback' => array( $this, 'has_any_items_callback' ),
			)
		);

		$this->load_hooks();
	}

	/**
	 * Hooks here
	 *
	 * @since 1.5.2.1
	 */
	public function load_hooks() {
		add_action(
			'wc_memberships_user_membership_created',
			array(
				$this,
				'bbapp_wc_memberships_user_membership_created',
			),
			10,
			2
		);
		add_action(
			'wc_memberships_user_membership_activated',
			array(
				$this,
				'bbapp_wc_memberships_user_membership_activated',
			),
			10,
			3
		);
		add_action(
			'wc_memberships_user_membership_expired',
			array(
				$this,
				'bbapp_wc_memberships_user_membership_expired',
			),
			10,
			1
		);

		add_action( 'wc_memberships_save_meta_box', array( $this, 'bbapp_wc_memberships_save_meta_box' ), 10, 4 );
		add_action(
			'wc_memberships_user_membership_cancelled',
			array(
				$this,
				'bbapp_wc_memberships_user_membership_cancelled',
			),
			10,
			1
		);
		add_action(
			'wc_memberships_user_membership_deleted',
			array(
				$this,
				'bbapp_wc_memberships_user_membership_deleted',
			),
			10,
			1
		);
		add_action(
			'wc_memberships_user_membership_saved',
			array(
				$this,
				'bbapp_wc_memberships_user_membership_saved',
			),
			10,
			2
		);
	}

	/**
	 * Items callback method.
	 *
	 * @param string $search Search the condition.
	 * @param int    $page   Page number.
	 * @param int    $limit  Limit the items to be fetched.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function woocommerce_membership_items_callback( $search = '', $page = 1, $limit = 20 ) {
		$list     = array();
		$products = wc_memberships_get_membership_plans();

		if ( ! empty( $products ) ) {
			foreach ( $products as $product ) {
				$list[ $product->id ] = array(
					'id'   => $product->id,
					'name' => $product->name,
					'link' => admin_url( "post.php?post={$product->id}&action=edit" ),
				);
			}
		}

		return $this->paginate_items_list( $list, $page, $limit, $search );
	}

	/**
	 * Item callback method.
	 *
	 * @param int $item_value Item value of condition.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array|false
	 */
	public function woocommerce_membership_item_callback( $item_value ) {
		$membership = wc_memberships_get_membership_plan( $item_value );

		if ( empty( $membership ) || false === $membership ) {
			return false;
		}

		return array(
			'id'   => $membership->get_id(),
			'name' => $membership->get_name(),
			'link' => admin_url( "post.php?post={$membership->get_id()}&action=edit" ),
		);
	}

	/**
	 * Users callback method.
	 *
	 * @param array $data     condition data.
	 * @param int   $page     current page number.
	 * @param int   $per_page limit.
	 *
	 * @since 1.5.2.1
	 * @return array
	 */
	public function woocommerce_membership_users_callback( $data, $page = 1, $per_page = 10 ) {
		$user_ids      = array();
		$offset        = ( $page - 1 ) * $per_page;
		$no_paging     = ( - 1 === $per_page ) ? true : false; // When we fetch all posts we need to set it as false.
		$sub_condition = ( ! empty( $data['sub_condition'] ) ) ? $data['sub_condition'] : '';
		$item_value    = ( ! empty( $data['item_value'] ) ) ? $data['item_value'] : '';
		$group_id      = ( ! empty( $data['group_id'] ) ) ? $data['group_id'] : 0;
		$rounds_count  = ( ! empty( $data['rounds_count'] ) ) ? $data['rounds_count'] : 0;
		$transient_key = "access_ctrls_wc_membership_condition_jb_data_{$group_id}";

		if ( 'specific' === $sub_condition ) {
			if ( empty( $item_value ) ) {
				return $user_ids;
			}

			$membership_plan = wc_memberships_get_membership_plan( $item_value );

			if ( $membership_plan instanceof \WC_Memberships_Membership_Plan ) {
				$memberships = $membership_plan->get_memberships(
					array(
						'post_status' => array( 'wcm-active' ),
						'numberposts' => $per_page,
						'nopaging'    => $no_paging,
						'offset'      => $offset,
					)
				);

				$user_ids = ( ! empty( $memberships ) ) ? wp_list_pluck( $memberships, 'user_id' ) : array();

				return $this->return_users( $user_ids );
			}
		} elseif ( 'any' === $sub_condition ) {

			// Check if it's a first round.
			if ( 1 === $rounds_count ) {
				$condition_data                       = array(
					'prepare_page'           => 1,
					'is_data_prepared'       => false,
					'pending_membership_ids' => $this->wc_membership_get_all_condition_ids(),
				);
				$condition_data['current_membership'] = array_key_first( $condition_data['pending_membership_ids'] );
			} else {
				$condition_data = get_transient( $transient_key );
			}
			/**
			 * Preparing the data into tmp items collections.
			 */
			if ( ! $condition_data['is_data_prepared'] ) {

				$membership_plan = wc_memberships_get_membership_plan( $item_value );

				if ( $membership_plan instanceof \WC_Memberships_Membership_Plan ) {
					$memberships = $membership_plan->get_memberships(
						array(
							'post_status' => array( 'wcm-active' ),
							'post_parent' => $condition_data['current_membership'],
							'numberposts' => $per_page,
							'nopaging'    => false,
							'offset'      => ( ! empty( $condition_data['prepare_page'] ) ) ? ( $condition_data['prepare_page'] - 1 ) * $per_page : 0,
						)
					);

					$user_ids = ( ! empty( $memberships ) ) ? $this->return_users( wp_list_pluck( $memberships, 'user_id' ) ) : array();

					if ( ! empty( $user_ids ) && ! empty( $user_ids['data'] ) ) {
						$items = array();

						foreach ( $user_ids['data'] as $user_id ) {
							$items[] = array(
								'item_type'         => "wc_memberships_{$group_id}",
								'item_id'           => $user_id,
								'secondary_item_id' => $condition_data['current_membership'],
							);
						}

						$insert = TmpItems::instance()->insert_items( $items );

						$condition_data['prepare_page'] ++;

						if ( empty( $insert ) ) {
							return $this->return_error( __( 'Error while preparing members data for LearnDash.', 'buddyboss-app' ) );
						}
					} else { // when all users are processed.

						unset( $condition_data['pending_membership_ids'][ $condition_data['current_membership'] ] ); // remove from pending membership id.

						if ( ! empty( $condition_data['pending_membership_ids'] ) ) {
							$condition_data['current_membership'] = array_key_first( $condition_data['pending_membership_ids'] ); // get next membership to process.
							$condition_data['prepare_page']       = 1; // reset pagination to 1.
						} else {
							// If pending memberships found empty finish the job.
							$condition_data['is_data_prepared'] = true;
						}
					}

					set_transient( $transient_key, $condition_data, DAY_IN_SECONDS );

					return $this->return_wait( __( 'Preparing WC Members Data', 'buddyboss-app' ) );
				} else {
					return $this->return_error( __( 'Error while getting users results from WC membership function.', 'buddyboss-app' ) );
				}
			}

			/**
			 * Start Sending Users Data Once Tmp Collection Preparations has done.
			 * Once the Preparations for any of specific memberships are done above the execution will reach here.
			 */

			$items = TmpItems::instance()->get_items(
				array(
					'include_item_types' => array( "wc_memberships_{$group_id}" ),
					'per_page'           => $per_page,
					'page'               => $page,
				)
			);

			$user_ids = array();

			foreach ( $items['results'] as $item ) {
				$user_ids[] = $item->item_id;
			}

			return $this->return_users( $user_ids );
		}

		return $this->return_error( __( 'Invalid sub condition.', 'buddyboss-app' ) );
	}

	/**
	 * Returns all the WC membership plan ids.
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed
	 */
	public function wc_membership_get_all_condition_ids() {
		$membership_plans = wc_memberships_get_membership_plans();
		$return           = array();

		if ( ! empty( $membership_plans ) ) {
			foreach ( $membership_plans as $membership_plan ) {
				if ( $membership_plan instanceof \WC_Memberships_Membership_Plan ) {
					$return[ $membership_plan->id ] = $membership_plan->id;
				}
			}
		}

		return $return;
	}

	/**
	 * Function to check if user has any condition.
	 *
	 * @param int $user_id User id to check.
	 *
	 * @since 1.5.2.1
	 *
	 * @return bool
	 */
	public function has_any_items_callback( $user_id ) {
		$levels = wc_memberships_get_user_active_memberships( $user_id );

		return ( ! empty( $levels ) ) ? true : false;
	}

	/**
	 * Fires after a user has been granted membership access.
	 *
	 * This action hook is similar to `wc_memberships_user_membership_saved`
	 * but doesn't fire when memberships are manually created from admin.
	 *
	 * @param \WC_Memberships_Membership_Plan $membership_plan    the plan that user was granted access to.
	 * @param array                           $args               array of User Membership arguments.
	 *                                                            {.
	 *
	 * @type int                              $user_id            the user ID the membership is assigned to
	 * @type int                              $user_membership_id the user membership ID being saved
	 * @type bool                             $is_update          whether this is a post update or a newly created membership
	 *                                                            }
	 *
	 * @since 1.5.2.1
	 * @see   \WC_Memberships_User_Memberships::save_user_membership()
	 */
	public function bbapp_wc_memberships_user_membership_created( $membership_plan, $args ) {
		if ( empty( $args['user_id'] ) || empty( $membership_plan->get_id() ) ) {
			return;
		}

		$this->condition_add_user( $args['user_id'], $this->condition_name, $membership_plan->get_id() );
	}

	/**
	 * Upon User Membership activation or re-activation.
	 *
	 * @param \WC_Memberships_User_Membership $user_membership the membership object.
	 * @param bool                            $was_paused      whether this is a reactivation of a paused membership.
	 * @param string                          $previous_status the status the membership was set before activation.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_wc_memberships_user_membership_activated( $user_membership, $was_paused, $previous_status ) {

		if ( empty( $user_membership ) ) {
			return;
		}
		if ( $user_membership instanceof \WC_Memberships_User_Membership ) {
			$this->condition_add_user( $user_membership->get_user_id(), $this->condition_name, $user_membership->get_plan_id() );
		}
	}

	/**
	 * Upon User Membership expiration.
	 *
	 * @param int $user_membership_id the expired user membership ID.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_wc_memberships_user_membership_expired( $user_membership_id ) {
		$user_membership = wc_memberships_get_user_membership( $user_membership_id );
		if ( empty( $user_membership ) ) {
			return;
		}

		$this->condition_remove_user( $user_membership->get_user_id(), $this->condition_name, $user_membership->get_plan_id() );
	}

	/**
	 * Fires upon saving a meta box data for a post object.
	 *
	 * @param array    $_post   The Post data.
	 * @param string   $id      The meta box id.
	 * @param int      $post_id WP_Post id.
	 * @param \WP_Post $post    WP_Post object.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function bbapp_wc_memberships_save_meta_box( $_post, $id, $post_id, $post ) {
		if ( empty( $post->post_author ) || empty( $post->post_parent ) ) {
			return;
		}
		$this->condition_add_user( $post->post_author, $this->condition_name, $post->post_parent );
	}

	/**
	 * Upon User Membership cancellation.
	 *
	 * @param \WC_Memberships_User_Membership $user_membership user membership being cancelled.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_wc_memberships_user_membership_cancelled( $user_membership ) {
		if ( empty( $user_membership ) ) {
			return;
		}
		if ( $user_membership instanceof \WC_Memberships_User_Membership ) {
			$this->condition_remove_user( $user_membership->get_user_id(), $this->condition_name, $user_membership->get_plan_id() );
		}
	}

	/**
	 * Fires before a user membership is deleted.
	 *
	 * @param \WC_Memberships_User_Membership $user_membership membership object.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_wc_memberships_user_membership_deleted( $user_membership ) {
		if ( empty( $user_membership ) ) {
			return;
		}
		if ( $user_membership instanceof \WC_Memberships_User_Membership ) {
			$this->condition_remove_user( $user_membership->get_user_id(), $this->condition_name, $user_membership->get_plan_id() );
		}
	}

	/**
	 * Fires after a user has been granted membership access.
	 *
	 * @param \WC_Memberships_Membership_Plan $user_membership_plan The plan that user was granted access to.
	 * @param array                           $user_membership_data User membership data.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function bbapp_wc_memberships_user_membership_saved( $user_membership_plan, $user_membership_data ) {
		if ( empty( $user_membership_data ) ) {
			return;
		}
		$this->condition_add_user( $user_membership_data['user_id'], $this->condition_name, $user_membership_data['user_membership_id'] );
	}

	/**
	 * Function will remove member/wc membership from access groups when remove membership from admin.
	 *
	 * @param string   $new_status New post status.
	 * @param string   $old_status Old post status.
	 * @param \WP_Post $post       Post object.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_remove_wc_memberships( $new_status, $old_status, $post ) {
		if ( 'publish' !== $new_status && 'wc_membership_plan' === $post->post_type ) {
			$this->condition_item_deleted( $this->condition_name, $post->ID );
			$this->condition_remove_all_users( $this->condition_name, $post->ID );
			bb_access_recalculate_member_for_has_any_membership_group( $this->condition_name );
		}
	}
}
