<?php
/**
 * WPML Support Integration file
 *
 * @package BuddyBossApp\Integrations\WPML
 */

namespace BuddyBossApp\Integrations\WPML;

use BuddyBossApp\Tools\Logger;
use WP_REST_Request;

/**
 * Class for handling WPML integration with BuddyBoss App REST API
 *
 * This class provides functionality to translate REST API requests and responses
 *
 * @since   2.4.10
 * @package BuddyBossApp\Integrations\WPML
 */
class WPMLSupport {

	/**
	 * Class instance.
	 *
	 * @var WPMLSupport
	 */
	private static $instance;

	/**
	 * Translation array for posts
	 *
	 * @var array
	 */
	private $translations = array();

	/**
	 * WPMLSupport constructor.
	 *
	 * This method is private to enforce the Singleton pattern.
	 *
	 * @since 2.4.10
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * This method returns the singleton instance of the WPMLSupport class.
	 *
	 * @since 2.4.10
	 *
	 * @return WPMLSupport
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Load the class and its hooks
	 *
	 * This method is called to register the module and its hooks.
	 *
	 * @since 2.4.10
	 */
	public function load() {
		add_action( 'init', array( $this, 'hook' ) );
	}

	/**
	 * Register main hooks
	 *
	 * This method is called to register the main hooks for WPML integration.
	 *
	 * @since 2.4.10
	 */
	public function hook() {
		// Check if WPML is installed.
		if ( bbapp_is_wpml_active() ) {
			// Check if we're in an admin edit screen - if so, don't run our hooks.
			$is_admin_edit = is_admin() && isset( $_GET['action'] ) && 'edit' === $_GET['action'] && isset( $_GET['post'] ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.NonceVerification.NoNonceVerification

			if ( ! $is_admin_edit ) {
				add_action( 'rest_api_init', array( $this, 'init' ), 1000 );

				if ( WP_DEBUG ) {
					Logger::instance()->add( 'info_log', 'BuddyBossApp\Integrations\WPML->hook() - WPML detected, registering REST API hooks' );
				}
			}
		}
	}

	/**
	 * Initialize WPML integration with REST API
	 *
	 *  This method is called to register the main hooks for WPML integration.
	 *
	 * @since 2.4.10
	 */
	public function init() {
		/**
		 * Filter to get the available languages.
		 *
		 * @param array $available_languages Array of available languages.
		 * @param array $args                Array of arguments.
		 *
		 * @since 2.4.10
		 *
		 * @return array Array of available languages.
		 */
		$available_languages = apply_filters( 'wpml_active_languages', null, array( 'skip_missing' => false ) );

		// Get language code from header.
		$language_code = bbapp_get_app_language_header();

		// Use the helper function to set WPML language.
		$lang = bbapp_set_wpml_language( $language_code );

		if ( ! empty( $lang ) && array_key_exists( $lang, $available_languages ) ) {
			// Add our custom REST API filters.
			add_filter( 'rest_pre_dispatch', array( $this, 'maybe_translate_request' ), 10, 3 );
			add_filter( 'rest_request_before_callbacks', array( $this, 'translate_post_id' ), 10, 3 );

			// Add language information to all API responses.
			add_filter( 'rest_request_after_callbacks', array( $this, 'add_language_info_to_response' ), 10, 3 );

			// Log language detection (only in debug mode).
			if ( WP_DEBUG ) {
				Logger::instance()->add( 'info_log', 'WPML language detected from appLang header: ' . $lang . ' (original: ' . $language_code . ')' );
			}
		}

		// Dynamically collect all post types that should have REST field registration.
		$post_types = array();

		// Get WordPress post types that are public and show in REST API.
		$public_post_types = get_post_types(
			array(
				'public'       => true,
				'show_in_rest' => true,
			),
		);

		$post_types = array_merge( $post_types, $public_post_types );

		// Add BuddyBoss specific post type that might not have show_in_rest.
		$buddyboss_types = array(
			'app_page',
		);

		// Merge all types and deduplicate.
		$post_types = array_merge( $post_types, $buddyboss_types );
		$post_types = array_unique( $post_types );

		// Apply filters to allow other plugins to register their post types.
		/**
		 * Filter to get the translatable post types.
		 *
		 * @param array $post_types Array of post types.
		 *
		 * @since 2.4.10
		 * @return array Array of post types.
		 */
		$post_types = apply_filters( 'bbapp_wpml_translatable_post_types', $post_types );

		// Register fields for all post types.
		foreach ( $post_types as $post_type ) {
			$this->register_rest_fields( $post_type );
		}
	}

	/**
	 * Helper method to create a new request with translated ID
	 *
	 * @param string $route         Original route.
	 * @param int    $original_id   Original post ID.
	 * @param int    $translated_id Translated post ID.
	 * @param string $method        HTTP method.
	 * @param array  $params        Original request parameters.
	 *
	 * @return WP_REST_Request New request with translated ID
	 */
	private function create_translated_request( $route, $original_id, $translated_id, $method, $params ) {
		// Create a new request with the translated ID.
		$new_path    = str_replace( "/{$original_id}", "/{$translated_id}", $route );
		$new_request = new WP_REST_Request( $method, $new_path );

		// Copy all parameters except language parameters.
		foreach ( $params as $key => $value ) {
			if ( 'wpml_lang' !== $key && 'lang' !== $key && 'appLang' !== $key ) {
				$new_request->set_param( $key, $value );
			}
		}

		return $new_request;
	}

	/**
	 * Helper to check if ID needs translation and redirect if necessary
	 *
	 * @param mixed           $result    Current result.
	 * @param \WP_REST_Server $server    Server instance.
	 * @param WP_REST_Request $request   Original request.
	 * @param int             $entity_id Original ID to check.
	 * @param string          $post_type Post type.
	 * @param string          $lang      Target language.
	 *
	 * @return mixed Original result or redirected response
	 */
	private function translate_and_redirect_if_needed( $result, $server, $request, $entity_id, $post_type, $lang ) {
		// Skip translation for user profiles.
		if ( 'user' === $post_type ) {
			return $result;
		}

		/**
		 * Filter to get the translated post ID.
		 *
		 * @param int    $original_id   Original post ID.
		 * @param string $post_type     Post type.
		 * @param bool   $is_translated Whether the post is translated.
		 * @param string $lang          Language code.
		 *
		 * @since 2.4.10
		 *
		 * @return int Translated post ID.
		 */
		$translated_id = apply_filters( 'wpml_object_id', $entity_id, $post_type, false, $lang );

		if ( $translated_id && $translated_id !== $entity_id ) {
			// Create a new request with the translated ID.
			$new_request = $this->create_translated_request(
				$request->get_route(),
				$entity_id,
				$translated_id,
				$request->get_method(),
				$request->get_params()
			);

			// Get result for the translated entity.
			return $server->dispatch( $new_request );
		}

		return $result;
	}

	/**
	 * Get post type from ID or type string
	 *
	 * @param int|string $id_or_type Post ID or type string.
	 * @param string     $context    Context for lookup ('id' or 'type').
	 *
	 * @since 2.4.10
	 * @return string Post type
	 */
	private function get_post_type( $id_or_type, $context = 'id' ) {
		// If we have a post ID, try to get post type directly from database.
		if ( 'id' === $context ) {
			$post = get_post( $id_or_type );
			if ( $post ) {
				return $post->post_type;
			}
		}

		// For BuddyPress and special content types, we need to use the mapping
		// as they might not be directly stored as posts.
		if ( is_string( $id_or_type ) ) {
			// Most WordPress content is just the post type directly.
			return $id_or_type;
		}

		// Return whatever we got as a fallback.
		return is_string( $id_or_type ) ? $id_or_type : '';
	}

	/**
	 * Pre-dispatch filter to handle language switching earlier in the request cycle
	 *
	 * @param mixed           $result  Response to replace the requested version with.
	 * @param \WP_REST_Server $server  Server instance.
	 * @param WP_REST_Request $request Request used to generate the response.
	 *
	 * @since 2.4.10
	 * @return mixed
	 */
	public function maybe_translate_request( $result, $server, $request ) {
		// Get language using the common method for consistency.
		$language_code = bbapp_get_app_language_header();

		// Convert locale to WPML language code.
		$lang = bbapp_get_wpml_lang_code( $language_code );

		if ( empty( $lang ) ) {
			return $result;
		}

		// Set the language for this request.
		global $sitepress;
		if ( $sitepress ) {
			$sitepress->switch_lang( $lang, true );
		}

		/**
		 * Switch language action.
		 *
		 * @param string $lang Language code.
		 *
		 * @since 2.4.10
		 *
		 * @return void
		 */
		do_action( 'wpml_switch_language', $lang );

		$route = $request->get_route();

		// Process based on endpoint type.
		if ( $this->is_buddyboss_app_endpoint( $route ) ) {
			// For BuddyBoss app-specific endpoints, check if they contain object IDs .
			// that might need translation.

			// Handle endpoints like /buddyboss-app/*/v1/[post_type]/[id].
			if ( preg_match( '/\/buddyboss-app\/.*\/([a-z0-9_-]+)\/(\d+)(?:\/|$)/', $route, $matches ) ) {
				$entity_type = $matches[1];
				$entity_id   = (int) $matches[2];

				// Skip translation for user-related endpoints.
				if ( in_array( $entity_type, array( 'members', 'users', 'member', 'user' ), true ) ) {
					return $result;
				}

				// Get post type from the entity ID or type.
				$post_type = $this->get_post_type( $entity_id, 'id' );
				if ( empty( $post_type ) ) {
					$post_type = $this->get_post_type( $entity_type, 'type' );
				}

				return $this->translate_and_redirect_if_needed(
					$result,
					$server,
					$request,
					$entity_id,
					$post_type,
					$lang
				);
			}
		} elseif ( $this->is_wp_core_endpoint( $route ) ) {
			// Handle standard WordPress post type endpoints.
			if ( preg_match( '/\/wp\/v2\/([a-z0-9_-]+)\/(\d+)(?:\/|$)/', $route, $matches ) ) {
				$route_type = $matches[1];
				$post_id    = (int) $matches[2];

				// Get post type from the post ID.
				$post_type = $this->get_post_type( $post_id, 'id' );
				if ( empty( $post_type ) ) {
					$post_type = $this->get_post_type( $route_type, 'type' );
				}

				return $this->translate_and_redirect_if_needed(
					$result,
					$server,
					$request,
					$post_id,
					$post_type,
					$lang
				);
			}
		}

		return $result;
	}

	/**
	 * Check if the current route is a BuddyBoss app endpoint
	 *
	 * @param string $route The current route.
	 *
	 * @since 2.4.10
	 * @return boolean
	 */
	private function is_buddyboss_app_endpoint( $route ) {
		// Match only buddyboss-app namespace (removing deprecated appboss).
		$patterns = array(
			'/\/buddyboss-app\//', // BuddyBoss-app namespace.
			'/\/buddyboss\//',     // BuddyBoss namespace.
		);

		foreach ( $patterns as $pattern ) {
			if ( preg_match( $pattern, $route ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Check if the current route is a WordPress core endpoint
	 *
	 * @param string $route The current route.
	 *
	 * @since 2.4.10
	 * @return boolean
	 */
	private function is_wp_core_endpoint( $route ) {
		return 0 === strpos( $route, '/wp/' );
	}

	/**
	 * Register REST API fields for translations
	 *
	 * @param string $post_type Post type to register fields for.
	 *
	 * @since 2.4.10
	 */
	public function register_rest_fields( $post_type ) {
		register_rest_field(
			$post_type,
			'wpml_current_locale',
			array(
				'get_callback'    => array( $this, 'get_current_locale' ),
				'update_callback' => null,
				'schema'          => null,
			)
		);

		register_rest_field(
			$post_type,
			'wpml_translations',
			array(
				'get_callback'    => array( $this, 'get_translations' ),
				'update_callback' => null,
				'schema'          => null,
			)
		);
	}

	/**
	 * REST API callback to get current locale
	 *
	 * @param array           $object     Post object array.
	 * @param string          $field_name REST field name.
	 * @param WP_REST_Request $request    REST request.
	 *
	 * @since 2.4.10
	 * @return string Current locale
	 */
	public function get_current_locale( $object, $field_name, $request ) {
		/**
		 * Filter to get the language details.
		 *
		 * @param array $lang_info Language details.
		 * @param int   $post_id   Post ID.
		 *
		 * @since 2.4.10
		 *
		 * @return array Language details.
		 */
		$lang_info = apply_filters( 'wpml_post_language_details', null, $object['id'] );

		if ( is_wp_error( $lang_info ) || empty( $lang_info ) ) {
			return '';
		}

		return isset( $lang_info['locale'] ) ? $lang_info['locale'] : '';
	}

	/**
	 * REST API callback to get translations
	 *
	 * @param array           $object     Post object array.
	 * @param string          $field_name REST field name.
	 * @param WP_REST_Request $request    REST request.
	 *
	 * @since 2.4.10
	 * @return array Array of translations
	 */
	public function get_translations( $object, $field_name, $request ) {
		// Reset a translation array for this request.
		$this->translations = array();

		/**
		 * Filter to get the available languages.
		 *
		 * @param array $available_languages Array of available languages.
		 *
		 * @since 2.4.10
		 *
		 * @return array Array of available languages.
		 */
		$languages = apply_filters( 'wpml_active_languages', null );

		if ( empty( $languages ) ) {
			return $this->translations;
		}

		foreach ( $languages as $language ) {
			$this->get_translations_for_language( $object, $language );
		}

		return $this->translations;
	}

	/**
	 * Get translations for a specific language
	 *
	 * @param array $object   Post object array.
	 * @param array $language Language data.
	 *
	 * @since 2.4.10
	 */
	private function get_translations_for_language( $object, $language ) {
		/**
		 * Filter to get the translated post ID.
		 *
		 * @param int    $original_id   Original post ID.
		 * @param string $post_type     Post type.
		 * @param bool   $is_translated Whether the post is translated.
		 * @param string $lang          Language code.
		 *
		 * @since 2.4.10
		 *
		 * @return int Translated post ID.
		 */
		$post_id = apply_filters( 'wpml_object_id', $object['id'], get_post_type( $object['id'] ), false, $language['language_code'] );

		if ( null === $post_id || $post_id === $object['id'] ) {
			return;
		}

		$translated_post = get_post( $post_id );

		if ( ! $translated_post ) {
			return;
		}

		$translation = array(
			'locale'     => $language['default_locale'],
			'id'         => $translated_post->ID,
			'slug'       => $translated_post->post_name,
			'post_title' => $translated_post->post_title,
			'href'       => get_permalink( $translated_post ),
		);

		/**
		 * Filter to get the translated post.
		 *
		 * @param array $translation     Translated post data.
		 * @param array $translated_post Translated post object.
		 * @param array $language        Language data.
		 *
		 * @since 2.4.10
		 *
		 * @return array Translated post data.
		 */
		$this->translations[ $language['default_locale'] ] = apply_filters( 'bbapp_wpml_translation', $translation, $translated_post, $language );
	}

	/**
	 * Add language information to all API responses
	 *
	 * @param \WP_REST_Response $response The response object.
	 * @param array             $handler  Route handler used for the request.
	 * @param WP_REST_Request   $request  Request used to generate the response.
	 *
	 * @since 2.4.10
	 *
	 * @return \WP_REST_Response
	 */
	public function add_language_info_to_response( $response, $handler, $request ) {
		// Skip if response is an error or not an object/array.
		if ( is_wp_error( $response ) || ! ( $response instanceof \WP_REST_Response ) ) {
			return $response;
		}

		/**
		 * Filter to get the current language.
		 *
		 * @param string $current_lang Current language code.
		 *
		 * @since 2.4.10
		 *
		 * @return string Current language code.
		 */
		$current_lang = apply_filters( 'wpml_current_language', null );

		/**
		 * Filter to get the language details.
		 *
		 * @param array  $language_details Language details.
		 * @param string $current_lang     Current language code.
		 *
		 * @since 2.4.10
		 *
		 * @return array Language details.
		 */
		$language_details = apply_filters( 'wpml_language_details', null, $current_lang );

		// Add language info to response headers.
		$response->header( 'X-WPML-Language', $current_lang );

		// Get language data for the response.
		$language_data = array(
			'code'   => $current_lang,
			'locale' => isset( $language_details['default_locale'] ) ? $language_details['default_locale'] : '',
			'name'   => isset( $language_details['display_name'] ) ? $language_details['display_name'] : '',
			'is_rtl' => isset( $language_details['is_rtl'] ) ? (bool) $language_details['is_rtl'] : false,
		);

		$route = $request->get_route();
		$data  = $response->get_data();

		// Always add language info in headers.
		$response->header( 'X-BuddyBoss-Language', wp_json_encode( $language_data ) );

		// Don't modify sequential arrays.
		if ( is_array( $data ) && ! empty( $data ) ) {
			$keys          = array_keys( $data );
			$is_sequential = ( range( 0, count( $data ) - 1 ) === $keys );

			if ( ! $is_sequential ) {
				// For standard associative arrays, add language info.
				if ( $this->is_buddyboss_app_endpoint( $route ) || $this->is_wp_core_endpoint( $route ) ) {
					// Only add if not already present.
					if ( ! isset( $data['language'] ) ) {
						$data['language'] = $language_data;
						$response->set_data( $data );
					}
				}

				// Add translations for content with ID and type.
				if ( isset( $data['id'] ) && isset( $data['type'] ) && 'user' !== $data['type'] ) {
					/**
					 * Filter to get the translated post data.
					 *
					 * @param array $translations Translated post data.
					 * @param int   $post_id      Post ID.
					 *
					 * @since 2.4.10
					 *
					 * @return array Translated post data.
					 */
					$translations = apply_filters( 'wpml_post_translations', array(), $data['id'] );

					if ( ! empty( $translations ) ) {
						$formatted_translations = array();

						foreach ( $translations as $lang_code => $translated_id ) {
							if ( $lang_code === $current_lang ) {
								continue; // Skip current language.
							}

							/**
							 * Filter to get the language details.
							 *
							 * @param array  $lang_details Language details.
							 * @param string $lang_code    Language code.
							 *
							 * @since 2.4.10
							 *
							 * @return array Language details.
							 */
							$lang_details = apply_filters( 'wpml_language_details', null, $lang_code );
							if ( ! empty( $lang_details ) ) {
								$formatted_translations[ $lang_code ] = array(
									'id'     => $translated_id,
									'locale' => $lang_details['default_locale'],
									'name'   => $lang_details['display_name'],
								);
							}
						}

						if ( ! empty( $formatted_translations ) ) {
							$data['translations'] = $formatted_translations;
							$response->set_data( $data );
						}
					}
				}
			}
		}

		return $response;
	}

	/**
	 * Translate post ID based on the requested language
	 *
	 * @param \WP_REST_Response $response The response object.
	 * @param array             $handler  Route handler used for the request.
	 * @param WP_REST_Request   $request  Request used to generate the response.
	 *
	 * @since 2.4.10
	 * @return \WP_REST_Response
	 */
	public function translate_post_id( $response, $handler, $request ) {
		$route = $request->get_route();

		// Don't process if response is an error.
		if ( is_wp_error( $response ) ) {
			return $response;
		}

		/**
		 * Filter to get the current language.
		 *
		 * @param string $current_lang Current language code.
		 *
		 * @since 2.4.10
		 *
		 * @return string Current language code.
		 */
		$current_lang = apply_filters( 'wpml_current_language', null );
		if ( empty( $current_lang ) ) {
			return $response;
		}

		// BuddyBoss app endpoints or any data-containing endpoint.
		if ( $response instanceof \WP_REST_Response ) {
			$data = $response->get_data();

			// For any kind of array data, use our recursive detection method.
			if ( is_array( $data ) ) {
				$translated_data = $this->detect_translatable_content( $data, $current_lang );

				// Only update if changes were made.
				if ( $translated_data !== $data ) {
					$response->set_data( $translated_data );
				}

				return $response;
			}
		}

		// WordPress core endpoints with post IDs in the route - redirect to translated version.
		if ( $this->is_wp_core_endpoint( $route ) && preg_match( '/\/wp\/v2\/([a-z0-9_-]+)\/(\d+)(?:\/|$)/', $route, $matches ) ) {
			$route_type = $matches[1];
			$post_id    = (int) $matches[2];

			// Get the post to determine its actual type.
			$post = get_post( $post_id );

			// Skip if post doesn't exist.
			if ( ! $post ) {
				return $response;
			}

			// Get post type directly from the post object.
			$post_type = $post->post_type;

			// Skip translation for user profiles.
			if ( 'user' === $post_type ) {
				return $response;
			}

			/**
			 * Filter to get the translated post ID.
			 *
			 * @param int    $original_id   Original post ID.
			 * @param string $post_type     Post type.
			 * @param bool   $is_translated Whether the post is translated.
			 * @param string $lang          Language code.
			 *
			 * @since 2.4.10
			 *
			 * @return int Translated post ID.
			 */
			$translated_id = apply_filters( 'wpml_object_id', $post_id, $post_type, false, $current_lang );

			if ( ! $translated_id || $translated_id === $post_id ) {
				return $response;
			}

			// Get the translated post.
			$translated_post = get_post( $translated_id );
			if ( ! $translated_post ) {
				return $response;
			}

			// Create a new request with the translated ID.
			$new_request = $this->create_translated_request(
				$route,
				$post_id,
				$translated_id,
				$request->get_method(),
				$request->get_params()
			);

			// Get a new response with the translated ID.
			$new_response = rest_do_request( $new_request );

			return $new_response;
		}

		return $response;
	}

	/**
	 * Detect and translate content in response data
	 *
	 * @param array  $data         The response data.
	 * @param string $current_lang The current language code.
	 *
	 * @return array The translated data
	 */
	private function detect_translatable_content( $data, $current_lang ) {
		if ( ! is_array( $data ) ) {
			return $data;
		}

		// Process each element in the array.
		foreach ( $data as $key => $value ) {
			// Check if this is a translatable post object.
			if ( is_array( $value ) && isset( $value['id'] ) && isset( $value['type'] ) ) {
				// Skip user type.
				if ( 'user' === $value['type'] || 'member' === $value['type'] ) {
					continue;
				}

				// Get post type from the item's ID or type.
				$post_type = $this->get_post_type( $value['id'], 'id' );
				if ( empty( $post_type ) ) {
					$post_type = $this->get_post_type( $value['type'], 'type' );
				}

				/**
				 * Filter to get the translated post ID.
				 *
				 * @param int    $original_id   Original post ID.
				 * @param string $post_type     Post type.
				 * @param bool   $is_translated Whether the post is translated.
				 * @param string $lang          Language code.
				 *
				 * @since 2.4.10
				 *
				 * @return int Translated post ID.
				 */
				$translated_id = apply_filters(
					'wpml_object_id',
					$value['id'],
					$post_type,
					false,
					$current_lang
				);

				if ( $translated_id && $translated_id !== $value['id'] ) {
					// Update translated ID.
					$data[ $key ]['id'] = $translated_id;

					// Also translate title and content if present.
					$translated_post = get_post( $translated_id );
					if ( $translated_post ) {
						if ( isset( $value['title'] ) ) {
							$data[ $key ]['title'] = $translated_post->post_title;
						}
						if ( isset( $value['content'] ) ) {
							$data[ $key ]['content'] = $translated_post->post_content;
						}
						if ( isset( $value['excerpt'] ) ) {
							$data[ $key ]['excerpt'] = $translated_post->post_excerpt;
						}
					}
				}
			} elseif ( is_array( $value ) ) {  // Recursively process nested arrays.
				$data[ $key ] = $this->detect_translatable_content( $value, $current_lang );
			}
		}

		return $data;
	}
}
