<?php
/**
 * Holds WPML related features/hooks/filters.
 *
 * @package BuddyBossApp\Integrations\WPML
 */

namespace BuddyBossApp\Integrations\WPML;

use BuddyBossApp\Integrations\WPML\WPMLSupport;
use BuddyBossApp\Integrations\WPML\DeepLinkingSupport;

/**
 * WPML related features/hooks/filters.
 *
 * This class is responsible for loading the WPML integration and initializing
 *
 * the necessary components for BuddyBoss App.
 *
 * @since   2.4.10
 */
class Main {

	/**
	 * Class instance.
	 *
	 * @var Main
	 */
	private static $instance;

	/**
	 * Main constructor.
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * @since 2.4.10
	 *
	 * @return Main
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Load the integration
	 *
	 * This method is called to register the module and its hooks.
	 *
	 * @since 2.4.10
	 */
	public function load() {
		// Check if WPML is installed.

		if ( bbapp_is_wpml_active() ) {
			// Check if we're in an admin edit screen - if so, don't initialize.
			$is_admin_edit = is_admin() && isset( $_GET['action'] ) && 'edit' === $_GET['action'] && isset( $_GET['post'] ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.NonceVerification.NoNonceVerification

			if ( ! $is_admin_edit ) {
				// Initialize WPML Support.
				WPMLSupport::instance();

				// Initialize Menu support.
				DeepLinkingSupport::instance();
			}

			add_filter( 'nav_menu_meta_box_object', array( $this, 'wpml_query_filters' ), 11 );
		}
	}

	/**
	 * Suppress WPML filter when 'lang' parameter is not present or empty.
	 *
	 * @param object $args The arguments object.
	 *
	 * @since 2.4.10
	 * @return object The modified arguments object.
	 */
	public function wpml_query_filters( $args ) {
		if ( isset( $args->_default_query ) ) {
			// Check if 'lang' parameter exists in query string and sanitize it.
			$lang_param = isset( $_GET['lang'] ) ? sanitize_text_field( wp_unslash( $_GET['lang'] ) ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended

			// Only suppress WPML filter when 'lang' parameter is not present or empty.
			$args->_default_query['suppress_wpml_where_and_join_filter'] = empty( $lang_param );
		}

		return $args;
	}
}
