<?php
/**
 * WPML DeepLinking Support Integration file
 *
 * @package BuddyBossApp\Integrations\WPML
 */

namespace BuddyBossApp\Integrations\WPML;

/**
 * Class for handling WPML integration with BuddyBoss App DeepLinking
 *
 * This class provides functionality to translate DeepLinking responses
 *
 * @since   2.4.10
 * @package BuddyBossApp\Integrations\WPML
 */
class DeepLinkingSupport {

	/**
	 * Class instance.
	 *
	 * @var DeepLinkingSupport
	 */
	private static $instance;

	/**
	 * DeepLinkingSupport constructor.
	 *
	 * This method is private to enforce the Singleton pattern.
	 *
	 * @since 2.4.10
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * This method returns the singleton instance of the DeepLinkingSupport class.
	 *
	 * @since 2.4.10
	 *
	 * @return DeepLinkingSupport
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Load the class and its hooks
	 *
	 * This method is called to register the module and its hooks.
	 *
	 * @since 2.4.10
	 */
	public function load() {
		// Check if WPML is active, if not, don't add any hooks.
		if ( ! bbapp_is_wpml_active() ) {
			return;
		}

		// Remove existing filters to avoid duplicates.
		remove_filter( 'bbapp_deeplinking_details', array( $this, 'translate_deeplinking_response' ), 20 );
		remove_filter( 'bbapp_deeplinking_details', array( $this, 'translate_deeplinking_response' ), 999 );
		remove_filter( 'bbapp_deeplinking_details', array( $this, 'translate_deeplinking_response' ), 9999 );

		// Add our filter with highest priority to ensure it runs last.
		add_filter( 'bbapp_deeplinking_details', array( $this, 'translate_deeplinking_response' ), 99999, 1 );

		// Also hook into REST API init to ensure our language is set early.
		add_action( 'rest_api_init', array( $this, 'set_language_from_header' ), 5 );
	}

	/**
	 * Set language from header early in the API request
	 *
	 * This method is called to set the language based on the appLang header.
	 *
	 * @since 2.4.10
	 */
	public function set_language_from_header() {
		// Use the helper function to set the WPML language.
		bbapp_set_wpml_language();
	}

	/**
	 * Determine the post type from the response item
	 *
	 * @param array $item Response item.
	 *
	 * @since 2.4.10
	 * @return string|null Post type or null if not determined
	 */
	private function determine_post_type( $item ) {
		// First check if we have an item_id, which is the most reliable way to get post type.
		if ( ! empty( $item['item_id'] ) ) {
			$post = get_post( $item['item_id'] );
			if ( $post ) {
				return $post->post_type;
			}
		}

		// Try to determine from URL patterns.
		if ( ! empty( $item['url'] ) ) {
			$url = $item['url'];

			$url_id = url_to_postid( $url );
			if ( ! empty( $url_id ) ) {
				$post      = get_post( $url_id );
				$post_type = $post->post_type;

				return $post_type;
			}
		}

		// Couldn't determine post type.
		return null;
	}

	/**
	 * Translate DeepLinking response based on current language
	 *
	 * @param array $response Current DeepLinking response.
	 *
	 * @since 2.4.10
	 * @return array Modified response with translated content
	 */
	public function translate_deeplinking_response( $response ) {
		// If WPML is not active or response is already an error/empty, return as is.
		if ( ! bbapp_is_wpml_active() || empty( $response ) || is_wp_error( $response ) ) {
			return $response;
		}

		// Only proceed if the appLang header is explicitly set.
		if ( empty( $_SERVER['HTTP_APPLANG'] ) ) {
			return $response;
		}

		// Get language from header.
		$language_code = bbapp_get_app_language_header();

		// If no language code is provided, return original response.
		if ( empty( $language_code ) ) {
			return $response;
		}

		// Get WPML language code (support both hi and hi_IN formats).
		$lang = bbapp_get_wpml_lang_code( $language_code );

		// Support locale format directly if the normalized format wasn't found.
		if ( empty( $lang ) && false !== strpos( $language_code, '_' ) ) {
			$lang = $language_code;
		}

		/**
		 * Filter to get the default language.
		 *
		 * @param string $default_lang Default language code.
		 *
		 * @since 2.4.10
		 *
		 * @return string Default language code.
		 */
		$default_lang = apply_filters( 'wpml_default_language', null );

		// If the requested language is the default language, just add language information.
		if ( $lang === $default_lang ) {
			// Create a new array with the desired order.
			$new_response = array();
			foreach ( $response as $key => $value ) {
				// Add all keys except _links and _embedded.
				if ( ! in_array( $key, array( '_links', '_embedded' ), true ) ) {
					$new_response[ $key ] = $value;
				}
			}

			// Add _embedded data if it exists.
			if ( isset( $response['_embedded'] ) ) {
				$new_response['_embedded'] = $response['_embedded'];
			}

			// Add _links at the end if present.
			if ( isset( $response['_links'] ) ) {
				$new_response['_links'] = $response['_links'];
			}

			return $new_response;
		}

		// For other languages or URLs, process normally through WPML API
		// Switch WPML language.
		global $sitepress;
		if ( $sitepress ) {
			$sitepress->switch_lang( $lang );
		}

		/**
		 * Switch language action.
		 *
		 * @param string $lang Language code.
		 *
		 * @since 2.4.10
		 *
		 * @return void
		 */
		do_action( 'wpml_switch_language', $lang );

		// Process the response based on content type.
		if ( is_array( $response ) ) {
			// Determine the post type.
			$post_type = $this->determine_post_type( $response );

			// Get original item ID.
			$original_id   = ! empty( $response['item_id'] ) ? $response['item_id'] : 0;
			$translated_id = null;

			if ( $original_id && ! empty( $post_type ) ) {
				/**
				 * Filter to get the translated post ID.
				 *
				 * @param int    $original_id   Original post ID.
				 * @param string $post_type     Post type.
				 * @param bool   $is_translated Whether the post is translated.
				 * @param string $lang          Language code.
				 *
				 * @since 2.4.10
				 *
				 * @return int Translated post ID.
				 */
				$translated_id = apply_filters( 'wpml_object_id', $original_id, $post_type, true, $lang );

				// Only use if different from original.
				if ( $translated_id && $translated_id !== $original_id ) {
					// Preserve _embedded data if it exists.
					$embedded_data = isset( $response['_embedded'] ) ? $response['_embedded'] : null;

					// Create a new array with the desired order.
					$new_response = array();

					// Add action and namespace first.
					if ( isset( $response['action'] ) ) {
						$new_response['action'] = $response['action'];
					}

					if ( isset( $response['namespace'] ) ) {
						$new_response['namespace'] = $response['namespace'];
					}

					// Update URL if available.
					if ( ! empty( $response['url'] ) ) {
						$translated_url = get_permalink( $translated_id );
						if ( $translated_url ) {
							$new_response['url'] = $translated_url;
						} else {
							$new_response['url'] = $response['url'];
						}
					}

					// Add translated ID.
					$new_response['item_id'] = $translated_id;

					// Add other fields except those we've already handled and _links, _embedded.
					foreach ( $response as $key => $value ) {
						if ( ! in_array( $key, array( 'action', 'namespace', 'url', 'item_id', 'wpml_current_language', '_links', '_embedded' ), true ) ) {
							$new_response[ $key ] = $value;
						}
					}

					// Add _embedded data if it exists.
					if ( $embedded_data ) {
						$new_response['_embedded'] = $embedded_data;
					}

					// Handle _links.
					if ( ! empty( $response['_links'] ) ) {
						$new_links = array();
						foreach ( $response['_links'] as $rel => $links ) {
							if ( is_array( $links ) ) {
								$new_rel_links = array();
								foreach ( $links as $index => $link ) {
									if ( ! empty( $link['href'] ) ) {
										// Replace the ID in the href URL pattern.
										$link['href'] = preg_replace(
											'/\/\d+(?:\/|$)/',
											"/{$translated_id}/",
											$link['href']
										);
									}
									$new_rel_links[] = $link;
								}
								$new_links[ $rel ] = $new_rel_links;
							}
						}
						$new_response['_links'] = $new_links;
					}

					return $new_response;
				}
			}
		}

		// If no translation found or couldn't determine post type, add language info to original response
		// Create a new array with the desired order.
		$new_response = array();
		foreach ( $response as $key => $value ) {
			// Add all keys except _links and _embedded.
			if ( ! in_array( $key, array( '_links', '_embedded' ), true ) ) {
				$new_response[ $key ] = $value;
			}
		}

		// Add _embedded data if it exists.
		if ( isset( $response['_embedded'] ) ) {
			$new_response['_embedded'] = $response['_embedded'];
		}

		// Add _links at the end if present.
		if ( isset( $response['_links'] ) ) {
			$new_response['_links'] = $response['_links'];
		}

		return $new_response;
	}
}
