<?php

namespace BuddyBossApp\Integrations\Vimeo;

use BuddyBossApp\AppSettings;
use BuddyBossApp\RestErrors;
use WP_Error;
use WP_REST_Server;

class RestAPI extends \WP_REST_Controller {

	protected $namespace = "buddyboss-app/core/v1";
	private static $instance;

	/**
	 * RestAPI constructor.
	 */
	public function __construct() {
	}

	/**
	 * @return mixed
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
			self::$instance->load(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Load hooks function on instance.
	 */
	public function load() {
		/** Nothing here */
		$this->hooks();
	}

	/**
	 * Adding hooks.
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, "register_routes" ), 99 );
	}

	/**
	 * Register routes.
	 */
	public function register_routes() {

		register_rest_route( $this->namespace, '/vimeo-fetch/video/(?P<video_id>\d+)', array(
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_item' ),
				'permission_callback' => array( $this, 'get_item_permissions_check' ),
				'args'                => array(
					'context' => $this->get_context_param( array( 'default' => 'view' ) ),
				),
			),
			'schema' => array( $this, 'get_public_item_schema' ),
		) );
	}

	/**
	 * Check if a given request has access to read a post.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_Error|boolean
	 */
	public function get_item_permissions_check( $request ) {
		return apply_filters( 'ap_rest_vimeo_video_item_permission', true );
	}

	/**
	 * Get video information from vimeo.
	 * This will only return the video information of video hosted on the account of auth code owner.
	 *
	 * @param \WP_REST_Request $request
	 *
	 * @return \WP_Error|\WP_HTTP_Response|\WP_REST_Response
	 */
	public function get_item( $request ) {

		if ( ! is_user_logged_in() ) {
			return RestErrors::instance()->user_not_logged_in();
		}

		global $bbapp_var;

		$vimeo_auth_code = AppSettings::instance()->get_setting_value( "vimeo.auth_code" );

		if(empty($vimeo_auth_code)){
			return new WP_Error( 'rest_bbapp_vimeo_fetch_not_available', __( 'Fetching data from Vimeo is not currently available.', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$url = esc_url( 'https://api.vimeo.com/videos/' . $request['video_id'] );

		$cache_name = "bbapp_vimeo_vid_cache".sha1($url.$vimeo_auth_code);

		$cache = get_transient( $cache_name );

		if ( $cache !== false ) {

			$response_body = $cache;

			if(is_wp_error($response_body)) {
				return $response_body;
			}

		} else {

			$args          = array(
				'user-agent' => isset( $request['user_agent'] ) ? $request['user_agent'] : $_SERVER['HTTP_USER_AGENT'],
				'method'     => 'GET',
				'headers'    => array( 'Authorization' => 'Bearer ' . $vimeo_auth_code ),
			);

			$response = wp_remote_request(
				$url,
				$args
			);

			// Get the results
			$response_code    = wp_remote_retrieve_response_code( $response );
			$response_message = wp_remote_retrieve_response_message( $response );
			$response_body    = wp_remote_retrieve_body( $response );

			// Decode the JSON in the body, if it is json
			if ( $response_body ) {
				$j = json_decode( $response_body );

				if ( $j ) {
					$response_body = $j;
				}
			}

			if ( 200 !== $response_code ) {
				$error = new WP_Error( str_replace( ' ', '_', strtolower( $response_message ) ), $response_body->error, array( 'status' => $response_code ) );
				set_transient( $cache_name, $error, 60 ); // we are caching for 60 sec if error. to handle high scale.
				return $error;
			}

			set_transient( $cache_name, $response_body, 300 );

		}

		$response = rest_ensure_response( $response_body );

		return $response;
	}

}
