<?php
/**
 * Holds TutorLMS related features/hooks/filters.
 *
 * @package BuddyBossApp\Integrations\TutorLMS
 */

namespace BuddyBossApp\Integrations\TutorLMS;

/**
 * TutorLMS related features/hooks/filters.
 */
class Main {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Main constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of this class.
	 *
	 * @since 2.2.80
	 * @return Main
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Load the class required code.
	 *
	 * @since 2.2.80
	 */
	public function load() {
		add_action( 'admin_notices', array( $this, 'tutor_pro_required_admin_notices' ) );
		add_filter( 'bbapp_block_filter_options', array( $this, 'add_filter_options' ), 10, 2 );
		add_filter( 'bbapp_block_filter_list', array( $this, 'add_filter_list' ), 10, 2 );
		add_filter( 'bbapp_block_query_args', array( $this, 'add_query_args' ), 10, 2 );
	}

	/**
	 * Check if Tutor LMS Pro is not active.
	 *
	 * @since 2.2.80
	 * @return void
	 */
	public function tutor_pro_required_admin_notices() {
		global $pagenow;

		if ( ! function_exists( 'tutor' ) ) {
			return;
		}

		$page_get      = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$post_type_get = ( ! empty( $_GET['post_type'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['post_type'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ( 'edit.php' === $pagenow && 'app_page' === $post_type_get ) || ( 'admin.php' === $pagenow && 'bbapp-' === substr( $page_get, 0, strlen( 'bbapp-' ) ) ) ) {
			if ( ! function_exists( 'tutor_pro' ) ) {
				?>
				<div class="notice notice-error">
					<p><?php esc_html_e( 'Tutor LMS Pro is required for BuddyBoss App native integration.', 'buddyboss-app' ); ?></p>
				</div>
				<?php
			}
		}
	}

	/**
	 * Function to get the course categories and tags.
	 *
	 * @param int $item_id Categories ID.
	 *
	 * @since 2.2.80
	 *
	 * @return array|object
	 */
	public function bbapp_get_course_cats_and_tags( $item_id = 0 ) {
		if ( empty( $item_id ) ) {
			return array();
		}

		$course_cats = wp_get_post_terms( $item_id, 'course-category', array( 'fields' => 'ids' ) );
		$course_tags = wp_get_post_terms( $item_id, 'course-tag', array( 'fields' => 'ids' ) );

		return $this->bbapp_iap_get_courses_iap_products( array( $item_id ), $course_cats, $course_tags );
	}

	/**
	 * Function to get the Tutor LMS course IAP.
	 *
	 * @param array $course_ids     specific course ids.
	 * @param array $course_cat_ids specific course category ids.
	 * @param array $course_tag_ids specific course tag ids.
	 *
	 * @since 2.2.80
	 *
	 * @return array|object
	 */
	public function bbapp_iap_get_courses_iap_products( $course_ids = array(), $course_cat_ids = array(), $course_tag_ids = array() ) {
		global $wpdb;

		$select               = "SELECT DISTINCT {$wpdb->prefix}bbapp_iap_products.id FROM {$wpdb->prefix}bbapp_iap_products INNER JOIN {$wpdb->prefix}bbapp_iap_productmeta iapm1 ON ({$wpdb->prefix}bbapp_iap_products.id = iapm1.iap_id ) INNER JOIN {$wpdb->prefix}bbapp_iap_productmeta iapm2 ON ({$wpdb->prefix}bbapp_iap_products.id = iapm2.iap_id)";
		$where                = array();
		$where['all_courses'] = "(iapm1.meta_key='course_access_type' AND iapm1.meta_value='all-courses')";

		if ( ! empty( $course_ids ) ) {
			$imploded_course_ids       = "'" . implode( "','", array_map( 'intval', $course_ids ) ) . "'";
			$where['specific_courses'] = "(iapm1.meta_key='course_access_type' AND iapm1.meta_value='specific-course' AND iapm2.meta_key='course_access_id' AND iapm2.meta_value IN ({$imploded_course_ids}))";
		}

		if ( ! empty( $course_cat_ids ) ) {
			$imploded_cat_ids       = "'" . implode( "','", array_map( 'intval', $course_cat_ids ) ) . "'";
			$where['specific_cats'] = "(iapm1.meta_key='course_access_type' AND iapm1.meta_value='specific-category-courses' AND iapm2.meta_key='course_access_id' AND iapm2.meta_value IN ({$imploded_cat_ids}))";
		}

		if ( ! empty( $course_tag_ids ) ) {
			$imploded_tag_ids       = "'" . implode( "','", array_map( 'intval', $course_tag_ids ) ) . "'";
			$where['specific_tags'] = "(iapm1.meta_key='course_access_type' AND iapm1.meta_value='specific-tag-courses' AND iapm2.meta_key='course_access_id' AND iapm2.meta_value IN ({$imploded_tag_ids}))";
		}

		$imploded_where = '(' . implode( ' OR ', $where ) . ')';
		$published_iap  = "{$wpdb->prefix}bbapp_iap_products.status='published'";

		$sql     = $select . ' WHERE ' . $published_iap . ' AND ' . $imploded_where;
		$results = $wpdb->get_results( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared

		return ! empty( $results ) ? $results : array();
	}

	/**
	 * Add filter options.
	 *
	 * @param array  $filter_options Filter options.
	 * @param string $post_type      Post type.
	 *
	 * @since 2.3.40
	 * @return array
	 */
	public function add_filter_options( $filter_options, $post_type ) {
		if ( bbapp_is_tutor_lms_enabled() && 'courses' === $post_type ) {
			$filter_options[] = array(
				'name'  => 'my_courses',
				'label' => __( 'My Courses', 'buddyboss-app' ),
			);
		}

		return $filter_options;
	}

	/**
	 * Add filter list.
	 *
	 * @param array  $filter_list Filter list.
	 * @param string $post_type   Post type.
	 *
	 * @since 2.3.40
	 * @return array
	 */
	public function add_filter_list( $filter_list, $post_type ) {
		if ( bbapp_is_tutor_lms_enabled() && 'courses' === $post_type ) {
			$my_courses_options = array(
				array(
					'label' => __( 'Yes', 'buddyboss-app' ),
					'key'   => true
				),
				array(
					'label' => __( 'No', 'buddyboss-app' ),
					'key'   => false
				)
			);

			$filter_list['my_courses'] = $my_courses_options;
		}

		return $filter_list;
	}

	/**
	 * Add query args.
	 *
	 * @param array  $query_args Query args.
	 * @param string $request  Request.
	 *
	 * @since 2.3.40
	 * @return array
	 */
	public function add_query_args( $query_args, $request ) {
		if ( bbapp_is_tutor_lms_enabled() && 'courses' === $query_args['post_type'] ) {
			if ( isset( $query_args['my_courses'] ) && $query_args['my_courses'] ) {
				$user_id      = get_current_user_id();
				// Get courses where the student is enrolled
				$mycourse_ids = tutor_utils()->get_enrolled_courses_ids_by_user( $user_id );

				if ( ! empty( $mycourse_ids ) && ! is_wp_error( $mycourse_ids ) ) {
					$query_args['post__in'] = ! empty( $query_args['post__in'] ) ? array_intersect( $mycourse_ids, $query_args['post__in'] ) : $mycourse_ids;
				}

				/*
				 * If we intersected, but there are no post ids in common,
				 * WP_Query won't return "no posts" for post__in = array()
				 * so we have to fake it a bit.
				*/
				if ( ! $query_args['post__in'] ) {
					$query_args['post__in'] = array( 0 );
				}

				unset( $query_args['my_courses'] );
			}
		}

		return $query_args;
	}
}
