<?php
/**
 * BuddyBossApp TutorLMS Course Integration for BuddyBossApp InAppPurchases.
 *
 * @package BuddyBossApp\Integrations\TutorLMSCourse
 */

namespace BuddyBossApp\Integrations\TutorLMSCourse;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\Admin\InAppPurchases\Helpers;
use BuddyBossApp\Admin\InAppPurchases\ProductHelper;
use BuddyBossApp\InAppPurchases\Controller;
use BuddyBossApp\InAppPurchases\IntegrationAbstract;
use BuddyBossApp\InAppPurchases\Orders;
use BuddyBossApp\Integrations\TutorLMS\Main;
use Tutor\Ecommerce\OrderController;
use TUTOR\Input;
use Tutor\Models\OrderModel;

/**
 * TutorLMS Course Integration for BuddyBossApp InAppPurchases.
 */
final class IAP extends IntegrationAbstract {

	/**
	 * Class instance.
	 *
	 * @var null $instance
	 */
	private static $instance = null;

	/**
	 * TutorLMS Course slug.
	 *
	 * @var string $course_slug
	 */
	private static $course_slug = 'courses';

	/**
	 * TutorLMSCourseIntegration constructor.
	 *
	 * @since 2.2.80
	 */
	private function __construct() {
		// ... leave empty, see Singleton below
	}

	/**
	 * Get the instance of this class.
	 *
	 * @since 2.2.80
	 * @return Controller|null
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Define all actions and filters here.
	 *
	 * @since 2.2.80
	 */
	public function hooks() {
		// TutorLMS integrated courses.
		add_filter( 'bbapp_iap_integrated_products', array( $this, 'bbapp_get_integrated_courses' ), 10, 3 );
	}

	/**
	 * Function to merge the iap product with course access.
	 *
	 * @param array  $integrated_products Integrated IAP products.
	 * @param int    $item_id             course id.
	 * @param string $integration_type    integration type.
	 *
	 * @since 2.2.80
	 *
	 * @return array
	 */
	public function bbapp_get_integrated_courses( $integrated_products, $item_id, $integration_type ) {
		global $wpdb;

		$global_prefix         = \bbapp_iap()->get_global_dbprefix();
		$iap_pluck_ids         = wp_list_pluck( $integrated_products, 'product_id' );
		$course_access_courses = Main::instance()->bbapp_get_course_cats_and_tags( $item_id );

		if ( empty( $course_access_courses ) ) {
			return $integrated_products;
		}

		$pluck_courses          = ! empty( $course_access_courses ) ? wp_list_pluck( $course_access_courses, 'id' ) : array();
		$merge_iaps             = array_unique( array_merge( $iap_pluck_ids, $pluck_courses ) );
		$diff_iaps              = array_diff( $merge_iaps, $iap_pluck_ids );
		$imploded_courses       = "'" . implode( "','", $diff_iaps ) . "'";
		$iaps                   = $wpdb->get_results( "SELECT * FROM {$global_prefix}bbapp_iap_products WHERE status = 'published' AND id IN ({$imploded_courses})" ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$course_access_products = array();

		if ( ! empty( $iaps ) ) {
			foreach ( $iaps as $iap ) {
				$misc_settings    = maybe_unserialize( $iap->misc_settings );
				$integration_data = maybe_unserialize( $iap->integration_data );

				if ( bbapp()->is_network_activated() ) {
					$misc_settings    = $misc_settings[ get_current_blog_id() ];
					$integration_data = $integration_data[ get_current_blog_id() ];
				}

				$integration_slug = $misc_settings['integration_type'];
				$integration_data = isset( $integration_data[ $integration_slug ] ) ? $integration_data[ $integration_slug ] : array();

				foreach ( $integration_data as $value ) {
					$store_data          = maybe_unserialize( $iap->store_data );
					$misc_settings       = Helpers::instance()->bbapp_iap_product_mics_setting( $iap );
					$integration_data    = Helpers::instance()->bbapp_iap_product_integration_data( $iap );
					$integration_type    = ( isset( $misc_settings['integration_type'] ) ? $misc_settings['integration_type'] : '' );
					$integrated_item_ids = Helpers::instance()->bbapp_iap_product_integration_ids( $integration_type, $integration_data );
					$ios                 = Helpers::instance()->bbapp_iap_ios_product_info( $store_data );
					$android             = Helpers::instance()->bbapp_iap_android_product_info( $store_data );
					$bbapp_product_type  = isset( $store_data['bbapp_product_type'] ) ? $store_data['bbapp_product_type'] : 'free';

					// Check Product is configured properly or not. If not it should not return in api response.
					if ( ( isset( $store_data['device_platforms'] ) && in_array(
						'ios',
						$store_data['device_platforms'],
						true
					) && ( empty( $ios['status'] ) || ( 'free' !== $bbapp_product_type && empty( $ios['store_product_id'] ) ) ) ) || ( isset( $store_data['device_platforms'] ) && in_array(
						'android',
						$store_data['device_platforms'],
						true
					) && ( empty( $android['status'] ) || ( 'free' !== $bbapp_product_type && empty( $android['store_product_id'] ) ) ) ) || empty( $integrated_item_ids ) ) {
						continue;
					}

					$group_active_product = 0;
					$has_access_order     = false;

					if ( is_user_logged_in() ) {
						// NOTE : Get user_id and check into bbapp_orders if this bbapp.
						$has_access_order = ProductHelper::has_active_order( $iap, get_current_user_id() );

						if ( ! empty( $iap->iap_group ) ) {
							$group_active_product = ProductHelper::get_group_active_order_product_id( $iap->iap_group, get_current_user_id() );
						}
					}

					$course_access_products[] = array(
						'product_id'           => (int) $iap->id,
						'product_name'         => $iap->name,
						'product_tagline'      => $iap->tagline,
						'product_desc'         => $iap->description,
						'benefits'             => $misc_settings['benefits'],
						'global_subscription'  => (bool) $misc_settings['global_subscription'],
						'bbapp_product_type'   => $bbapp_product_type,
						'ios'                  => $ios,
						'android'              => $android,
						'has_access'           => $has_access_order,
						'group_active_product' => $group_active_product,
						'sort_order'           => (int) $iap->menu_order,
					);
				}
			}
		}

		return array_merge( $course_access_products, $integrated_products );
	}

	/**
	 * Overriding the parent (from base-class) function.
	 *
	 * @param string $integration_type  Integration type.
	 * @param string $integration_label Integration label.
	 *
	 * @since 2.2.80
	 */
	public function set_up( $integration_type, $integration_label ) {
		$this->integration_slug = Controller::$tutor_lms_course_slug;

		parent::set_up( $integration_type, $integration_label );

		$this->item_label = __( 'Course', 'buddyboss-app' );

		// Register Instance.
		bbapp_iap()->integration[ $integration_type ] = $this::instance();
	}

	/**
	 * Below function get triggers when(hook) order is completed.
	 *
	 * @param array        $item_ids Array of item ID's.
	 * @param array|object $order    Order object.
	 *
	 * @since 2.2.80
	 * @return void
	 */
	public function on_order_completed( $item_ids, $order ) {
		$readable_item_ids = array();
		$monetize_by       = tutor_utils()->get_option( 'monetize_by' );
		$generate_invoice  = tutor_utils()->get_option( 'tutor_woocommerce_invoice' );

		foreach ( $item_ids as $item_identifier ) {
			$split               = explode( ':', $item_identifier );
			$course_id           = $split[0];
			$readable_item_ids[] = "<a href=\"post.php?post=$course_id&action=edit\" target='_blank'>$course_id</a>";
			$is_paid_course      = tutor_utils()->is_course_purchasable( $course_id );
			$user_id             = $order->user_id;
			$order_id            = 0;

			/**
			 * Check generate invoice settings along with monetize by
			 */
			if ( $is_paid_course && 'wc' === $monetize_by && $generate_invoice ) {
				// Make manual order for student with this course.
				$product_id = tutor_utils()->get_course_product_id( $course_id );
				$wc_order   = wc_create_order();
				$wc_order->add_product( wc_get_product( $product_id ), 1 );
				$wc_order->set_customer_id( $user_id );
				$wc_order->calculate_totals();
				$wc_order->update_status( 'completed', __( 'Manual Enrollment Order', 'buddyboss-app' ), true );
				$order_id = $wc_order->get_id();
			}

			/**
			 * If user disable generate invoice from tutor settings these will be happen.
			 * 1. Paid course enrollment will automatically completed without generate a WC order.
			 * 2. Earning data will not reflect on report.
			 */
			if ( ! $generate_invoice && $is_paid_course && 'wc' === $monetize_by ) {
				add_filter(
					'tutor_enroll_data',
					function ( $data ) {
						$data['post_status'] = 'completed';
						return $data;
					}
				);
			}

			if ( 'tutor' === $monetize_by ) {
				$items       = array(
					array(
						'item_id'        => $course_id,
						'regular_price'  => '0',
						'sale_price'     => '0',
						'discount_price' => '0',
						'coupon_code'    => null,
					),
				);
				$order_type  = 'single_order';
				$coupon_code = '';
				$args        = array(
					'payment_method'  => 'bbapp-iap',
					'coupon_amount'   => '0',
					'discount_amount' => '0',
				);

				try {
					$tutor_order_id = ( new OrderController( false ) )->create_order(
						$user_id,
						$items,
						OrderModel::PAYMENT_PAID,
						$order_type,
						$coupon_code,
						$args
					);
				} catch ( \Exception $e ) {
					Orders::instance()->add_history( $order->id, 'error', __( 'Tutor LMS native order creation failed.', 'buddyboss-app' ) );
				}

				if ( ! empty( $tutor_order_id ) ) {
					Orders::instance()->add_history( $order->id, 'info', sprintf( __( 'Tutor LMS native order has been created. ID: %s', 'buddyboss-app' ), $tutor_order_id ) );
					Orders::instance()->update_meta( $order->id, '_tutor_lms_order_id', $tutor_order_id );
					( new OrderModel )->mark_as_paid( $tutor_order_id, esc_html__( 'Order created by BuddyBoss App', 'buddyboss-app' ) );
				}
			} else {
				// Enroll to course/bundle.
				$enrolled_id = tutor_utils()->do_enroll( $course_id, $order_id, $user_id );

				// Mark enrollment as completed.
				tutor_utils()->course_enrol_status_change( $enrolled_id, 'completed' );

				// update user course count.
				$this->user_update_count( $user_id, $user_id );
			}
		}

		$readable_item_ids = implode( ', ', $readable_item_ids );

		/* translators: %s: Item ids. */
		Orders::instance()->add_history( $order->id, 'info', sprintf( __( 'User enrolled in course(s), ID(s) are : %s', 'buddyboss-app' ), $readable_item_ids ) );

		Orders::instance()->update_meta( $order->id, '_tutor_lms_course_ids', maybe_serialize( $item_ids ) );
	}

	/**
	 * Below function get triggers when(hook) order is activated.
	 *
	 * @param array        $item_ids Array of item ID's.
	 * @param array|object $order    Order object.
	 *
	 * @since 2.2.80
	 * @return void
	 */
	public function on_order_activate( $item_ids, $order ) {
		// NOTE: Similar to onOrderCompleted($order) until something needs to be changed?
		$this->on_order_completed( $item_ids, $order );
	}

	/**
	 * Below function get triggers when(hook) order is expired.
	 *
	 * @param array $item_ids Array of item ID's.
	 * @param array $order    Order object.
	 *
	 * @since 2.2.80
	 * @return mixed|void
	 */
	public function on_order_expired( $item_ids, $order ) {
		// NOTE: Similar to onOrderCancelled($order) until something needs to be changed?
		$this->on_order_cancelled( $item_ids, $order );
	}

	/**
	 * Below function get triggers when(hook) order is cancelled.
	 *
	 * @param array        $item_ids Array of item ID's.
	 * @param array|object $order    Order object.
	 *
	 * @since 2.2.80
	 * @return void
	 */
	public function on_order_cancelled( $item_ids, $order ) {
		$readable_item_ids = array();

		foreach ( $item_ids as $item_identifier ) {
			$split               = explode( ':', $item_identifier );
			$id                  = $split[0];
			$readable_item_ids[] = "<a href=\"post.php?post=$id&action=edit\" target='_blank'>$id</a>";

			// Revoke the course access.
			tutor_utils()->cancel_course_enrol( $id, $order->user_id );

			// update user course count.
			$this->user_update_count( $id, $order->user_id, 'minus' );
		}

		$tutor_order_id = Orders::instance()->get_meta( $order->id, '_tutor_lms_order_id' );

		if ( ! empty( $tutor_order_id ) ) {
			$params = array(
				'id'           => $tutor_order_id,
				'order_status' => OrderModel::ORDER_CANCELLED,
			);

			/**
			 * Action from Tutor LMS.
			 */
			do_action( 'tutor_before_order_cancel', $params );

			$response = ( new OrderModel )->update_order( $params['id'], $params );

			if ( ! $response ) {
				Orders::instance()->add_history( $order->id, 'error', __( 'Tutor LMS native order cancellation failed.', 'buddyboss-app' ) );
			}

			/**
			 * Action from Tutor LMS.
			 */
			do_action( 'tutor_order_payment_status_changed', $params['id'], '', OrderModel::ORDER_CANCELLED );
		}

		$readable_item_ids = implode( ', ', $readable_item_ids );

		/* translators: %s: Item id. */
		Orders::instance()->add_history( $order->id, 'info', sprintf( __( 'User un-enrolled in course(s), ID(s) are : %s ', 'buddyboss-app' ), $readable_item_ids ) );
	}

	/**
	 * Helper function to update users courses counts.
	 *
	 * @param int    $course_id Course id.
	 * @param int    $user_id   User id.
	 * @param string $action    Action.
	 *
	 * @since 2.2.80
	 */
	public function user_update_count( $course_id, $user_id, $action = 'plus' ) {
		$courses = get_user_meta( $user_id, '_tutor_lms_inapp_purchase_enrolled_courses_access_counter', true );

		if ( ! empty( $courses ) ) {
			$courses = maybe_unserialize( $courses );
		} else {
			$courses = array();
		}

		if ( isset( $courses[ $course_id ] ) ) {
			if ( 'plus' === $action ) {
				$courses[ $course_id ] += 1;
			} else {
				$courses[ $course_id ] -= 1;
			}
		} else {
			$courses[ $course_id ] = ( 'plus' === $action ) ? 1 : 0;
		}

		update_user_meta( $user_id, '_tutor_lms_inapp_purchase_enrolled_courses_access_counter', $courses );
	}

	/**
	 * Handle bbapp ajax iap_linking_options for this integration.
	 *
	 * @param array $results levels data.
	 *
	 * @since 2.2.80
	 * @return array|mixed
	 */
	public function iap_linking_options( $results ) {
		global $wpdb;

		$slug  = self::$course_slug;
		$query = "SELECT CONCAT(posts.ID, \":\" , posts.post_title, \"\") as 'id', CONCAT(posts.post_title, \" (#\" , posts.ID, \")\")  as 'text' FROM $wpdb->posts posts WHERE posts.post_type = \"$slug\"  AND posts.post_status = 'publish' ";

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		return $wpdb->get_results( $query, ARRAY_A );
	}

	/**
	 * Handle bbapp ajax iap_integration_ids for this integration.
	 * it's a return items label with id.
	 *
	 * @param array $results         levels data.
	 * @param array $integration_ids selected integration id.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function iap_integration_ids( $results, $integration_ids ) {
		return $results;
	}

	/**
	 * Get an item edit link to show on the order page.
	 *
	 * @param string $link    URL.
	 * @param int    $item_id Item id.
	 *
	 * @since 2.2.80
	 * @return string
	 */
	public function item_id_permalink( $link, $item_id ) {
		return "post.php?post=$item_id&action=edit";
	}

	/**
	 * Check given post has given membership configured or not.
	 *
	 * @param bool $is_available        Flag for product is available or not.
	 * @param int  $item_id             Post id or membership id. It's directly or indirectly [ through integration's item id ] connected with bbapp product.
	 * @param int  $integration_item_id Integration item id which is connected with bbapp product.
	 *
	 * @since 2.2.80
	 * @return bool
	 */
	public function is_purchase_available( $is_available, $item_id, $integration_item_id ) {
		return $item_id === $integration_item_id;
	}

	/**
	 * Check given integration item has access.
	 *
	 * @param array $item_ids Array of item ID's.
	 * @param int   $user_id  User ID.
	 *
	 * @since 2.2.80
	 * @return bool
	 */
	public function has_access( $item_ids, $user_id ) {
		if ( is_admin() ) {
			return true;
		}

		$has_access = false;

		foreach ( $item_ids as $item_identifier ) {
			$split     = explode( ':', $item_identifier );
			$course_id = $split[0];

			if ( tutor_utils()->get_enrolled_data( $user_id, $course_id ) ) {
				$has_access = true;
				break;
			}
		}

		return $has_access;
	}
}
