<?php
/**
 * Holds deepliking functionality related TutorLMS.
 *
 * @package BuddyBossApp\Integrations\TutorLMS
 */

namespace BuddyBossApp\Integrations\TutorLMS;

use BuddyBossApp\DeepLinking\Type\TypeAbstract;
use WP_Post;

/**
 * Deeplink class for TutorLMS.
 */
class DeepLinking extends TypeAbstract {

	/**
	 * Post types array.
	 *
	 * @var array
	 */
	public $tutor_arr_post_types = array();

	/**
	 * Array post type mapping.
	 *
	 * @var array
	 */
	public $tutor_arr_post_types_mapping = array();

	/**
	 * Taxonomy array.
	 *
	 * @var array
	 */
	public $tutor_arr_taxonomy = array();

	/**
	 * Construct function.
	 *
	 * @since 2.2.80
	 */
	public function __construct() {
		parent::__construct();
		$this->tutor_lms_init();
		add_filter( 'bbapp_deeplinking_cpt_action', array( $this, 'update_cpt_action' ), 10, 2 );
		add_filter( 'bbapp_deeplinking_cpt_namespace', array( $this, 'update_cpt_namespace' ), 10, 2 );
		add_filter( 'bbapp_deeplinking_term_namespace', array( $this, 'update_term_namespace' ), 10, 2 );
		add_filter( 'bbapp_deeplinking_taxonomy_namespace', array( $this, 'update_taxonomy_namespace' ), 10, 2 );
		add_filter( 'bbapp_deeplinking_cpt', array( $this, 'update_cpt_object' ), 10, 2 );
	}

	/**
	 * Init all Tutor LMS maps.
	 *
	 * @since 2.2.80
	 */
	public function tutor_lms_init() {
		$this->tutor_arr_post_types_mapping = array(
			'courses'    => 'course',
			'topics'     => 'topic',
			'lesson'     => 'lesson',
			'tutor_quiz' => 'quiz',
		);
		$this->tutor_arr_post_types         = array_keys( $this->tutor_arr_post_types_mapping );
		$this->tutor_arr_taxonomy           = array(
			'course-category',
			'course-tag',
		);
	}

	/**
	 * Add item embed with response
	 *
	 * @param array   $response Response.
	 * @param WP_Post $post     Post-type object.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function update_cpt_object( $response, $post ) {
		if ( 'courses' === $post->post_type ) {
			$response['_link_action'] = $post->post_type;
		}

		if ( 'topics' === $post->post_type ) {
			$response['_link_action'] = $post->post_type;
		}

		if ( 'lesson' === $post->post_type ) {
			$response['_link_action'] = $post->post_type;
		}

		if ( 'tutor_quiz' === $post->post_type ) {
			$response['_link_action'] = $post->post_type;
		}

		return $response;
	}

	/**
	 * Update Tutor Course CPT action.
	 *
	 * @param string  $action Action.
	 * @param WP_Post $post   Post object.
	 *
	 * @since 2.2.80
	 *
	 * @return string
	 */
	public function update_cpt_action( $action, $post ) {
		if ( $post instanceof WP_Post && in_array( $post->post_type, $this->tutor_arr_post_types, true ) ) {
			return 'open_' . $this->tutor_arr_post_types_mapping[ $post->post_type ];
		}

		return $action;
	}

	/**
	 * Update Tutor LMS CPT namespace
	 *
	 * @param object $namespace_name Namespace object.
	 * @param string $post_type      Post type.
	 *
	 * @since 2.2.80
	 * @return string
	 */
	public function update_cpt_namespace( $namespace_name, $post_type ) {
		if ( isset( $post_type ) && in_array( $post_type, $this->tutor_arr_post_types, true ) ) {
			return 'bbapp_lms';
		}

		return $namespace_name;
	}

	/**
	 * Update Tutor LMS terms namespace
	 *
	 * @param object $namespace_name Namespace object.
	 * @param object $term           Term object.
	 *
	 * @since 2.2.80
	 *
	 * @return string
	 */
	public function update_term_namespace( $namespace_name, $term ) {
		if ( $term instanceof \WP_Term && in_array( $term->taxonomy, $this->tutor_arr_taxonomy, true ) ) {
			return 'bbapp_lms';
		}

		return $namespace_name;
	}

	/**
	 * Update Tutor LMS taxonomy namespace.
	 *
	 * @param object $namespace_name Namespace object.
	 * @param object $taxonomy       Taxonomy object.
	 *
	 * @since 2.2.80
	 * @return string
	 */
	public function update_taxonomy_namespace( $namespace_name, $taxonomy ) {
		if ( $taxonomy instanceof \WP_Taxonomy && in_array( $taxonomy->name, $this->tutor_arr_taxonomy, true ) ) {
			return 'bbapp_lms';
		}

		return $namespace_name;
	}

	/**
	 * Function to parse URL.
	 *
	 * @param string $url URL to parse.
	 *
	 * @since 2.2.80
	 * @return mixed|void
	 */
	public function parse( $url ) {
		$url_meta                = $this->get_url_data( $url );
		$tutor_dashboard_page_id = (int) tutor_utils()->get_option( 'tutor_dashboard_page_id' );

		if ( url_to_postid( $url ) === $tutor_dashboard_page_id ) {
			if ( ! is_user_logged_in() ) {
				return array(
					'action'    => 'open_login',
					'namespace' => 'bbapp_lms',
					'url'       => $url,
				);
			}

			$parse_data = array(
				'namespace' => 'bbapp_lms',
				'url'       => $url,
			);

			if ( isset( $url_meta['pagename'] ) ) {
				$parse_data['action'] = 'open_dashboard';
			}

			if ( isset( $url_meta['tutor_dashboard_page'] ) ) {
				switch ( $url_meta['tutor_dashboard_page'] ) {
					case 'enrolled-courses':
						$parse_data['sub_action'] = 'enrolled_courses';
						break;
					case 'my-courses':
						$parse_data['sub_action'] = 'my_courses';
						break;
					case 'my-profile':
						$parse_data['sub_action'] = 'my_profile';
						break;
					case 'my-quiz-attempts':
						$parse_data['sub_action'] = 'my_quiz_attempts';
						break;
					case 'question-answer':
						$parse_data['sub_action'] = 'question_answer';
						break;
					case 'quiz-attempts':
						$parse_data['sub_action'] = 'quiz_attempts';
						break;
					case 'google-meet':
						$parse_data['sub_action'] = 'google_meet';
						break;
					default:
						$parse_data['sub_action'] = $url_meta['tutor_dashboard_page'];
						break;
				}
			}

			if ( isset( $url_meta['tutor_dashboard_sub_page'] ) ) {
				switch ( $url_meta['tutor_dashboard_sub_page'] ) {
					case 'withdraw-settings':
						$parse_data['sub_tab'] = 'withdraw_settings';
						break;
					case 'active-courses':
						$parse_data['sub_tab'] = 'active_courses';
						break;
					case 'completed-courses':
						$parse_data['sub_tab'] = 'completed_courses';
						break;
					case 'pending-courses':
						$parse_data['sub_tab'] = 'pending_courses';
						break;
					case 'draft-courses':
						$parse_data['sub_tab'] = 'draft_courses';
						break;
					default:
						$parse_data['sub_tab'] = $url_meta['tutor_dashboard_sub_page'];
						break;
				}
			}

			return $parse_data;
		}
	}
}
