<?php
/**
 * Holds TutorLMS functionality related Access Controls.
 *
 * @package BuddyBossApp\Integrations\TutorLMS
 */

namespace BuddyBossApp\Integrations\TutorLMS;

use BuddyBossApp\AccessControls\Core\Core;
use BuddyBossApp\AccessControls\TmpItems;
use BuddyBossApp\Admin\SetupAdmin;
use BuddyBossApp\Integrations\TutorLMS\AccessControls\TutorLMSCourse;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * TutorLMS Access Controls class.
 */
class AccessControls extends \BuddyBossApp\AccessControls\Integration_Abstract {

	/**
	 * Conditiona name to register.
	 *
	 * @var string $condition_name condition name.
	 */
	private $condition_name = 'tutor-lms-course';

	/**
	 * Function to set up the conditions.
	 *
	 * @since 2.2.80
	 *
	 * @return void
	 */
	public function setup() {
		add_action( 'admin_init', array( $this, 'admin_init' ), 15 );

		$this->register_condition(
			array(
				'condition'              => $this->condition_name,
				'items_callback'         => array( $this, 'tutor_lms_course_items_callback' ),
				'item_callback'          => array( $this, 'tutor_lms_course_item_callback' ),
				'users_callback'         => array( $this, 'tutor_lms_course_users_callback' ),
				'labels'                 => array(
					'condition_name'          => __( 'TutorLMS Courses', 'buddyboss-app' ),
					'item_singular'           => __( 'TutorLMS Course', 'buddyboss-app' ),
					'member_of_specific_item' => __( 'Enrolled in a specific course', 'buddyboss-app' ),
					'member_of_any_items'     => __( 'Enrolled in any course', 'buddyboss-app' ),
				),
				'support_any_items'      => true,
				'has_any_items_callback' => array( $this, 'has_any_items_callback' ),
			)
		);

		// Register Rules Item Type.
		$this->register_rules_item_type( 'courses' );
		$this->register_rules_item_type( 'courses_term' );
		$this->register_rules_item_type( 'default_tutor_lms_course' );

		// Register Settings Screens.
		$this->register_screens();
		$this->load_hooks();
		// run job hook. (not empty meta run the job.).
	}

	/**
	 * Load Access Controls.
	 *
	 * @since 2.2.80
	 *
	 * @return void
	 */
	public function admin_init() {
		$page_get = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'admin.php' === SetupAdmin::instance()->get_page_now() && ! empty( $page_get ) && 'bbapp-access-controls' === $page_get ) {
			if ( 'tutor-lms-courses' === Core::instance()->get_current_tab() && bbapp_is_admin_page() ) {
				TutorLMSCourse::instance()->will_render( true );
			}
		}
	}

	/**
	 * Load screens settings classes.
	 *
	 * @since 2.2.80
	 */
	public function register_screens() {
		self::instance()->register_settings_screen(
			array(
				'id'              => TutorLMSCourse::instance()->screen_name,
				'menu_title'      => __( 'Courses', 'buddyboss-app' ),
				'page_title'      => __( 'Courses', 'buddyboss-app' ),
				'render_callback' => array( TutorLMSCourse::instance(), 'render_callback' ),
				'options'         => array(
					'tutorial_link' => admin_url( '#' ),
					'sub_tab'       => array(
						TutorLMSCourse::instance()->screen_name => __( 'Courses', 'buddyboss-app' ),
						'cat_tag'          => __( 'Categories & Tags', 'buddyboss-app' ),
						'default_settings' => __( 'Default Settings', 'buddyboss-app' ),
					),
				),
			)
		);
	}


	/**
	 * Items callback method.
	 *
	 * @param string $search Search the condition.
	 * @param int    $page   Page number.
	 * @param int    $limit  Limit the items to be fetched.
	 *
	 * @since 2.2.80
	 *
	 * @return array
	 */
	public function tutor_lms_course_items_callback( $search = '', $page = 1, $limit = 20 ) {
		$results = new \WP_Query(
			array(
				'post_status'    => 'publish',
				'post_type'      => 'courses',
				'paged'          => $page,
				'posts_per_page' => $limit,
				's'              => $search,
			)
		);

		$list = array();

		if ( ! empty( $results->get_posts() ) ) {
			foreach ( $results->get_posts() as $item ) {
				$list[ $item->ID ] = array(
					'id'   => $item->ID,
					'name' => $item->post_title,
					'link' => admin_url( "post.php?post={$item->ID}&action=edit" ),
				);
			}
		}

		return $list;
	}

	/**
	 * Item callback method.
	 *
	 * @param int $item_value Item value of condition.
	 *
	 * @since 2.2.80
	 *
	 * @return array|false
	 */
	public function tutor_lms_course_item_callback( $item_value ) {
		$course = get_post( $item_value );

		if ( empty( $course ) || ! $course instanceof \WP_Post || 'trash' === $course->post_status ) {
			return false;
		}

		return array(
			'id'   => $course->ID,
			'name' => $course->post_title,
			'link' => admin_url( "post.php?post={$course->ID}&action=edit" ),
		);
	}


	/**
	 * Users callback method.
	 *
	 * @param array $data     condition data.
	 * @param int   $page     current page number.
	 * @param int   $per_page limit.
	 *
	 * @since 2.2.80
	 * @return array|string
	 */
	public function tutor_lms_course_users_callback( $data, $page = 1, $per_page = 10 ) {
		$sub_condition = ( ! empty( $data['sub_condition'] ) ) ? $data['sub_condition'] : '';
		$item_value    = ( ! empty( $data['item_value'] ) ) ? $data['item_value'] : '';
		$group_id      = ( ! empty( $data['group_id'] ) ) ? $data['group_id'] : 0;
		$rounds_count  = ( ! empty( $data['rounds_count'] ) ) ? $data['rounds_count'] : 0;

		if ( 'specific' === $sub_condition ) {
			if ( empty( $item_value ) ) {
				return self::instance()->return_users( array() );
			}

			/**
			 * Preparing the data into tmp items collections.
			 */
			$query_args = array(
				'number' => $per_page,
				'paged'  => $page,
			);

			$tutor_lms_members_ids = $this->tutor_lms_get_users_for_course( $item_value, $query_args );

			if ( is_array( $tutor_lms_members_ids ) ) {
				return self::instance()->return_users( $tutor_lms_members_ids );
			} else {
				return self::instance()->return_error( __( 'Error while getting users results from TutorLMS function.', 'buddyboss-app' ) );
			}
		} elseif ( 'any' === $sub_condition ) {
			$transient_key = "access_control_tutor_lms_course_condition_jb_data_{$group_id}";

			// Check if it's a first round.
			if ( 1 === $rounds_count ) {
				$condition_data                   = array(
					'prepare_page'       => 1,
					'is_data_prepared'   => false,
					'pending_course_ids' => $this->tutor_lms_get_all_course_ids(),
				);
				$condition_data['current_course'] = array_key_first( $condition_data['pending_course_ids'] );
			} else {
				$condition_data = get_transient( $transient_key );
			}

			/**
			 * Preparing the data into tmp items collections.
			 */
			if ( ! $condition_data['is_data_prepared'] ) {
				$query_args            = array(
					'per_page' => $per_page,
					'paged'    => ( ! empty( $condition_data['prepare_page'] ) ) ? $condition_data['prepare_page'] : 1,
				);
				$tutor_lms_members_ids = $this->tutor_lms_get_users_for_course( $condition_data['current_course'], $query_args );

				if ( is_array( $tutor_lms_members_ids ) ) {
					if ( ! empty( $tutor_lms_members_ids ) ) {
						$items = array();

						foreach ( $tutor_lms_members_ids as $user_id ) {
							$items[] = array(
								'item_type'         => "tutor_lms_courses_{$group_id}",
								'item_id'           => $user_id,
								'secondary_item_id' => $condition_data['current_course'],
							);
						}

						$insert = TmpItems::instance()->insert_items( $items );

						++$condition_data['prepare_page'];

						if ( empty( $insert ) ) {
							return self::instance()->return_error( __( 'Error while preparing members data for TutorLMS.', 'buddyboss-app' ) );
						}
					} else { // when all users are processed.
						unset( $condition_data['pending_course_ids'][ $condition_data['current_course'] ] ); // remove from pending course id.

						if ( ! empty( $condition_data['pending_course_ids'] ) ) {
							$condition_data['current_course'] = array_key_first( $condition_data['pending_course_ids'] ); // get next course to process.
							$condition_data['prepare_page']   = 1; // reset pagination to 1.
						} else {
							// If pending courses found empty finish the job.
							$condition_data['is_data_prepared'] = true;
						}
					}

					set_transient( $transient_key, $condition_data, DAY_IN_SECONDS );

					return self::instance()->return_wait( __( 'Preparing TutorLMS Members Data', 'buddyboss-app' ) );
				} else {
					return self::instance()->return_error( __( 'Error while getting users results from TutorLMS function.', 'buddyboss-app' ) );
				}
			}

			/**
			 * Start Sending Users Data Once Tmp Collection Preparations has done.
			 * Once the Preparations for any of specific courses are done above the execution will reach here.
			 */
			$items = TmpItems::instance()->get_items(
				array(
					'include_item_types' => array( "tutor_lms_courses_{$group_id}" ),
					'per_page'           => $per_page,
					'page'               => $page,
				)
			);

			$user_ids = array();

			foreach ( $items['results'] as $item ) {
				$user_ids[] = $item->item_id;
			}

			return self::instance()->return_users( $user_ids );
		}

		return self::instance()->return_error( __( 'Invalid sub condition.', 'buddyboss-app' ) );
	}

	/**
	 * Get users for course.
	 *
	 * @param int   $course_id  Course id.
	 * @param array $query_args Query args.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function tutor_lms_get_users_for_course( $course_id = 0, $query_args = array() ) {
		$paged    = $query_args['paged'] ?? 1;
		$per_page = $query_args['per_page'] ?? 20;
		$offset   = ( $per_page * $paged ) - $per_page;
		$search   = '';
		$date     = '';

		$students_list = tutor_utils()->get_students( $offset, $per_page, $search, $course_id, $date );

		return wp_list_pluck( $students_list, 'ID' );
	}

	/**
	 * Return all courses IDs
	 *
	 * @since 2.2.80
	 *
	 * @return array
	 */
	public function tutor_lms_get_all_course_ids() {
		$query_args = array(
			'post_type'   => 'courses',
			'post_status' => 'publish',
			'fields'      => 'ids',
			'orderby'     => 'title',
			'order'       => 'ASC',
			'nopaging'    => true,    // Turns OFF paging logic to get ALL courses.
		);

		$query = new \WP_Query( $query_args );

		if ( ! empty( $query->posts ) ) {
			$ids    = $query->posts;
			$return = array();

			foreach ( $ids as $id ) {
				$return[ $id ] = $id;
			}

			return $return;
		}

		return array();
	}

	/**
	 * Function to check if user has any condition.
	 *
	 * @param int $user_id User id to check.
	 *
	 * @since 2.2.80
	 *
	 * @return bool
	 */
	public function has_any_items_callback( $user_id ) {
		$course_ids = tutor_utils()->get_enrolled_courses_by_user( $user_id );

		return ! empty( $course_ids );
	}

	/**
	 * Function to load all hooks of this condition.
	 *
	 * @since 2.2.80
	 */
	public function load_hooks() {
		// *******************************************************************//
		// ********************* TutorLMS Course hooks **********************//
		// *******************************************************************//

		add_action( 'tutor_after_enrolled', array( $this, 'tutor_lms_after_enroll' ), 10, 2 );
		add_action( 'tutor_after_enrollment_deleted', array( $this, 'tutor_lms_after_enroll_deleted' ), 10, 2 );
		add_action( 'tutor_after_enrollment_cancelled', array( $this, 'tutor_lms_after_enroll_cancelled' ), 10, 2 );
		add_action( 'tutor_enrollment/after/completed', array( $this, 'tutor_lms_after_enrolled' ), 10, 1 );
		add_action( 'tutor_enrollment/after/cancel', array( $this, 'tutor_lms_after_enroll_revoke' ), 10, 1 );
		add_action( 'tutor_enrollment/after/delete', array( $this, 'tutor_lms_after_enroll_revoke' ), 10, 1 );
		add_action( 'tutor_enrollment/after/expired', array( $this, 'tutor_lms_after_enroll_revoke' ), 10, 1 );
	}

	/**
	 * Function to run after user enrolled in a course.
	 *
	 * @param int $course_id Course id.
	 * @param int $user_id   User id.
	 *
	 * @since 2.2.80
	 * @return false|void
	 */
	public function tutor_lms_after_enroll( $course_id, $user_id ) {
		// Get the course.
		$course = get_post( $course_id );

		if ( empty( $course ) || ! $course instanceof \WP_Post || 'trash' === $course->post_status ) {
			return false;
		}

		if ( empty( $user_id ) || empty( $course_id ) ) {
			return false;
		}

		self::instance()->condition_add_user( $user_id, $this->condition_name, $course_id );
	}

	/**
	 * Function to run after user enrollment deleted.
	 *
	 * @param int $course_id Course id.
	 * @param int $user_id   User id.
	 *
	 * @since 2.2.80
	 * @return void
	 */
	public function tutor_lms_after_enroll_deleted( $course_id, $user_id ) {
		if ( empty( $user_id ) || empty( $course_id ) ) {
			return;
		}

		self::instance()->condition_remove_user( $user_id, $this->condition_name, $course_id );
	}

	/**
	 * Function to run after user enrollment cancelled.
	 *
	 * @param int $course_id Course id.
	 * @param int $user_id   User id.
	 *
	 * @since 2.2.80
	 * @return void
	 */
	public function tutor_lms_after_enroll_cancelled( $course_id, $user_id ) {
		if ( empty( $user_id ) || empty( $course_id ) ) {
			return;
		}

		self::instance()->condition_remove_user( $user_id, $this->condition_name, $course_id );
	}

	/**
	 * Function to run after user enrollment completed.
	 *
	 * @param int $enrol_id Enrolment id.
	 *
	 * @since 2.2.80
	 * @return void
	 */
	public function tutor_lms_after_enrolled( $enrol_id ) {
		if ( empty( $enrol_id ) ) {
			return;
		}

		$enrolment = tutor_utils()->get_enrolment_by_enrol_id( $enrol_id );

		if ( ! empty( $enrolment ) ) {
			self::instance()->condition_add_user( $enrolment->student_id, $this->condition_name, $enrolment->course_id );
		}
	}
	/**
	 * Function to run after user enrollment revoked.
	 *
	 * @param int $enrol_id Enrolment id.
	 *
	 * @since 2.2.80
	 * @return void
	 */
	public function tutor_lms_after_enroll_revoke( $enrol_id ) {
		if ( empty( $enrol_id ) ) {
			return;
		}

		$enrolment = tutor_utils()->get_enrolment_by_enrol_id( $enrol_id );

		if ( ! empty( $enrolment ) ) {
			self::instance()->condition_remove_user( $enrolment->student_id, $this->condition_name, $enrolment->course_id );
		}
	}
}
