<?php

namespace BuddyBossApp\Integrations\RestrictContentPro;

use BuddyBossApp\InAppPurchases\Controller;
use BuddyBossApp\InAppPurchases\IntegrationAbstract;
use BuddyBossApp\InAppPurchases\Orders;

final class IAP extends IntegrationAbstract {

	private static $instance;

	/**
	 * MemberPress constructor.
	 */
	public function __construct() {
		//Using Singleton, see instance()
	}

	/**
	 * @return Main
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$className      = __CLASS__;
			self::$instance = new $className;
		}

		return self::$instance;
	}

	/**
	 * Overriding the parent(from base-class) function
	 *
	 * @param $integration_type
	 * @param $integration_label
	 */
	public function set_up( $integration_type, $integration_label ) {

		$this->integration_slug = Controller::$restrict_content_pro_slug;

		parent::set_up( $integration_type, $integration_label );
		$this->item_label = __( 'Membership', 'buddyboss-app' );

		// Register Instance
		bbapp_iap()->integration[ $integration_type ] = $this::instance();
	}

	/**
	 * Trigger following code when BuddyBoss App order completed
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return false|mixed
	 */
	public function on_order_completed( $item_ids, $order ) {

		if ( ! class_exists( 'Restrict_Content_Pro' ) ) {
			Orders::instance()->add_history( $order["id"], 'iap-error', __( "Restrict Content Pro plugin is not activated.", "buddyboss-app" ) );

			return false;
		}

		$customer = rcp_get_customer_by_user_id( $order->user_id );

		// Create a new customer.
		if ( empty( $customer ) ) {
			$customer_id = rcp_add_customer( array(
				'user_id' => $order->user_id
			) );
			if ( ! empty( $customer_id ) ) {
				$customer = rcp_get_customer( $customer_id );
				Orders::instance()->add_history( $order->id, 'warning', __( "Restrict Content Pro customer created.", "buddyboss-app" ) );
			}
		}
		if ( is_object( $customer ) ) {

			$trans_ids = array();
			foreach ( $item_ids as $item_identifier ) {
				$split    = explode( ':', $item_identifier );
				$level_id = $split[0];

				$data = array(
					'object_id'               => $level_id,
					'auto_renew'              => 0,
					'maximum_renewals'        => ! empty( $membership_level->maximum_renewals ) ? absint( $membership_level->maximum_renewals ) : 0,
					'status'                  => 'active',
					'gateway_customer_id'     => $order->user_id,
					'gateway_subscription_id' => $order->id,
					'gateway'                 => 'manual',
					'signup_method'           => 'manual',
					'notes'                   => 'BuddyBoss App Purchase ID is:' . $order->id
				);

				/**
				 * Add new membership.
				 */
				$membership_id = $customer->add_membership( $data );
				if ( ! empty( $membership_id ) ) {
					$trans_ids[] = $membership_id;
					Orders::instance()->add_history( $order->id, 'info', __( "Created new transaction on Restrict Content Pro. Membership ID is #$membership_id", 'buddyboss-app' ) );
				}
			}
			Orders::instance()->update_meta( $order->id, "_restrict_content_pro_order_ids", serialize( $trans_ids ) );
		} else {
			Orders::instance()->add_history( $order->id, 'warning', __( "Admin Error : Restrict Content Pro customer not found.", 'buddyboss-app' ) );
		}
	}

	/**
	 * Trigger following code when BuddyBoss App order Activated
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return false|mixed
	 */
	public function on_order_activate( $item_ids, $order ) {
		return $this->on_order_completed( $item_ids, $order );
	}

	/**
	 * Trigger following code when BuddyBoss App order cancelled
	 *
	 * @param $item_ids
	 * @param $order
	 * @param string $status
	 *
	 * @return false
	 */
	public function on_order_cancelled( $item_ids, $order, $status = 'cancelled' ) {

		if ( ! class_exists( 'Restrict_Content_Pro' ) ) {
			Orders::instance()->add_history( $order["id"],'iap-error' ,__( "WishList Member plugin is not activated.", "buddyboss-app" ) );

			return false;
		}

		$trans_ids = unserialize( Orders::instance()->get_meta( $order->id, "_restrict_content_pro_order_ids" ) );

		$customer = rcp_get_customer_by_user_id( $order->user_id );
		if ( is_object( $customer ) ) {
			$memberships = $customer->get_memberships();
			foreach ( $item_ids as $item_identifier ) {
				$split    = explode( ':', $item_identifier );
				$level_id = $split[0];
				foreach ( $memberships as $membership ) {
					$membership_id = $membership->get_id();
					if ( $membership->get_object_id() == $level_id && in_array( $membership_id, $trans_ids ) ) {
						//( 'cancelled' === $status ) ? $membership->cancel() : $membership->expire();
						$membership->expire();
						Orders::instance()->add_history( $order->id, 'info', __( "Access revoked for Restrict Content Pro. Membership ID is #$membership_id", "buddyboss-app" ) );
					}
				}
			}
		} else {
			Orders::instance()->add_history( $order->id, 'warning', __( "Admin Error : Restrict Content Pro customer not found.", "buddyboss-app" ) );
		}
	}

	/**
	 * Trigger following code when BuddyBoss App order expired
	 *
	 * @param $item_ids
	 * @param $order
	 */
	public function on_order_expired( $item_ids, $order ) {
		$this->on_order_cancelled( $item_ids, $order, 'expired' );
	}

	/**
	 * Handle bbapp ajax iap_linking_options for this integration
	 *
	 * @param $results levels data
	 * @param $search : Search term
	 *
	 * @return mixed
	 */
	public function iap_linking_options( $results ) {
		$levels = rcp_get_subscription_levels();
		foreach ( $levels as $level ) {
			$results[] = [
				'id'   => $level->id . ':' . $level->name,
				'text' => $level->name . ':' . $level->id,
			];
		}

		return $results;
	}

	/**
	 * Handle bbapp ajax iap_integration_ids for this integration
	 * it's return items label with id.
	 *
	 * @param $results levels data
	 * @param $integration_ids selected integration id
	 *
	 * @return mixed
	 */
	public function iap_integration_ids( $results, $integration_ids ) {
		foreach ( $integration_ids as $key => $integration_id ) {
			$results[ $key ]['id']   = $integration_id;
			$results[ $key ]['text'] = "Not found: $integration_id";

			$levels = rcp_get_subscription_levels();
			foreach ( $levels as $level ) {
				/**
				 * Split[ level id, level name ];
				 */
				$split = explode( ':', $integration_id );
				if ( (int) $level->id === (int) $split[0] ) {
					$results[ $key ]['text'] = $level->name;
					break;
				}
			}
		}

		return $results;
	}

	/**
	 * Get item edit link to show on order page.
	 *
	 * @param $link
	 * @param $item_id
	 *
	 * @return string
	 */
	public function item_id_permalink( $link, $item_id ) {
		return 'admin.php?page=rcp-member-levels&edit_subscription=' . $item_id;
	}

	/**
	 * Check given post has given membership configured or not
	 * Ref: restrict-content-pro/includes/memberships/class-rcp-membership.php : can_access
	 *
	 * @param $has_access
	 * @param $forItemId
	 * @param $integration_item_id
	 *
	 * @return bool
	 */
	public function is_purchase_available( $has_access, $forItemId, $integration_item_id ) {
		$post_levels = self::get_level_ids_for_post( $forItemId, [ $integration_item_id ] );

		return in_array( $integration_item_id, $post_levels );
	}

	/**
	 * Check given integration item has access.
	 *
	 * @param $item_ids
	 * @param $user_id
	 *
	 * @return false
	 */
	function has_access( $item_ids, $user_id ) {
		if ( is_admin() ) {
			return true;
		}
		$has_access = false;

		$customer = rcp_get_customer_by_user_id( $user_id );

		if ( is_object( $customer ) ) {
			
			$memberships = $customer->get_memberships();

			foreach ( $item_ids as $item_identifier ) {
				$split    = explode( ':', $item_identifier );
				$level_id = $split[0];

				foreach ( $memberships as $membership ) {
					if ( $membership->get_object_id() == $level_id && $membership->is_active() ) {
						$has_access = true;
						break;
					}
				}
			}
		}
		return $has_access;
	}

	/**
	 * get levels available for post
	 *
	 * @param int $post_id
	 * @param array $level_ids
	 *
	 * @return array
	 */
	static public function get_level_ids_for_post( $post_id, $level_ids = [] ) {
		$post_level_ids = [];

		if ( ! empty( $level_ids ) ) {
			$levels = rcp_get_subscription_levels();
			if ( ! empty( $levels ) ) {
				$level_ids = wp_list_pluck( $levels, 'id' );
			}
		}

		if ( ! empty( $level_ids ) ) {
			foreach ( $level_ids as $integration_item_id ) {
				if ( 'sfwd-courses' === get_post_type( $post_id ) ) {
					$level      = new \RCP_Levels();
					$ld_courses = maybe_unserialize( $level->get_meta( $integration_item_id, '_learndash_restrict_content_pro_courses', true ) );
					if ( ! empty( $ld_courses ) && in_array( $post_id, $ld_courses ) ) {
						$post_level_ids[] = $integration_item_id;
					}
				} elseif ( self::check_level_id_for_post_in_rcp_metabox( $post_id, $integration_item_id ) ) {
					$post_level_ids[] = $integration_item_id;
				}
			}
		}

		return $post_level_ids;
	}

	/**
	 * Check $integration_item_id is config in $post_id RCP MetaBox setting or not.
	 *
	 * @param int $post_id
	 * @param int $integration_item_id
	 *
	 * @return array
	 */
	static public function check_level_id_for_post_in_rcp_metabox( $post_id, $integration_item_id ) {

		$has_access             = false;
		$post_type_restrictions = rcp_get_post_type_restrictions( get_post_type( $post_id ) );
		if ( empty( $post_type_restrictions ) ) {
			$membership_levels = rcp_get_content_subscription_levels( $post_id );
			$access_level      = get_post_meta( $post_id, 'rcp_access_level', true );
		} else {
			$membership_levels = array_key_exists( 'subscription_level', $post_type_restrictions ) ? $post_type_restrictions['subscription_level'] : false;
			$access_level      = array_key_exists( 'access_level', $post_type_restrictions ) ? $post_type_restrictions['access_level'] : false;
		}

		if ( ! empty( $membership_levels ) || "" !== $access_level ) {
			$membership = rcp_get_membership( $integration_item_id );

			// Check membership level restrictions.
			if ( ! empty( $membership_levels ) ) {

				if ( is_string( $membership_levels ) ) {

					switch ( $membership_levels ) {

						case 'any' :
							$has_access = ! empty( $integration_item_id );
							break;

						case 'any-paid' :
							$has_access = $membership->is_paid();
							break;
					}

				} else {
					$has_access = in_array( $integration_item_id, $membership_levels );
				}
			}

			// Check post access level restrictions.
			if ( "" !== $access_level && $membership->has_access_level( $access_level ) ) {
				$has_access = true;
			}
		}

		return $has_access;
	}
}
