<?php
/**
 * Holds Restrict Content Pro functionality for the Access Controls.
 *
 * @package BuddyBossApp\Integrations\RestrictContentPro
 */

namespace BuddyBossApp\Integrations\RestrictContentPro;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Restrict Content Pro class for Access Controls.
 */
class AccessControls extends \BuddyBossApp\AccessControls\Integration_Abstract {

	/**
	 * Conditiona name.
	 *
	 * @var string $condition_name condition name.
	 */
	private $condition_name = 'restrict-content-pro';

	/**
	 * Function to set up the conditions.
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed|void
	 */
	public function setup() {
		$this->register_condition(
			array(
				'condition'              => $this->condition_name,
				'items_callback'         => array( $this, 'restrict_content_pro_items_callback' ),
				'item_callback'          => array( $this, 'restrict_content_pro_item_callback' ),
				'users_callback'         => array( $this, 'restrict_content_pro_users_callback' ),
				'labels'                 => array(
					'condition_name'          => __( 'Restrict Content Pro', 'buddyboss-app' ),
					'item_singular'           => __( 'Membership', 'buddyboss-app' ),
					'member_of_specific_item' => __( 'Has specific membership', 'buddyboss-app' ),
					'member_of_any_items'     => __( 'Has any membership', 'buddyboss-app' ),
				),
				'support_any_items'      => true,
				'has_any_items_callback' => array( $this, 'has_any_items_callback' ),
			)
		);

		$this->load_hooks();
	}

	/**
	 * Hooks here
	 *
	 * @since 1.5.2.1
	 */
	public function load_hooks() {
		add_action( 'rcp_new_membership_added', array( $this, 'membership_added_add_user' ), 10, 2 );
		add_action( 'rcp_update_payment_status_complete', array( $this, 'add_user_to_membership' ) );
		add_action( 'rcp_update_payment_status_failed', array( $this, 'remove_user_from_membership' ) );
		add_action( 'rcp_update_payment_status_refunded', array( $this, 'remove_user_from_membership' ) );
		add_action( 'rcp_update_payment_status_abandoned', array( $this, 'remove_user_from_membership' ) );
		add_action( 'rcp_membership_post_disable', array( $this, 'bbapp_delete_customer' ), 10, 2 );
		add_action( 'rcp_membership_post_cancel', array( $this, 'bbapp_delete_customer' ), 10, 2 );
		add_action( 'rcp_membership_post_renew', array( $this, 'renew_add_user_membership' ), 10, 3 );
		add_action( 'rcp_member_pre_renew', array( $this, 'renew_remove_user_membership' ), 10, 3 );
		add_action( 'rcp_remove_level', array( $this, 'bbapp_rcp_remove_level' ), 10, 1 );
		add_action(
			'rcp_transition_membership_status_expired',
			array(
				$this,
				'bbapp_rcp_transition_membership_status_expired',
			),
			10,
			2
		);
	}

	/**
	 * Items callback method.
	 *
	 * @param string $search Search the condition.
	 * @param int    $page   Page number.
	 * @param int    $limit  Limit the items to be fetched.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function restrict_content_pro_items_callback( $search = '', $page = 1, $limit = 20 ) {
		$list   = array();
		$levels = rcp_get_membership_levels();

		if ( ! empty( $levels ) ) {
			foreach ( $levels as $level ) {
				$list[ $level->id ] = array(
					'id'   => $level->id,
					'name' => $level->name,
					'link' => admin_url( "admin.php?page=rcp-member-levels&edit_subscription={$level->id}" ),
				);
			}
		}

		return $this->paginate_items_list( $list, $page, $limit, $search );
	}

	/**
	 * Item callback method.
	 *
	 * @param int $item_value Item value of condition.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array|false
	 */
	public function restrict_content_pro_item_callback( $item_value ) {
		$membership = rcp_get_membership_level( $item_value );

		if ( empty( $membership ) || false === $membership ) {
			return false;
		}

		return array(
			'id'   => $membership->get_id(),
			'name' => $membership->get_name(),
			'link' => admin_url( "admin.php?page=rcp-member-levels&edit_subscription={$membership->get_id()}" ),
		);
	}

	/**
	 * Users callback method.
	 *
	 * @param array $data     condition data.
	 * @param int   $page     current page number.
	 * @param int   $per_page limit.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function restrict_content_pro_users_callback( $data, $page = 1, $per_page = 10 ) {
		$user_ids      = array();
		$sub_condition = ( ! empty( $data['sub_condition'] ) ) ? $data['sub_condition'] : '';
		$item_value    = ( ! empty( $data['item_value'] ) ) ? $data['item_value'] : '';
		$offset        = ( $page - 1 ) * $per_page;
		$limit         = ( - 1 === $per_page ) ? 0 : $per_page; // Doesn't support -1 in case of all members.

		if ( 'specific' === $sub_condition ) {
			if ( empty( $item_value ) ) {
				return $user_ids;
			}

			$user_ids = rcp_get_memberships(
				array(
					'status'    => 'active',
					'object_id' => $item_value,
					'fields'    => 'user_id',
					'number'    => $limit,
					'offset'    => $offset,
				)
			);
		} elseif ( 'any' === $sub_condition ) {
			$user_ids = rcp_get_memberships(
				array(
					'status' => 'active',
					'fields' => 'user_id',
					'number' => $limit,
					'offset' => $offset,
				)
			);
		}

		return $this->return_users( $user_ids );
	}

	/**
	 * Function to check if user has any condition.
	 *
	 * @param int $user_id User id to check.
	 *
	 * @since 1.5.2.1
	 *
	 * @return bool
	 */
	public function has_any_items_callback( $user_id ) {
		$rcp_get_customer = rcp_get_customer_by_user_id( $user_id );
		$levels           = rcp_get_customer_memberships( $rcp_get_customer->get_id() );

		return ( ! empty( $levels ) ) ? true : false;
	}

	/**
	 * Triggers after a new membership is added.
	 *
	 * @param int   $membership_id ID of the membership that was just added.
	 * @param array $data          Membership data.
	 *
	 * @since 1.5.2.1
	 */
	public function membership_added_add_user( $membership_id, $data ) {
		if ( empty( $data['user_id'] ) || empty( $data['object_id'] ) || empty( $data['object_id'] ) || 'active' !== $data['status'] ) {
			return;
		}
		$this->condition_add_user( $data['user_id'], $this->condition_name, $data['object_id'] );
	}

	/**
	 * Function to add user when new member added to RCP membership.
	 *
	 * @param int $payment_id RCP membership payment id.
	 *
	 * @since 1.5.2.1
	 */
	public function add_user_to_membership( $payment_id ) {

		$payments = new \RCP_Payments();
		$payment  = $payments->get_payment( $payment_id );

		if ( empty( $payment ) ) {
			return;
		}

		$this->condition_add_user( $payment->user_id, $this->condition_name, $payment->object_id );
	}

	/**
	 * Function to remove user from RCP membership.
	 *
	 * @param int $payment_id RCP membership payment id.
	 */
	public function remove_user_from_membership( $payment_id ) {
		$payments = new \RCP_Payments();
		$payment  = $payments->get_payment( $payment_id );

		if ( empty( $payment ) ) {
			return;
		}

		$this->condition_remove_user( $payment->user_id, $this->condition_name, $payment->object_id );
	}

	/**
	 * Function to delete user when customer deleted from RCP.
	 *
	 * @param int    $membership_id Membership id.
	 * @param object $membership    RCP membership object.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_delete_customer( $membership_id, $membership ) {
		if ( empty( $membership->get_user_id() ) || empty( $membership->get_object_id() ) ) {
			return;
		}
		$this->condition_remove_user( $membership->get_user_id(), $this->condition_name, $membership->get_object_id() );
	}

	/**
	 * Triggers after the membership renewal.
	 *
	 * @param string          $expiration    New expiration date to be set.
	 * @param int             $membership_id ID of the membership.
	 * @param \RCP_Membership $membership    ID of the membership.
	 *
	 * @since 1.5.2.1
	 */
	public function renew_add_user_membership( $expiration, $membership_id, $membership ) {
		if ( empty( $membership->get_user_id() ) || empty( $membership->get_object_id() ) || empty( $membership->get_status() ) || 'active' !== $membership->get_status() ) {
			return;
		}
		$this->condition_add_user( $membership->get_user_id(), $this->condition_name, $membership->get_object_id() );
	}

	/**
	 * Triggers before the membership renewal.
	 *
	 * @param string          $expiration    New expiration date to be set.
	 * @param int             $membership_id ID of the membership.
	 * @param \RCP_Membership $membership    Membership object.
	 *
	 * @since 1.5.2.1
	 */
	public function renew_remove_user_membership( $expiration, $membership_id, $membership ) {
		if ( empty( $membership->get_user_id() ) || empty( $membership->get_object_id() ) ) {
			return;
		}
		$this->condition_remove_user( $membership->get_user_id(), $this->condition_name, $membership->get_object_id() );
	}

	/**
	 * Function will remove member/restrict content pro membership from access groups when remove restrict content pro
	 * membership from admin.
	 *
	 * @param int $level_id Membership level Id.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_rcp_remove_level( $level_id ) {
		$this->condition_item_deleted( $this->condition_name, $level_id );
		$this->condition_remove_all_users( $this->condition_name, $level_id );
		bb_access_recalculate_member_for_has_any_membership_group( $this->condition_name );
	}

	/**
	 * Remove user from access group.
	 *
	 * @param string $old_status    Previous membership status.
	 * @param int    $membership_id ID of the membership.
	 *
	 * @since 1.7.7
	 * @return void
	 */
	public function bbapp_rcp_transition_membership_status_expired( $old_status, $membership_id ) {
		$membership = rcp_get_membership( $membership_id );

		if ( empty( $membership->get_user_id() ) || $membership->get_object_id() ) {
			return;
		}

		$this->condition_remove_user( $membership->get_user_id(), $this->condition_name, $membership->get_object_id() );
	}
}
