<?php

namespace BuddyBossApp\Integrations\PmProMembership;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\InAppPurchases\Controller;
use BuddyBossApp\InAppPurchases\IntegrationAbstract;
use BuddyBossApp\InAppPurchases\Orders;
use BuddyBossApp\Integrations\PmPro\PmProMembershipSupport;

//PmPro Membership Integration for InAppPurchases.
final class IAP extends IntegrationAbstract {

	private static $instance = null;

	private function __construct() {
		// ... leave empty, see Singleton below
	}

	/**
	 * Get the instance of this class.
	 * @return IAP|null
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$className      = __CLASS__;
			self::$instance = new $className;
		}

		return self::$instance;
	}

	/**
	 * Overriding the parent(from base-class) function
	 *
	 * @param $integration_type
	 * @param $integration_label
	 */
	public function set_up( $integration_type, $integration_label ) {

		$this->integration_slug = Controller::$pmpro_membership_slug;

		parent::set_up( $integration_type, $integration_label );

		$this->item_label = __( 'Membership', 'buddyboss-app' );

		// Register Instance
		bbapp_iap()->integration[ $integration_type ] = $this::instance();
	}

	/**
	 * Below function get triggers when(hook) order is completed.
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return mixed
	 */
	public function on_order_completed( $item_ids, $order ) {

		if ( ! function_exists( "pmpro_changeMembershipLevel" ) ) {
			Orders::instance()->add_history( $order["id"],'iap-error', __( "Paid Memberships Pro is not activated.", "buddyboss-app" ) );

			return false;
		}

		$membership_ids = array();
		foreach ( $item_ids as $item_identifier ) {
			$split            = explode( ':', $item_identifier );
			$id               = $split[0];
			$membership_added = $this->update_membership_status( $order->user_id, $id, 'completed', $order );

			if ( class_exists( 'MemberOrder' ) ) {
				$pm_pro_order                         = new \MemberOrder();
				$pm_pro_order->membership_id          = $id;
				$pm_pro_order->user_id                = $order->user_id;
				$pm_pro_order->gateway                = "BuddyBossApp-InAppPurchase";
				$pm_pro_order->status                 = 'success'; //completed is not option in pmPro
				$pm_pro_order->payment_type           = 'InAppPurchase';
				$pm_pro_order->payment_transaction_id = $order->id;
				$pm_pro_order->notes                  = "Generated by the BuddyBoss App InAppPurchase. Get more details about InAppPurchase order at # " . $order->id;
				$pm_pro_order->saveOrder();

				if ( $membership_added ) {
					// NOTE : We are using pmPro Order code as Transaction Number
					$trans_num = $pm_pro_order->code;
					Orders::instance()->add_history( $order->id, 'info', __( "Created new membership for PmPro Memberships. Membership's internal order ID is #" . $trans_num, 'buddyboss-app' ) );
					$membership_ids[] = $id;
				}
			} else {
				Orders::instance()->add_history( $order->id, 'warning', __( "Admin Error: Paid Memberships Pro integration is disabled", 'buddyboss-app' ) );
			}


		}
		Orders::instance()->update_meta( $order->id, "_pmpro_user_membership_ids", serialize( $membership_ids ) );
	}

	/**
	 * Below function get triggers when(hook) order is activated.
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return mixed
	 */
	public function on_order_activate( $item_ids, $order ) {
		return $this->on_order_completed( $item_ids, $order );
	}

	/**
	 * Below function get triggers when(hook) order is cancelled.
	 *
	 * @param $item_ids
	 * @param        $order
	 * @param string $status
	 *
	 * @return mixed
	 */
	public function on_order_cancelled( $item_ids, $order, $status = 'cancelled' ) {
		foreach ( $item_ids as $item_identifier ) {

			$split            = explode( ':', $item_identifier );
			$id               = $split[0];
			$membership_id    = $this->update_membership_status( $order->user_id, $id, $status, $order );

			Orders::instance()->add_history( $order->id, 'info', __( "Access revoked for PmPro Memberships. Membership Level ID is #$membership_id", 'buddyboss-app' ) );
		}
	}

	/**
	 * Below function get triggers when(hook) order is expired.
	 *
	 * @param $item_ids
	 * @param        $order
	 * @param string $status
	 *
	 * @return mixed
	 */
	public function on_order_expired( $item_ids, $order, $status = 'expired' ) {
		// NOTE : Similar to onOrderCancelled($order) until something needs to be changed?
		$this->on_order_cancelled( $item_ids, $order, $status );
	}

	/**
	 * Updates membership status smartly.
	 *
	 * @param $user_id
	 * @param $level_id
	 * @param $status (e.g. inactive, cancelled, admin_cancelled, expired) Defaults to 'inactive'.)
	 * @param $order
	 *
	 * @return boolean
	 */
	public function update_membership_status( $user_id, $level_id, $status, $order ) {

		// TODO : Do we need to know current user already have access to levelId
		if ( function_exists( 'pmpro_getMembershipLevelsForUser' ) ) {
			$userMemberships = pmpro_getMembershipLevelsForUser( $user_id );
		}
		$membership_assigned = false;
		$update_flag         = false;

		switch ( $status ) {
			case 'completed' :
				// Usage : 'pmpro_changeMembershipLevel($levelId, $userId)'
				$membership_assigned = pmpro_changeMembershipLevel( $level_id, $user_id );
				$update_flag         = true;
				break;

			// NOTE : Merging all 3 belows cases
			case 'cancelled' :
			case 'expired':
			case 'inactive':
				$new_level           = 0;
				$membership_assigned = pmpro_changeMembershipLevel( $new_level, $user_id, $status, $level_id );
				$update_flag         = true;
				break;

		}

		return $membership_assigned;
	}

	/**
	 * Handle bbapp ajax iap_linking_options for this integration
	 *
	 * @param $results Levels array
	 * @param $search : Search term
	 *
	 * @return mixed
	 */
	public function iap_linking_options( $results ) {
		$pmProLevels = PmProMembershipSupport::instance()->get_pmpro_levels();

		foreach ( $pmProLevels as $pmProLevel ) {
			$results[] = [
				'id'   => $pmProLevel->id . ':' . $pmProLevel->name,
				'text' => $pmProLevel->name . ':' . $pmProLevel->id
			];
		}

		return $results;
	}

	/**
	 * Get item edit link to show on order page.
	 *
	 * @param $link
	 * @param $item_id
	 *
	 * @return string
	 */
	public function item_id_permalink( $link, $item_id ) {
		return 'admin.php?page=pmpro-membershiplevels&edit=' . $item_id;
	}

	/**
	 * Check given post has given membership configured or not
	 *
	 * @param $has_access
	 * @param $forItemId
	 * @param $integration_item_id
	 *
	 * @return bool
	 */
	public function is_purchase_available( $has_access, $forItemId, $integration_item_id ) {
		$coursePmProLevels = get_option( '_level_course_option' );
		$levelAsArray      = explode( ",", $coursePmProLevels[ $forItemId ] );

		return in_array( $integration_item_id, $levelAsArray );
	}

	/**
	 * Check given integration item has access.
	 *
	 * @param $item_ids
	 * @param $user_id
	 *
	 * @return false
	 */
	function has_access( $item_ids, $user_id ) {
		if ( is_admin() ) {
			return true;
		}
		$has_access = false;

		foreach ( $item_ids as $item_identifier ) {
			$split    = explode( ':', $item_identifier );
			$level_id = $split[0];
			$plan = pmpro_hasMembershipLevel( $level_id, $user_id );
			if ( $plan ) {
				$has_access = true;
				break;
			}
		}
		return $has_access;
	}

	/**
	 * Handle bbapp ajax iap_integration_ids for this integration
	 *
	 * @param $results levels data
	 * @param $integration_ids selected integration id
	 * @param $slug IAP integration slug
	 *
	 * @return mixed
	 */
	public function iap_integration_ids( $results, $integration_ids ) {
		foreach ( $integration_ids as $key => $integration_id ) {
			$results[ $key ]['id']   = $integration_id;
			$results[ $key ]['text'] = '';
			$levels                  = PmProMembershipSupport::instance()->get_pmpro_levels();

			foreach ( $levels as $key => $level ) {
				/**
				 * Split[ level id, level name ];
				 */
				$split = explode( ':', $integration_id );

				if ( $level->id === (int) $split[0] ) {
					$ui_friendly_results[ $key ]['text'] = $level->name;
				}
			}
		}

		return $results;
	}
}
