<?php
/**
 * Holds Paid Membership Pro functionality related Access Controls.
 *
 * @package BuddyBossApp\Integrations\PmProMembership
 */

namespace BuddyBossApp\Integrations\PmProMembership;

use BuddyBossApp\Integrations\PmPro\PmProMembershipSupport;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Paid Membership Pro class for Access Controls.
 */
class AccessControls extends \BuddyBossApp\AccessControls\Integration_Abstract {

	/**
	 * Conditiona name.
	 *
	 * @var string $condition_name condition name.
	 */
	private $condition_name = 'paid-membership-pro';

	/**
	 * Limits items per page.
	 *
	 * @var int $limit Per page.
	 */
	private $limit = 0;

	/**
	 * Function to set up the conditions.
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed|void
	 */
	public function setup() {
		$this->register_condition(
			array(
				'condition'              => $this->condition_name,
				'items_callback'         => array( $this, 'paid_membership_pro_items_callback' ),
				'item_callback'          => array( $this, 'paid_membership_pro_item_callback' ),
				'users_callback'         => array( $this, 'paid_membership_pro_users_callback' ),
				'labels'                 => array(
					'condition_name'          => __( 'Paid Membership Pro', 'buddyboss-app' ),
					'item_singular'           => __( 'Membership', 'buddyboss-app' ),
					'member_of_specific_item' => __( 'Has specific membership', 'buddyboss-app' ),
					'member_of_any_items'     => __( 'Has any membership', 'buddyboss-app' ),
				),
				'support_any_items'      => true,
				'has_any_items_callback' => array( $this, 'has_any_items_callback' ),
			)
		);

		$this->load_hooks();
	}

	/**
	 * Function to load all hooks of this condition.
	 *
	 * @since 1.5.2.1
	 */
	public function load_hooks() {
		add_action(
			'pmpro_after_change_membership_level',
			array(
				$this,
				'bbapp_pmpro_after_change_membership_level',
			),
			10,
			3
		);
		add_action(
			'pmpro_before_change_membership_level',
			array(
				$this,
				'bbapp_pmpro_before_change_membership_level',
			),
			10,
			4
		);
		add_action(
			'pmpro_delete_membership_level',
			array(
				$this,
				'bbapp_pmpro_delete_membership_level',
			),
			10,
			1
		);
	}

	/**
	 * Items callback method.
	 *
	 * @param string $search Search the condition.
	 * @param int    $page   Page number.
	 * @param int    $limit  Limit the items to be fetched.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function paid_membership_pro_items_callback( $search = '', $page = 1, $limit = 20 ) {
		$list          = array();
		$pm_pro_levels = PmProMembershipSupport::instance()->get_pmpro_levels();

		if ( ! empty( $pm_pro_levels ) ) {
			foreach ( $pm_pro_levels as $pm_pro_level ) {
				$list[ $pm_pro_level->id ] = array(
					'id'   => $pm_pro_level->id,
					'name' => $pm_pro_level->name,
					'link' => admin_url( "admin.php?page=pmpro-membershiplevels&edit={$pm_pro_level->id}" ),
				);
			}
		}

		return $this->paginate_items_list( $list, $page, $limit, $search );
	}

	/**
	 * Item callback method.
	 *
	 * @param int $item_value Item value of condition.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array|false
	 */
	public function paid_membership_pro_item_callback( $item_value ) {
		$membership = pmpro_getLevel( $item_value );

		if ( empty( $membership ) || false === $membership ) {
			return false;
		}

		return array(
			'id'   => $membership->id,
			'name' => $membership->name,
			'link' => admin_url( "admin.php?page=pmpro-membershiplevels&edit={$membership->id}" ),
		);
	}

	/**
	 * Users callback method.
	 *
	 * @param array $data     condition data.
	 * @param int   $page     current page number.
	 * @param int   $per_page limit.
	 *
	 * @since 1.5.2.1
	 * @return array
	 */
	public function paid_membership_pro_users_callback( $data, $page = 1, $per_page = 10 ) {
		$this->limit = $per_page;

		if ( ! class_exists( 'PMPro_Members_List_Table' ) ) {
			return self::instance()->return_error( __( 'PMPro_Members_List_Table class is not loaded', 'buddyboss-app' ) );
		}

		$sub_condition     = ( ! empty( $data['sub_condition'] ) ) ? $data['sub_condition'] : '';
		$item_value        = ( ! empty( $data['item_value'] ) ) ? $data['item_value'] : '';
		$_REQUEST['paged'] = $page;

		if ( 'specific' === $sub_condition ) {
			$_REQUEST['l'] = $item_value;
		}

		$user_list_table = new \PMPro_Members_List_Table();

		if ( ! class_exists( 'ReflectionObject' ) ) {
			return self::instance()->return_error( __( 'ReflectionObject class is not loaded', 'buddyboss-app' ) );
		}

		add_filter( 'users_per_page', array( $this, 'users_per_page' ) );

		// sql_table_data method is private, so we can not access it directly.
		$reflector = new \ReflectionObject( $user_list_table );
		$method    = $reflector->getMethod( 'sql_table_data' );
		$method->setAccessible( true );
		$user_data = $method->invoke( $user_list_table );

		remove_filter( 'users_per_page', array( $this, 'users_per_page' ) );

		$user_ids = ( ! empty( $user_data ) ) ? wp_list_pluck( $user_data, 'ID' ) : array();

		return $this->return_users( $user_ids );
	}

	/**
	 * Function to make per page work with our class.
	 *
	 * @param int $per_page Items per page.
	 *
	 * @since 1.5.2.1
	 *
	 * @return int
	 */
	public function users_per_page( $per_page ) {
		return ( ! empty( $this->limit ) ) ? $this->limit : $per_page;
	}

	/**
	 * Function to check if user has any condition.
	 *
	 * @param int $user_id User id to check.
	 *
	 * @since 1.5.2.1
	 *
	 * @return bool
	 */
	public function has_any_items_callback( $user_id ) {
		$levels = pmpro_getMembershipLevelsForUser( $user_id );

		return ( ! empty( $levels ) ) ? true : false;
	}

	/**
	 * This function to add/remove user from access control.
	 *
	 * @param int $level_id     ID of the level changed to.
	 * @param int $user_id      ID of the user changed.
	 * @param int $cancel_level ID of the level being cancelled if specified.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_pmpro_after_change_membership_level( $level_id, $user_id, $cancel_level ) {
		if ( empty( $user_id ) ) {
			return;
		}
		if ( $level_id ) {
			$this->condition_add_user( $user_id, $this->condition_name, $level_id );
		}
		if ( $cancel_level ) {
			$this->condition_remove_user( $user_id, $this->condition_name, $cancel_level );
		}
	}

	/**
	 * This function to remove user from access control.
	 *
	 * @param int   $level_id     ID of the level changed to.
	 * @param int   $user_id      ID of the user changed.
	 * @param array $old_levels   array of prior levels the user belonged to.
	 * @param int   $cancel_level ID of the level being cancelled if specified.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_pmpro_before_change_membership_level( $level_id, $user_id, $old_levels, $cancel_level ) {
		if ( empty( $user_id ) ) {
			return;
		}
		if ( ! empty( $old_levels ) ) {
			foreach ( $old_levels as $old_level ) {
				if ( empty( $old_level->ID ) || ( (int) $old_level->ID !== (int) $cancel_level ) ) {
					continue;
				}

				$this->condition_remove_user( $user_id, $this->condition_name, $old_level->ID );
			}
		}
	}

	/**
	 * Function will remove member/pmpro membership from access groups when remove pmpro membership from admin.
	 *
	 * @param int $ml_id Membership level Id.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_pmpro_delete_membership_level( $ml_id ) {
		$this->condition_item_deleted( $this->condition_name, $ml_id );
		$this->condition_remove_all_users( $this->condition_name, $ml_id );
		bb_access_recalculate_member_for_has_any_membership_group( $this->condition_name );
	}
}
