<?php

namespace BuddyBossApp\Integrations\Memberium;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\InAppPurchases\Controller;
use BuddyBossApp\InAppPurchases\IntegrationAbstract;
use BuddyBossApp\InAppPurchases\Orders;

// MemberPress Integration for BuddyBossApp InAppPurchases.
final class IAP extends IntegrationAbstract {

	private static $instance = null;
	private static $memberium_app = null;
	private static $m4ac = false;

	/**
	 * MemberpressIntegration constructor.
	 */
	private function __construct() {
		// ... leave empty, see Singleton below
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return Controller|null
	 */
	public static function instance() {
		global $i2sdk;

		if ( null === self::$instance ) {
			$className      = __CLASS__;
			self::$instance = new $className;
		}

		self::$memberium_app = memberium_app();

		if ( 'm4ac' == MEMBERIUM_SKU ) {
			self::$m4ac          = true;
		}

		return self::$instance;
	}

	/**
	 * Overriding the parent(from base-class) function
	 *
	 * @param $integration_type
	 * @param $integration_label
	 */
	public function set_up( $integration_type, $integration_label ) {

		$this->integration_slug = Controller::$memberium_slug;

		parent::set_up( $integration_type, $integration_label );

		$this->item_label = __( 'Membership', 'buddyboss-app' );

		// Register Instance
		bbapp_iap()->integration[ $integration_type ] = $this::instance();

		// NOTE : Customized actions
		add_action( 'save_bbapp_iap_product', array( $this, 'save_settings' ), 10, 2 );

	}

	/**
	 * Below function get triggers when(hook) order is completed.
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return mixed
	 */
	public function on_order_completed( $item_ids, $order ) {

		// Verify if Memberium plugin is active
		if ( ! defined( 'MEMBERIUM_SKU' ) ) {
			Orders::instance()->add_history( $order->id, 'iap-error', __( "Error while creating transaction as Memberium plugin is not found.", 'buddyboss-app' ) );

			return false;
		}

		$contact_id = memb_getContactIdByUserId( $order->user_id );
		if ( empty( $contact_id ) ) {
			$contact_id = $this->add_contact( $order->user_id );
		}
		if ( ! empty( $contact_id ) ) {
			foreach ( $item_ids as $item_identifier ) {
				$split    = explode( ':', $item_identifier );
				$level_id = $split[0];
				if ( ! memb_hasAnyTags( $level_id, $contact_id ) ) {
					memb_setTags( $level_id, $contact_id, true );
					$this->sync_contact( $contact_id, (int) $order->user_id, $order );
					Orders::instance()->add_history( $order->id, 'info', __( "Assigned tag for user into infusion soft. Tag ID is #$level_id", 'buddyboss-app' ) );
				}
			}
		} else {
			Orders::instance()->add_history( $order->id, 'info', __( "Contact not synced with Infusionsoft.", 'buddyboss-app' ) );
		}
	}

	/**
	 * Below function get triggers when(hook) order is activated.
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return mixed
	 */
	public function on_order_activate( $item_ids, $order ) {
		// NOTE : Similar to onOrderCompleted($order) until something needs to be changed?
		return $this->on_order_completed( $item_ids, $order );
	}

	/**
	 * Below function get triggers when(hook) order is cancelled.
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return mixed
	 */
	public function on_order_cancelled( $item_ids, $order ) {

		// Verify if Memberium plugin is active
		if ( ! defined( 'MEMBERIUM_SKU' ) ) {
			Orders::instance()->add_history( $order->id, 'iap-error', __( "Error while creating transaction as Memberium plugin is not found.", 'buddyboss-app' ) );

			return false;
		}

		$contact_id = memb_getContactIdByUserId( $order->user_id );
		if ( ! empty( $contact_id ) ) {
			foreach ( $item_ids as $item_identifier ) {
				$split    = explode( ':', $item_identifier );
				$level_id = $split[0];
				if ( memb_hasAnyTags( $level_id, $contact_id ) ) {
					$rm_level = - $level_id;
					memb_setTags( $rm_level, $contact_id, true );
					$this->sync_contact( $contact_id, $order->user_id, $order );
					Orders::instance()->add_history( $order->id, 'info', __( "Removed tag for user from infusion soft. Tag ID is #$level_id", 'buddyboss-app' ) );
				}
			}
		} else {
			Orders::instance()->add_history( $order->id, 'info', __( "Contact not synced with Infusionsoft.", 'buddyboss-app' ) );
		}
	}

	/**
	 * Below function get triggers when(hook) order is expired.
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return mixed
	 */
	public function on_order_expired( $item_ids, $order ) {
		// NOTE : Similar to onOrderCancelled($order) until something needs to be changed?
		$this->on_order_cancelled( $item_ids, $order );
	}

	function iap_linking_options( $results ) {
		if ( 'm4ac' == MEMBERIUM_SKU ) {
			$levels = get_option( MEMBERIUM_MEMBERSHIP_SETTINGS, [] );
		} else {
			$settings = get_option( 'memberium', [] );
			$levels   = $settings['memberships'] ? $settings['memberships'] : [];
		}
		if ( ! empty( $levels ) ) {
			foreach ( $levels as $level_id => $level ) {
				$results[] = [
					'id'   => $level_id . ':' . $level['name'],
					'text' => $level['name'] . ':' . $level_id,
				];
			}
		}

		return $results;
	}

	function iap_integration_ids( $results, $integration_ids ) {
		if ( 'm4ac' == MEMBERIUM_SKU ) {
			$levels = get_option( MEMBERIUM_MEMBERSHIP_SETTINGS, [] );
		} else {
			$settings = get_option( 'memberium', [] );
			$levels   = $settings['memberships'] ? $settings['memberships'] : [];
		}
		if ( ! empty( $levels ) ) {
			foreach ( $integration_ids as $key => $integration_id ) {
				$results[ $key ]['id']   = $integration_id;
				$results[ $key ]['text'] = "Not found: $integration_id";
				foreach ( $levels as $level_id => $level ) {
					/**
					 * Split[ level id, level name ];
					 */
					$split = explode( ':', $integration_id );
					if ( $level_id === (int) $split[0] ) {
						$results[ $key ]['text'] = $level['name'];
						break;
					}
				}
			}
		}

		return $results;
	}

	function item_id_permalink( $link, $item_id ) {
		return "admin.php?page=memberium-memberships&action=edit&id=" . $item_id;
	}

	function is_purchase_available( $is_available, $item_id, $integration_item_id ) {

		if ( self::$m4ac ) {
			$course_level = get_post_meta( $item_id, '_memberium_lms_autoenroll', true );
		} else {
			$course_level = get_post_meta( $item_id, '_is4wp_membership_levels', true );
		}

		if ( is_string( $course_level ) ) {
			$course_level = explode( ',', $course_level );
		}

		return in_array( $integration_item_id, $course_level );
	}

	/**
	 * Check given integration item has access.
	 *
	 * @param $item_ids
	 * @param $user_id
	 *
	 * @return false
	 */
	function has_access( $item_ids, $user_id ) {
		if ( is_admin() ) {
			return true;
		}
		$has_access = false;

		$contact_id = memb_getContactIdByUserId( $user_id );
		if ( ! empty( $contact_id ) ) {
			foreach ( $item_ids as $item_identifier ) {
				$split    = explode( ':', $item_identifier );
				$level_id = $split[0];
				if ( memb_hasAnyTags( $level_id, $contact_id ) ) {
					$has_access = true;
				}
			}
		}

		return $has_access;
	}

	/**
	 * Create a contact for use on infusionsoft/activecampaign
	 *
	 * @param $user_id
	 *
	 * @return int
	 */
	public function add_contact( $user_id ) {
		$user = get_userdata( $user_id );

		if ( self::$m4ac ) {
			$result = self::$memberium_app->ACAPI->api( 'contact/view?email=' . urlencode( $user->user_email ) );
			if ( ! empty( $result->success ) && 200 === $result->http_code && ! empty( $result->id ) ) {
				$contact_id = (int) $result->id;
			} else {
				$contact_details = array(
					'email'      => $user->user_email,
					'first_name' => $user->first_name,
					'last_name'  => $user->last_name,
				);
				$meta_key        = 'memberium_ac_contact_id';
				$result          = self::$memberium_app->ACAPI->api( 'contact/add', $contact_details );
				if ( ! empty( $result->success ) && 200 === $result->http_code ) {
					$contact_id = (int) $result->subscriber_id;
				}
			}
		} else {
			$contact_details = array(
				'Email'     => $user->user_email,
				'FirstName' => $user->first_name,
				'LastName'  => $user->last_name,
			);
			$meta_key        = 'infusionsoft_user_id';
			$contact_id      = self::$memberium_app->isdk->addWithDupCheck( $contact_details, $checkType = 'Email' );
		}

		if ( ! empty( $contact_id ) ) {
			update_user_meta( $user_id, $meta_key, $contact_id );
		}

		return $contact_id;
	}

	/**
	 * Sync User data to infusionsoft/activecampaign
	 *
	 * @param $contact_id
	 * @param null $user
	 * @param null $order
	 */
	public function sync_contact( $contact_id, $user = null, $order = null ) {

		if ( is_numeric( $user ) ) {
			$user = get_userdata( $user );
		}

		if ( ! $user instanceof \WP_User ) {
			return;
		}

		if ( self::$m4ac ) {

			self::$memberium_app->ACAPI->api( 'contact/sync', [ 'id' => $contact_id, 'email' => $user->user_email ] );

			if ( method_exists( self::$memberium_app, 'wpalm4is_w01_xcq48k' ) ) {
				self::$memberium_app->wpalm4is_w01_xcq48k( $user->user_email );
			} else {
				if ( is_object( $order ) ) {
					Orders::instance()->add_history( $order->id, 'info', __( "Contact sync incomplete due to wpalm4is_w01_xcq48k method not found.", 'buddyboss-app' ) );
				}
			}
		} else {
			memb_syncContact( $contact_id, true );
			if ( method_exists( self::$memberium_app, 'm4is_gzx9' ) ) {
				self::$memberium_app->m4is_gzx9( $user->ID );
			} else {
				if ( is_object( $order ) ) {
					Orders::instance()->add_history( $order->id, 'info', __( "Contact sync incomplete due to m4is_gzx9 method not found.", 'buddyboss-app' ) );
				}
			}
		}
	}
}
