<?php
/**
 * Holds MemberPress Courses integration functionality.
 *
 * @package BuddyBossApp\Integrations\MemberPressCourses
 */

namespace BuddyBossApp\Integrations\MemberPressCourses;

use BuddyBossApp\App\App;
use BuddyBossApp\Menus\Types\CoreMenus;
use memberpress\assignments\models\Assignment;
use memberpress\quizzes\models\Quiz;
use WP_REST_Request;
use memberpress\courses\models as lmodels;

/**
 * Main class.
 */
class Main {

	/**
	 * MemberPressCourses instance.
	 */
	private static $instance;

	/**
	 * @var array Required plugin versions.
	 */
	private $plugin_required_versions = array();

	/**
	 * MemberPressCourses constructor.
	 *
	 * @since 2.2.80
	 */
	public function __construct() {
		//Using Singleton, see instance()
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return Main
	 * @since 2.2.80
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Load hooks.
	 *
	 * @since 2.2.80
	 */
	public function load() {
		add_action( 'plugins_loaded', array( $this, 'hook' ) );
		add_action( 'init', array( $this, 'memberpress_lms_init' ) );
		if ( bbapp_is_memberpress_lms_enabled() ) {

			$this->plugin_required_versions['memberpress-developer-tools/main.php'] = array(
				'plugin_name'      => esc_html__( 'MemberPress Developer Tools', 'buddyboss-app' ),
				'version'          => '1.3.0',
				'setting_link'     => admin_url( 'admin.php?page=memberpress-addons' ),
				'is_plugin_active' => is_plugin_active( 'memberpress-developer-tools/main.php' ),
				'required'         => true,
			);

			$this->plugin_required_versions['memberpress-course-quizzes/main.php'] = array(
				'plugin_name'      => esc_html__( 'MemberPress Course Quizzes', 'buddyboss-app' ),
				'version'          => '1.0.6',
				'is_plugin_active' => is_plugin_active( 'memberpress-course-quizzes/main.php' ),
				'required'         => false,
			);

			$this->plugin_required_versions['memberpress-course-assignments/main.php'] = array(
				'plugin_name'      => esc_html__( 'MemberPress Course Assignments', 'buddyboss-app' ),
				'version'          => '1.0.3',
				'is_plugin_active' => is_plugin_active( 'memberpress-course-assignments/main.php' ),
				'required'         => false,
			);
		}

		if ( bbapp_is_memberpress_lms_enabled() && ! empty( $_GET['mobile-view-content'] ) ) {
			add_filter( 'mpcs_classroom_page_router', array( $this, 'mpcs_course_remove_curriculum' ), 99, 2 );

			add_filter( 'the_content', array( $this, 'mpcs_mobile_view_prepare_content' ), 1 );

			add_filter( 'the_content', array( $this, 'mpcs_mobile_view_remove_navigation' ), 11 );
		}
	}

	/**
	 * Hooks and Filters.
	 *
	 * @since 2.2.80
	 */
	public function hook() {
		if ( bbapp_is_memberpress_lms_enabled() ) {
			add_action( 'bbapp_load_core_menus', array( $this, 'register_screen' ) );
			$this->local_features();

			// Add Memberpress key in settings response.
			add_filter( 'bbapp_settings_rest_response', array( $this, 'mp_courses_api_auth_key' ), 10, 2 );
		}
	}

	/**
	 * Check the installed plugin update.
	 *
	 * @since 2.2.80
	 */
	public function check_plugin_version_validation() {
		$required_update = array();

		foreach ( $this->plugin_required_versions as $plugin_name => $plugin_info ) {
			// Skip version validation for specific plugins unless they are active
			if ( in_array( $plugin_name, array( 'memberpress-course-quizzes/main.php', 'memberpress-course-assignments/main.php' ), true ) && ! $plugin_info['is_plugin_active'] ) {
				continue;
			}

			$plugin_data = get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin_name );

			// Logic to validate plugin requirements
			$plugin_status = array(
				'plugin_name'      => $plugin_info['plugin_name'],
				'current_version'  => $plugin_data['Version'] ?? esc_html__( 'Not Installed', 'buddyboss-app' ),
				'latest_version'   => $plugin_info['version'],
				'is_plugin_active' => $plugin_info['is_plugin_active'],
				'required'         => $plugin_info['required'],
			);

			// Check if the plugin is missing or not activated
			if ( empty( $plugin_data['Version'] ) || ! $plugin_info['is_plugin_active'] ) {
				$plugin_status['issue'] = esc_html__( 'Plugin not installed or inactive', 'buddyboss-app' );
				$required_update[]      = $plugin_status;
				continue;
			}

			// Version validation
			if ( version_compare( $plugin_data['Version'], $plugin_info['version'], '<' ) ) {
				$plugin_status['issue'] = esc_html__( 'Update required', 'buddyboss-app' );
				$required_update[]      = $plugin_status;
				continue;
			}

			// Additional checks for specific plugins
			if ( 'memberpress-developer-tools/main.php' === $plugin_name && ! is_plugin_active( 'memberpress-courses/main.php' ) ) {
				$plugin_status['issue'] = esc_html__( 'Requires MemberPress Courses plugin', 'buddyboss-app' );
				$required_update[]      = $plugin_status;
				continue;
			}
		}

		return $required_update;
	}

	/**
	 * Add Memberpress key in settings response.
	 *
	 * @param array $response Response.
	 * @param WP_REST_Request $request Request.
	 *
	 * @return array
	 * @since 2.2.80
	 */
	public function mp_courses_api_auth_key( $response, $request ) {
		$response['mp_api_auth_key'] = get_option( 'mpdt_api_key', '' );

		return $response;
	}

	/**
	 * Load Segment.
	 * @since 2.2.80
	 */
	public function memberpress_lms_init() {
		if ( bbapp_is_memberpress_lms_enabled() ) {
			IAPRest::instance();
			add_filter( 'bbapp_user_segment_integrations', array( $this, 'load_segment' ), 11 );
			add_filter( 'user_has_cap', array( $this, 'custom_user_has_cap' ), 10, 4 );
			add_filter( 'mepr_developer_tools_rest_item_response', array( $this, 'rest_add_extra_data' ), 10, 2 );

			add_filter( 'bbapp_block_filter_options', array( $this, 'add_filter_options' ), 10, 2 );
			add_filter( 'bbapp_block_filter_list', array( $this, 'add_filter_list' ), 10, 2 );
			add_filter( 'bbapp_block_query_args', array( $this, 'add_query_args' ), 10, 2 );
		}
	}

	/**
	 * Load MemberPress LMS related user segments
	 *
	 * @param array $integrations List of integration.
	 *
	 * @since 2.2.80
	 * @return mixed
	 */
	public function load_segment( $integrations ) {
		$integrations['memberpress_lms'] = new Segment();

		return $integrations;
	}

	/**
	 * @param CoreMenus $core_menu
	 *
	 * @since 2.2.80
 	 * @return void
	 */
	public function register_screen( $core_menu ) {
		$core_menu->register_screen(
			'courses_all',
			__( 'Courses', 'buddyboss-app' ),
			array(
				'id'   => 'course',
				'type' => 'buddyboss',
			)
		);

		$core_menu->register_screen(
			'my_library',
			__( 'My Library', 'buddyboss-app' ),
			array(
				'id'   => 'books',
				'type' => 'buddyboss',
			)
		);
	}

	/**
	 * Add location features related to social features.
	 *
	 * @since 2.2.80
	 */
	public function local_features() {
		// LMS Support.
		App::instance()->add_local_feature(
			'memberpress_lms',
			array(
				'is_enabled_android' => true,
				'is_enabled_ios'     => true,
			)
		);
	}

	/**
	 * Custom user capabilities for MemberPress LMS
	 *
	 * @param array   $allcaps An array of all the user's capabilities.
	 * @param array   $caps    The actual capabilities being checked.
	 * @param array   $args    Adds the context to the cap. Typically the object ID.
	 * @param \WP_User $wp_user The user object.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function custom_user_has_cap( $allcaps, $caps, $args, $wp_user ) {
		// Define specific routes and methods allowed for certain users
		$allowed_routes = array(
			'mp/v1/courses'     => array( 'GET', 'POST' ),
			'mp/v1/lessons'     => array( 'GET', 'POST' ),
			'mp/v1/quizzes'     => array( 'GET', 'POST' ),
			'mp/v1/assignments' => array( 'GET', 'POST' ),
		);

		// Define a dynamic route with DELETE method
		$dynamic_routes = array(
			'#/mp/v1/assignments/(?P<id>[\d]+)/submissions/(?P<user_id>[\d]+)$#' => array( 'DELETE' ),
		);

		// Ensure this is a REST request and we have a valid user
		if ( defined( 'REST_REQUEST' ) && REST_REQUEST && isset( $wp_user->ID ) && $wp_user->ID ) {
			$request_uri    = untrailingslashit( $_SERVER['REQUEST_URI'] );
			$request_method = strtoupper( $_SERVER['REQUEST_METHOD'] );
			$rest_prefix    = untrailingslashit( rest_get_url_prefix() );

			// Check static routes
			foreach ( $allowed_routes as $route => $methods ) {
				$target_endpoint = '/' . $rest_prefix . '/' . $route;

				if ( strpos( $request_uri, $target_endpoint ) !== false && in_array( $request_method, $methods, true ) ) {
					$allcaps['remove_users'] = true;
				}
			}

			// Check for the main batch endpoint
			if ( false !== strpos( $request_uri, '/' . $rest_prefix . '/buddyboss-app/v1/batch' ) && 'POST' === $request_method ) {
				// Decode the batch payload to check each request in the batch
				$batch_payload = json_decode( file_get_contents( 'php://input' ), true );

				if ( isset( $batch_payload['requests'] ) && is_array( $batch_payload['requests'] ) ) {
					foreach ( $batch_payload['requests'] as $sub_request ) {
						// Remove query parameters from the path
						$sub_request_path   = isset( $sub_request['path'] ) ? parse_url( untrailingslashit( ltrim( $sub_request['path'], '/' ) ), PHP_URL_PATH ) : '';
						$sub_request_method = isset( $sub_request['method'] ) ? strtoupper( $sub_request['method'] ) : '';

						// Check if this specific path and method are allowed in the batch
						if ( isset( $allowed_routes[ $sub_request_path ] ) && in_array( $sub_request_method, $allowed_routes[ $sub_request_path ], true ) ) {
							// If any sub-request is not allowed, deny the capability
							$allcaps['remove_users'] = true;
						}
					}
				}
			}

			// Check dynamic routes with regex
			foreach ( $dynamic_routes as $pattern => $methods ) {
				if ( preg_match( $pattern, $request_uri ) && in_array( $request_method, $methods, true ) ) {
					$allcaps['remove_users'] = true;
				}
			}
		}

		return $allcaps;
	}

	/**
	 * Add extra data to the rest response.
	 *
	 * @param object $item     Post object.
	 * @param object $request  Request object.
	 *
	 * @since 2.2.80
	 * @return mixed
	 */
	public function rest_add_extra_data( $item, $request ) {
		$user_id                    = get_current_user_id();
		$course_started             = get_user_meta( $user_id, 'mpcs_course_started_' . $item['id'], true );
		$course_progress            = \memberpress\courses\models\UserProgress::has_started_course( $user_id, $item['id'] );
		$item['has_course_started'] = $course_started || $course_progress;

		return $item;
	}

	/**
	 * Remove navigation when visited from the app.
	 *
	 * @param string $content The content to be filtered.
	 *
	 * @since 2.3.21
	 *
	 * @return string
	 */
	public function mpcs_mobile_view_remove_navigation( $content ) {
		global $post;

		if (
			( $post instanceof \WP_Post )
			&& is_single()
			&& is_user_logged_in()
		) {
			if ( class_exists( '\memberpress\courses\models\Lesson' ) && property_exists( '\memberpress\courses\models\Lesson', 'cpt' ) && $post->post_type === lmodels\Lesson::$cpt ) {
				$content = preg_replace( '/<div id="mpcs-lesson-navigation".*?<\/div>/s', '', $content );
			} elseif ( class_exists( '\memberpress\assignments\models\Assignment' ) && property_exists( '\memberpress\assignments\models\Assignment', 'cpt' ) && $post->post_type === Assignment::$cpt ) {
				$content = preg_replace( '/<div id="mpcs-assignment-navigation".*?<\/div>/s', '', $content );
			} elseif ( class_exists( '\memberpress\quizzes\models\Quiz' ) && property_exists( '\memberpress\quizzes\models\Quiz', 'cpt' ) && $post->post_type === Quiz::$cpt ) {
				$content = preg_replace( '/<div id="mpcs-quiz-navigation".*?<\/div>/s', '', $content );
			}
		}

		return $content;
	}

	/**
	 * Remove additional content sections when visited from the app.
	 *
	 * @param string $content The content to be filtered.
	 *
	 * @since 2.3.21
	 *
	 * @return string
	 */
	public function mpcs_mobile_view_prepare_content( $content ) {
		global $post;

		if (
			( $post instanceof \WP_Post )
			&& is_single()
			&& is_user_logged_in()
		) {
			if ( class_exists( '\memberpress\assignments\models\Assignment' ) && property_exists( '\memberpress\assignments\models\Assignment', 'cpt' ) && $post->post_type === Assignment::$cpt ) {
				$content = preg_replace( '/<!-- wp:memberpress-courses\/assignment-submission.*?\/-->/s', '', $content );
			} elseif ( class_exists( '\memberpress\quizzes\models\Quiz' ) && property_exists( '\memberpress\quizzes\models\Quiz', 'cpt' ) && $post->post_type === Quiz::$cpt ) {
				$content = preg_replace( '/<!-- wp:memberpress-courses\/.*?{"questionId":.*?\/-->/s', '', $content );
			}
		}

		return $content;
	}

	/**
	 * Remove course overview from the course page when visited from the app.
	 *
	 * @param string $content The content to be filtered.
	 * @param string $action Current tab.
	 *
	 * @since 2.3.21
	 *
	 * @return string
	 */
	public function mpcs_course_remove_curriculum( $content, $action ) {
		global $post;

		if (
			( $post instanceof \WP_Post )
			&& $post->post_type === \memberpress\courses\models\Course::$cpt
			&& is_single()
			&& empty( $action )
			&& method_exists( '\memberpress\courses\helpers\Courses', 'display_course_overview' )
		) {
			$course_overview = \memberpress\courses\helpers\Courses::display_course_overview();
			$content         = str_replace( $course_overview, '', $content );
		}

		return $content;
	}

	/**
	 * Add filter options.
	 *
	 * @param array  $filter_options Filter options.
	 * @param string $post_type      Post type.
	 *
	 * @since 2.3.40
	 * @return array
	 */
	public function add_filter_options( $filter_options, $post_type ) {
		if ( 'mpcs-course' === $post_type ) {
			$filter_options[] = array(
				'name'  => 'my_courses',
				'label' => __( 'My Courses', 'buddyboss-app' ),
			);
		}

		return $filter_options;
	}

	/**
	 * Add filter list.
	 *
	 * @param array  $filter_list Filter list.
	 * @param string $post_type   Post type.
	 *
	 * @since 2.3.40
	 * @return array
	 */
	public function add_filter_list( $filter_list, $post_type ) {
		if ( 'mpcs-course' === $post_type ) {
			$my_courses_options = array(
				array(
					'label' => __( 'Yes', 'buddyboss-app' ),
					'key'   => true
				),
				array(
					'label' => __( 'No', 'buddyboss-app' ),
					'key'   => false
				)
			);

			$filter_list['my_courses'] = $my_courses_options;
		}

		return $filter_list;
	}

	/**
	 * Add query args.
	 *
	 * @param array  $query_args Query args.
	 * @param string $request  Request.
	 *
	 * @since 2.3.40
	 * @return array
	 */
	public function add_query_args( $query_args, $request ) {
		if ( 'mpcs-course' === $query_args['post_type'] ) {
			if ( isset( $query_args['my_courses'] ) && $query_args['my_courses'] ) {
				$user_id           = get_current_user_id();
				$mpdt_course_utils = new \MpdtCourseUtils();
				$mycourse_ids      = $mpdt_course_utils->get_user_course_ids( $user_id );

				if ( ! empty( $mycourse_ids ) && ! is_wp_error( $mycourse_ids ) ) {
					$query_args['post__in'] = ! empty( $query_args['post__in'] ) ? array_intersect( $mycourse_ids, $query_args['post__in'] ) : $mycourse_ids;
				}

				/*
				 * If we intersected, but there are no post ids in common,
				 * WP_Query won't return "no posts" for post__in = array()
				 * so we have to fake it a bit.
				*/
				if ( ! $query_args['post__in'] ) {
					$query_args['post__in'] = array( 0 );
				}

				unset( $query_args['my_courses'] );
			}
		}

		return $query_args;
	}
}
