<?php
/**
 * Holds MemberPress related functionality for IAP.
 *
 * @package BuddyBossApp\InAppPurchases
 */

namespace BuddyBossApp\Integrations\MemberPressCourses;

use BuddyBossApp\Admin\InAppPurchases\Helpers;
use BuddyBossApp\Tools\Logger;
use MeprHooks;
use MeprRule;
use MeprRulesCtrl;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * MemberPress IAPRest class.
 */
final class IAPRest {

	/**
	 * Class instance.
	 *
	 * @var null $instance
	 */
	private static $instance = null;

	/**
	 * Product property.
	 *
	 * @var string $rest_response_product_property
	 */
	protected $rest_response_product_property = 'bbapp_products';

	/**
	 * Rest constructor.
	 * @since 2.2.80
	 */
	private function __construct() {
		// ... leave empty, see Singleton below
	}

	/**
	 * Get the instance of this class.
	 *
	 * @since 2.2.80
	 * @return object
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Define all actions and filters here
	 *
	 * @since 2.2.80
	 */
	public function hooks() {
		// Learner API Integration.
		add_filter( 'mepr_developer_tools_rest_item_response', array( $this, 'rest_integrate_iap_product' ), 20, 2 );
	}

	/**
	 * Verify need of InAppPurchase(bbapp_products) details in course-response.
	 *
	 * @param object $item    Post object.
	 * @param object $request Request object.
	 *
	 * @since 2.2.80
	 * @return mixed
	 */
	public function rest_integrate_iap_product( $item, $request ) {

		if( ! bbapp_is_active( 'iap' ) ) {
			return $item;
		}

		// This thing keep here only for backward support.
		$item = $this->rest_iap_product_response( $item, $item['id'] );

		return $item;
	}

	/**
	 * Adds InAppPurchase object into rest endpoint of attached one.
	 *
	 * @param object $rest_item Rest object.
	 * @param int    $item_id   Item id.
	 *
	 * @since 2.2.80
	 * @return array|object
	 */
	public function rest_iap_product_response( $rest_item, $item_id ) {


		if ( 'mpcs-course' !== get_post_type( $item_id ) ) {
			return $rest_item;
		}

		$found_product_ids = array();

		// NOTE : Defaults.
		$response = array(
			'integration_type'  => 'mp-course',
			'bbapp_product_ids' => $found_product_ids,
			'purchase_item_id'  => $item_id,
			'purchase_require'  => false,
		);

		$default_args = array(
			'orderby' => 'id',
			'order'   => 'desc',
		);

		// LearnDash group now working when we pass integration type.
		$product_ids = Helpers::search_integration( $item_id, '', $default_args );
		$iap_ids     = $this->bbapp_get_course_cats_and_tags( $item_id );

		if ( ! empty( $iap_ids ) ) {
			$iap_pluck         = wp_list_pluck( $iap_ids, 'id' );
			$found_product_ids = array_map( 'intval', array_merge( $found_product_ids, $iap_pluck ) );
		}

		if ( ! empty( $product_ids ) ) {
			$product_ids       = wp_list_pluck( $product_ids, 'product_id' );
			$found_product_ids = array_merge( $found_product_ids, $product_ids );
		}

		foreach ( bbapp_iap_get_types() as $platform_type => $platform_label ) {
			$response[ strtolower( $platform_type ) ]['status'] = false;
		}

		$response['bbapp_product_ids'] = array_values( array_unique( $found_product_ids ) );

		$rest_item[ $this->rest_response_product_property ] = $response;

		return $rest_item;
	}

	/**
	 * Function to get the course categories and tags.
	 *
	 * @param int $item_id Categorirs ID.
	 *
	 * @since 2.2.80
	 *
	 * @return mixed
	 */
	public function bbapp_get_course_cats_and_tags( $item_id = 0 ) {
		if ( empty( $item_id ) ) {
			return array();
		}

		$course_cats = wp_get_post_terms( $item_id, 'mpcs-course-categories', array( 'fields' => 'ids' ) );
		$course_tags = wp_get_post_terms( $item_id, 'mpcs-course-tags', array( 'fields' => 'ids' ) );

		$iap_ids = $this->bbapp_iap_get_courses_iap_products( array( $item_id ), $course_cats, $course_tags );

		return $iap_ids;
	}

	/**
	 * Function to get the MP course IAP.
	 *
	 * @param array $course_ids     specific course ids.
	 * @param array $course_cat_ids specific course category ids.
	 * @param array $course_tag_ids specific course tag ids.
	 *
	 * @since 2.2.80
	 *
	 * @return array|object
	 */
	public function bbapp_iap_get_courses_iap_products( $course_ids = array(), $course_cat_ids = array(), $course_tag_ids = array() ) {
		global $wpdb;

		$select = "SELECT DISTINCT {$wpdb->prefix}bbapp_iap_products.id FROM {$wpdb->prefix}bbapp_iap_products INNER JOIN {$wpdb->prefix}bbapp_iap_productmeta iapm1 ON ({$wpdb->prefix}bbapp_iap_products.id = iapm1.iap_id ) INNER JOIN {$wpdb->prefix}bbapp_iap_productmeta iapm2 ON ({$wpdb->prefix}bbapp_iap_products.id = iapm2.iap_id)";

		$where = array();

		$where['all_courses'] = "(iapm1.meta_key='course_access_type' AND iapm1.meta_value='all-courses')";

		if ( ! empty( $course_ids ) ) {
			$imploded_course_ids       = "'" . implode( "','", array_map( 'intval', $course_ids ) ) . "'";
			$where['specific_courses'] = "(iapm1.meta_key='course_access_type' AND iapm1.meta_value='specific-course' AND iapm2.meta_key='course_access_id' AND iapm2.meta_value IN ({$imploded_course_ids}))";
		}

		if ( ! empty( $course_cat_ids ) ) {
			$imploded_cat_ids       = "'" . implode( "','", array_map( 'intval', $course_cat_ids ) ) . "'";
			$where['specific_cats'] = "(iapm1.meta_key='course_access_type' AND iapm1.meta_value='specific-category-courses' AND iapm2.meta_key='course_access_cat_id' AND iapm2.meta_value IN ({$imploded_cat_ids}))";
		}

		if ( ! empty( $course_tag_ids ) ) {
			$imploded_tag_ids       = "'" . implode( "','", array_map( 'intval', $course_tag_ids ) ) . "'";
			$where['specific_tags'] = "(iapm1.meta_key='course_access_type' AND iapm1.meta_value='specific-tag-courses' AND iapm2.meta_key='course_access_tag_id' AND iapm2.meta_value IN ({$imploded_tag_ids}))";
		}

		$imploded_where = '(' . implode( ' OR ', $where ) . ')';
		$published_iap  = "{$wpdb->prefix}bbapp_iap_products.status='published'";

		$sql = $select . ' WHERE ' . $published_iap . ' AND ' . $imploded_where;

		$results = $wpdb->get_results( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared

		return ! empty( $results ) ? $results : array();
	}
}
