<?php
/**
 * Holds deep-linking functionality related MemberPress Courses.
 *
 * @package BuddyBossApp\Integrations\MemberPressCourses
 */

namespace BuddyBossApp\Integrations\MemberPressCourses;

use BuddyBossApp\DeepLinking\Type\TypeAbstract;
use WP_Post;
use memberpress\quizzes;
use memberpress\assignments\models as amodels;
use memberpress\courses\models as lmodels;

/**
 * Deeplink class for MP Courses.
 */
class DeepLinking extends TypeAbstract {

	/**
	 * Array post type mapping.
	 *
	 * @var array
	 */
	public $arr_post_types_mapping = array();

	/**
	 * Post types array.
	 *
	 * @var array
	 */
	public $arr_post_types = array();

	/**
	 * Construct function.
	 *
	 * @since 2.3.60
	 */
	public function __construct() {
		parent::__construct();
		$this->init();
		add_filter( 'bbapp_deeplinking_cpt_action', array( $this, 'update_cpt_action' ), 10, 2 );
		add_filter( 'bbapp_deeplinking_cpt_namespace', array( $this, 'update_cpt_namespace' ), 10, 2 );
		add_filter( 'bbapp_deeplinking_cpt_type', array( $this, 'merge_course_id' ), 10, 1 );
	}

	/**
	 * Init all MemberPress Course maps.
	 *
	 * @since 2.3.60
	 */
	public function init() {
		$this->arr_post_types_mapping = array(
			'mpcs-course'      => 'mpcs-course',
			'mpcs-lesson'      => 'mpcs-lesson',
			'mpcs-quiz'        => 'mpcs-quiz',
			'mpcs-assignment'  => 'mpcs-assignment',
			'memberpressgroup' => 'group',
		);

		$this->arr_post_types = array_keys( $this->arr_post_types_mapping );
	}

	/**
	 * Update MP Course CPT action
	 *
	 * @param string  $action Action.
	 * @param WP_Post $post Post object.
	 *
	 * @since 2.3.60
	 * @return string
	 */
	public function update_cpt_action( $action, $post ) {
		if ( $post instanceof WP_Post && in_array( $post->post_type, $this->arr_post_types, true ) ) {
			return 'open_' . $this->arr_post_types_mapping[ $post->post_type ];
		}

		return $action;
	}

	/**
	 * Update MP Course cpts namespace
	 *
	 * @param object $cpt_namespace Namespace obeject.
	 * @param string $post_type Post type.
	 *
	 * @since 2.3.60
	 * @return string
	 */
	public function update_cpt_namespace( $cpt_namespace, $post_type ) {
		if ( isset( $post_type ) && in_array( $post_type, $this->arr_post_types, true ) ) {
			return 'bbapp_lms';
		}

		return $cpt_namespace;
	}

	/**
	 * Merge course id.
	 *
	 * @param array $deep_link_data The post data.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function merge_course_id( $deep_link_data ) {
		if ( ! empty( $deep_link_data['item_id'] ) ) {
			if ( class_exists( 'memberpress\courses\models\Lesson' ) && get_post_type( $deep_link_data['item_id'] ) === lmodels\Lesson::$cpt ) {
				$lesson                      = new lmodels\Lesson( $deep_link_data['item_id'] );
				$course                      = $lesson->course();
				$deep_link_data['course_id'] = ! empty( $course->ID ) ? $course->ID : 0;
			} elseif ( class_exists( 'memberpress\assignments\models\Assignment' ) && get_post_type( $deep_link_data['item_id'] ) === amodels\Assignment::$cpt ) {
				$assignment                  = new amodels\Assignment( $deep_link_data['item_id'] );
				$course                      = $assignment->course();
				$deep_link_data['course_id'] = ! empty( $course->ID ) ? $course->ID : 0;
			} elseif ( class_exists( 'memberpress\quizzes\models\Quiz' ) && get_post_type( $deep_link_data['item_id'] ) === quizzes\models\Quiz::$cpt ) {
				$quiz                        = new quizzes\models\Quiz( $deep_link_data['item_id'] );
				$course                      = $quiz->course();
				$deep_link_data['course_id'] = ! empty( $course->ID ) ? $course->ID : 0;
			}
		}

		return $deep_link_data;
	}

	/**
	 * Inherited method.
	 *
	 * @param string $url URL.
	 *
	 * @since 2.3.60
	 * @inheritDoc
	 */
	public function parse( $url ) {
		// TODO: Implement parse() method.
	}
}
