<?php

namespace BuddyBossApp\Integrations\MemberPress;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\InAppPurchases\Controller;
use BuddyBossApp\InAppPurchases\IntegrationAbstract;
use BuddyBossApp\InAppPurchases\Orders;
use MeprTransaction;

// MemberPress Integration for BuddyBossApp InAppPurchases.
final class IAP extends IntegrationAbstract {

	private static $instance = null;

	/**
	 * MemberpressIntegration constructor.
	 */
	private function __construct() {
		// ... leave empty, see Singleton below
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return Controller|null
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$className      = __CLASS__;
			self::$instance = new $className;
		}

		return self::$instance;
	}

	/**
	 * Overriding the parent(from base-class) function
	 *
	 * @param $integration_type
	 * @param $integration_label
	 */
	public function set_up( $integration_type, $integration_label ) {

		$this->integration_slug = Controller::$memberpress_slug;

		parent::set_up( $integration_type, $integration_label );

		$this->item_label = __( 'Membership', 'buddyboss-app' );

		// Register Instance
		bbapp_iap()->integration[ $integration_type ] = $this::instance();
	}

	/**
	 * Below function get triggers when(hook) order is completed.
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return mixed
	 */
	public function on_order_completed( $item_ids, $order ) {

		// Verify if memberpress plugin is active
		if ( ! class_exists( "MeprTransaction" ) ) {
			Orders::instance()->add_history( $order->id, 'iap-error', __( "Error while creating MemberPress transaction as MemberPress plugin is not found.", 'buddyboss-app' ) );

			return false;
		}

		$trans_num = "bbapp-in-app-purchase-txn-{$order->id}-" . uniqid();
		$trans_ids = Orders::instance()->get_meta( $order->id, "_memberpress_trans_ids", true );
		$trans_ids = ! empty( $trans_ids ) ? $trans_ids : array();
		$trans_ids = is_serialized( $trans_ids ) ? maybe_unserialize( $trans_ids ) : $trans_ids;
		
		foreach ( $item_ids as $item_id ) {

			/** Note : Make sure if user already has permission or not.
			 * Usually it shouldn't have if this method is being triggered..
			 * But in case this function is called from on active func.
			 * Crucial key : memberpress_product_authorized_#
			 */
			// TODO : NOTE => Currently by-passing(breaking) this condition, not sure why we need that
			if ( user_can( $order->user_id, "memberpress_product_authorized_{itemId}" ) && "break" == "this-condition" ) {
				Orders::instance()
				      ->add_history( $order->id, 'warning', __( "Creating MemberPress transaction ID was skipped, as access was already found.", 'buddyboss-app' ) );
			} else {

				$split      = explode( ':', $item_id );
				$product_id = ! empty( $split[0] ) ? (int) $split[0] : 0;

				/**
				 * Create new transaction on order completed.
				 * NOTE : We are creating lifetime access as it has to be controlled via BuddyBossApp's InAppPurchase order instead.
				 */
				$txn                        = new MeprTransaction();
				$txn->parent_transaction_id = $order->id;
				$txn->trans_num             = $trans_num;
				$txn->amount                = 0;
				$txn->user_id               = $order->user_id;
				$txn->product_id            = $product_id;
				$txn->status                = MeprTransaction::$complete_str;
				$txn->txn_type              = MeprTransaction::$payment_str;
				// Note : No need of any payment gateway since all mobile payment will go through InAppPurchase
				$txn->gateway         = 'manual'; //bbapp-in-app-purchase: Possible?
				$txn->expires_at      = 0;
				$txn->subscription_id = 0;
				$trans_id             = $txn->store();
				$trans_ids[]          = $trans_id;

				Orders::instance()
				      ->add_history( $order->id, 'info', __( "Created new transaction on MemberPress. Transaction ID is #$trans_id and Number is : $trans_num", 'buddyboss-app' ) );
			}

		}

		//Crucial key
		Orders::instance()->update_meta( $order->id, "_memberpress_trans_ids", serialize( $trans_ids ) );
	}

	/**
	 * Below function get triggers when(hook) order is activated.
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return mixed
	 */
	public function on_order_activate( $item_ids, $order ) {
		// NOTE : Similar to onOrderCompleted($order) until something needs to be changed?
		return $this->on_order_completed( $item_ids, $order );
	}

	/**
	 * Below function get triggers when(hook) order is cancelled.
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return mixed
	 */
	public function on_order_cancelled( $item_ids, $order ) {

		$trans_ids = unserialize( Orders::instance()->get_meta( $order->id, "_memberpress_trans_ids" ) );

		if ( empty( $trans_ids ) ) {
			Orders::instance()
			      ->add_history( $order->id, 'warning', __( "Admin Error : No MemberPress Transaction ID(s) found, can't proceed for revoke operation.", 'buddyboss-app' ) );

			return false;
		}

		foreach ( $trans_ids as $trans_id ) {

			$txn = new MeprTransaction( $trans_id );

			if ( empty( $txn->id ) ) {
				if ( empty( $trans_id ) ) {
					Orders::instance()
					      ->add_history( $order->id, 'info', __( "MemberPress Transaction ID #$trans_id doesn't exists, assuming access is revoked already.", 'buddyboss-app' ) );

					return false;
				}
			}

			$txn->status = MeprTransaction::$failed_str;
			$updated_id  = $txn->store();

			Orders::instance()->add_history( $order->id, 'info', __( "Access revoked for MemberPress transaction ID #$updated_id", 'buddyboss-app' ) );
		}

	}

	/**
	 * Below function get triggers when(hook) order is expired.
	 *
	 * @param $item_ids
	 * @param $order
	 *
	 * @return mixed
	 */
	public function on_order_expired( $item_ids, $order ) {
		// NOTE : Similar to onOrderCancelled($order) until something needs to be changed?
		$this->on_order_cancelled( $item_ids, $order );
	}

	function iap_linking_options( $results ) {
		return $results;
	}

	function iap_integration_ids( $results, $integration_ids ) {
		return $results;
	}

	function item_id_permalink( $link, $item_id ) {
		return "post.php?post=$item_id&action=edit";
	}

	function is_purchase_available( $is_available, $item_id, $integration_item_id ) {
		return $is_available;
	}

	/**
	 * Check given integration item has access.
	 *
	 * @param $item_ids
	 * @param $user_id
	 *
	 * @return false
	 */
	function has_access( $item_ids, $user_id ) {
		if ( is_admin() ) {
			return true;
		}
		$has_access = false;

		foreach ( $item_ids as $item_identifier ) {
			$split    = explode( ':', $item_identifier );
			$level_id = $split[0];
			$plan = user_can( $user_id, "memberpress_product_authorized_$level_id" );
			if ( $plan ) {
				$has_access = true;
				break;
			}
		}
		return $has_access;
	}
}
