<?php
/**
 * Holds MemberPress functionality related Access Controls.
 *
 * @package BuddyBossApp\Integrations\MemberPress
 */

namespace BuddyBossApp\Integrations\MemberPress;

use MeprDb;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * MemberPress class for Access Controls.
 */
class AccessControls extends \BuddyBossApp\AccessControls\Integration_Abstract {

	/**
	 * Condition name.
	 *
	 * @var string $condition_name condition name.
	 */
	private $condition_name = 'memberpress';

	/**
	 * Function to set up the conditions.
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed|void
	 */
	public function setup() {
		$this->register_condition(
			array(
				'condition'              => $this->condition_name,
				'items_callback'         => array( $this, 'memberpress_membership_items_callback' ),
				'item_callback'          => array( $this, 'memberpress_membership_item_callback' ),
				'users_callback'         => array( $this, 'memberpress_membership_users_callback' ),
				'labels'                 => array(
					'condition_name'          => __( 'MemberPress', 'buddyboss-app' ),
					'item_singular'           => __( 'Membership', 'buddyboss-app' ),
					'member_of_specific_item' => __( 'Has specific membership', 'buddyboss-app' ),
					'member_of_any_items'     => __( 'Has any membership', 'buddyboss-app' ),
				),
				'support_any_items'      => true,
				'has_any_items_callback' => array( $this, 'has_any_items_callback' ),
			)
		);

		$this->load_hooks();
	}

	/**
	 * Function to load all hooks of this condition.
	 *
	 * @since 1.5.2.1
	 */
	public function load_hooks() {
		add_action( 'mepr-txn-transition-status', array( $this, 'bbapp_mepr_txn_transition_status' ), 10, 3 );
		add_action( 'mepr_post_delete_transaction', array( $this, 'bbapp_mepr_post_delete_transaction' ), 10, 4 );
		add_action( 'transition_post_status', array( $this, 'bbapp_remove_mepr_memberships' ), 10, 3 );
	}

	/**
	 * Items callback method.
	 *
	 * @param string $search Search the condition.
	 * @param int    $page   Page number.
	 * @param int    $limit  Limit the items to be fetched.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function memberpress_membership_items_callback( $search = '', $page = 1, $limit = 20 ) {
		$list     = array();
		$products = \MeprCptModel::all( 'MeprProduct' );

		if ( ! empty( $products ) ) {
			foreach ( $products as $product ) {
				$list[ $product->ID ] = array(
					'id'   => $product->ID,
					'name' => esc_html( $product->post_title ),
					'link' => admin_url( "post.php?post={$product->ID}&action=edit" ),
				);
			}
		}

		return $this->paginate_items_list( $list, $page, $limit, $search );
	}

	/**
	 * Item callback method.
	 *
	 * @param int $item_value Item value of condition.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array|false
	 */
	public function memberpress_membership_item_callback( $item_value ) {
		$memberships = new \MeprProduct( $item_value );

		if ( empty( $memberships ) || empty( $memberships->ID ) ) {
			return false;
		}

		return array(
			'id'   => $memberships->ID,
			'name' => get_the_title( $memberships->ID ),
			'link' => admin_url( "post.php?post={$memberships->ID}&action=edit" ),
		);
	}

	/**
	 * Users callback method.
	 *
	 * @param array $data     condition data.
	 * @param int   $page     current page number.
	 * @param int   $per_page limit.
	 *
	 * @since 1.5.2.1
	 * @return array
	 */
	public function memberpress_membership_users_callback( $data, $page = 1, $per_page = 10 ) {
		$sub_condition = ( ! empty( $data['sub_condition'] ) ) ? $data['sub_condition'] : '';
		$item_value    = ( ! empty( $data['item_value'] ) ) ? $data['item_value'] : '';
		$params        = array(
			'status' => 'active',
		);

		if ( 'specific' === $sub_condition ) {
			$params['membership'] = $item_value;
		}

		add_filter( 'mepr-list-table-joins', array( $this, 'mepr_list_table_joins_callback' ), 99, 1 );
		add_filter( 'mepr-list-table-args', array( $this, 'mepr_list_table_args_callback' ), 10, 1 );
		$list_table = \MeprUser::list_table( 'ID', 'ASC', $page, '', 'ID', $per_page, $params );
		$user_ids   = ( ! empty( $list_table['results'] ) ) ? array_unique( wp_list_pluck( $list_table['results'], 'ID' ) ) : array();
		remove_filter( 'mepr-list-table-args', array( $this, 'mepr_list_table_args_callback' ) );
		remove_filter( 'mepr-list-table-joins', array( $this, 'mepr_list_table_joins_callback' ), 99 );
		return $this->return_users( $user_ids );
	}

	/**
	 * Function to check if user has any condition.
	 *
	 * @param int $user_id User id to check.
	 *
	 * @since 1.5.2.1
	 *
	 * @return bool
	 */
	public function has_any_items_callback( $user_id ) {
		$user = new \MeprUser( $user_id );

		$active_memberships = array_unique( $user->active_product_subscriptions( 'ids' ), true );

		return ( ! empty( $active_memberships ) ) ? true : false;
	}

	/**
	 * This function add/remove user to access control.
	 *
	 * @param string $old_txn_status       Old transaction status.
	 * @param string $new_status           New transaction status.
	 * @param object $mepr_transaction_obj Memberpress transaction object.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_mepr_txn_transition_status( $old_txn_status, $new_status, $mepr_transaction_obj ) {
		if ( empty( $new_status ) || empty( $mepr_transaction_obj ) ) {
			return;
		}

		$args       = (array) $mepr_transaction_obj->get_values();
		$product_id = $args['product_id'];
		$user_id    = $args['user_id'];

		$active_status = [ \MeprTransaction::$complete_str, \MeprTransaction::$confirmed_str ];
		$now           = time();
		$expires       = 0; // Lifetime

		if ( ! empty( $mepr_transaction_obj->expires_at ) && $mepr_transaction_obj->expires_at != \MeprUtils::db_lifetime() ) {
			$expires = strtotime( $mepr_transaction_obj->expires_at );
		}

		if ( in_array( $mepr_transaction_obj->status, $active_status ) ) {
			if ( $expires === 0 || $expires >= $now ) {
				$this->condition_add_user( $user_id, $this->condition_name, $product_id );
			} else {
				$this->condition_remove_user( $user_id, $this->condition_name, $product_id );
			}
		} else {
			$this->condition_remove_user( $user_id, $this->condition_name, $product_id );
		}
	}

	/**
	 * This function remove user from access control.
	 *
	 * @param int $id     Memberpress transaction id.
	 * @param int $user   Memberpress user id.
	 * @param int $result Memberpress transaction result.
	 *
	 * @since 2.3.50
	 */
	public function bbapp_mepr_post_delete_transaction( $id, $user, $result, $mepr_txn ) {
		if ( empty( $mepr_txn ) ) {
			return;
		}

		$args       = (array) $mepr_txn->get_values();
		$product_id = $args['product_id'];
		$user_id    = $args['user_id'];

		$this->condition_remove_user( $user_id, $this->condition_name, $product_id );
	}

	/**
	 * Function will remove member/mepr membership from access groups when remove mepr membership from admin.
	 *
	 * @param string   $new_status New post status.
	 * @param string   $old_status Old post status.
	 * @param \WP_Post $post       Post object.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_remove_mepr_memberships( $new_status, $old_status, $post ) {
		if ( 'publish' !== $new_status && 'memberpressproduct' === $post->post_type ) {
			$this->condition_item_deleted( $this->condition_name, $post->ID );
			$this->condition_remove_all_users( $this->condition_name, $post->ID );
			bb_access_recalculate_member_for_has_any_membership_group( $this->condition_name );
		}
	}

	/**
	 * Function will pass only those membership level which status has publish.
	 * It will not fetch member which membership levels are draft or trash.
	 *
	 * @param array $args Where args to pass in where query.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function mepr_list_table_args_callback( $args ) {
		global $wpdb;
		$args[] = 'm.memberships NOT IN ( ' . $wpdb->prepare( "SELECT ID FROM {$wpdb->prefix}posts AS p WHERE p.post_type=%s AND p.post_status IN ( 'draft', 'trash' )", 'memberpressproduct' ) . ' )';
		return $args;
	}

	/**
	 * Function to modify the joins.
	 *
	 * @since 2.1.40
	 *
	 * @param array $joins Joins in the query.
	 *
	 * @return array
	 */
	public function mepr_list_table_joins_callback( $joins ) {
		$mepr_db = new MeprDb();

		$position = array_search( "/* IMPORTANT */ LEFT JOIN {$mepr_db->transaction_meta} AS tr_meta ON tr.id=tr_meta.transaction_id", $joins, true );

		if ( false !== $position ) {
			unset( $joins[ $position ] );
		}

		return $joins;
	}
}
