<?php
/**
 * Holds Learndash Group Integration for BuddyBossApp InAppPurchases.
 *
 * @package BuddyBossApp\Integrations\LearndashGroup
 */

namespace BuddyBossApp\Integrations\LearndashGroup;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\InAppPurchases\Controller;
use BuddyBossApp\InAppPurchases\IntegrationAbstract;
use BuddyBossApp\InAppPurchases\Orders;

/**
 * Learndash Group Integration for BuddyBossApp InAppPurchases.
 */
final class IAP extends IntegrationAbstract {

	/**
	 * Class instance.
	 *
	 * @var null $instance
	 */
	private static $instance = null;

	/**
	 * LearnDashGroupIntegration constructor.
	 */
	private function __construct() {
		// ... leave empty, see Singleton below
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return Controller|null
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
		}

		return self::$instance;
	}

	/**
	 * Overriding the parent(from base-class) function.
	 *
	 * @param string $integration_type  Integration type.
	 * @param string $integration_label Integration label.
	 */
	public function set_up( $integration_type, $integration_label ) {
		$this->integration_slug = Controller::$learndash_group_slug;

		parent::set_up( $integration_type, $integration_label );

		$this->item_label = __( 'LearnDash Group', 'buddyboss-app' );

		// Register Instance.
		bbapp_iap()->integration[ $integration_type ] = $this::instance();
	}

	/**
	 * Below function get triggers when(hook) order is completed.
	 *
	 * @param array $item_ids Array of item ID's.
	 * @param array $order Order object.
	 *
	 * @return mixed
	 */
	public function on_order_completed( $item_ids, $order ) {
		foreach ( $item_ids as $item_identifier ) {

			$split               = explode( ':', $item_identifier );
			$group_id            = $split[0];
			$readable_item_ids[] = "<a href=\"post.php?post=$group_id&action=edit\" target='_blank'>$group_id</a>";

			// Grant the group access.
			ld_update_group_access( $order->user_id, $group_id, false );
			// update user group count.
			$this->user_update_count( $group_id, $order->user_id, 'plus' );
		}

		$readable_item_ids = implode( ', ', $readable_item_ids );

		/* translators: %s: Item ids. */
		Orders::instance()->add_history( $order->id, 'info', sprintf( __( 'User enrolled in Group(s), ID(s) are : %s', 'buddyboss-app' ), $readable_item_ids ) );

		Orders::instance()->update_meta( $order->id, '_learndash_group_ids', maybe_serialize( $item_ids ) );
	}

	/**
	 * Below function get triggers when(hook) order is activated.
	 *
	 * @param array $item_ids Array of item ID's.
	 * @param array $order Order object.
	 *
	 * @return mixed
	 */
	public function on_order_activate( $item_ids, $order ) {
		// NOTE : Similar to onOrderCompleted($order) until something needs to be changed?
		return $this->on_order_completed( $item_ids, $order );
	}

	/**
	 * Below function get triggers when(hook) order is expired.
	 *
	 * @param array $item_ids Array of item ID's.
	 * @param array $order Order object.
	 *
	 * @return mixed|void
	 */
	public function on_order_expired( $item_ids, $order ) {
		// NOTE : Similar to onOrderCancelled($order) until something needs to be changed?
		$this->on_order_cancelled( $item_ids, $order );
	}

	/**
	 * Below function get triggers when(hook) order is cancelled.
	 *
	 * @param array $item_ids Array of item ID's.
	 * @param array $order Order object.
	 *
	 * @return mixed|void
	 */
	public function on_order_cancelled( $item_ids, $order ) {

		foreach ( $item_ids as $item_identifier ) {
			$split               = explode( ':', $item_identifier );
			$group_id            = $split[0];
			$readable_item_ids[] = "<a href=\"post.php?post=$group_id&action=edit\" target='_blank'>$group_id</a>";

			// Revoke the group access.
			ld_update_group_access( $order->user_id, $group_id, true );
			// update user group count.
			$this->user_update_count( $group_id, $order->user_id, 'minus' );

		}

		$readable_item_ids = implode( ', ', $readable_item_ids );

		/* translators: %s: Item id. */
		Orders::instance()->add_history( $order->id, 'info', sprintf( __( 'User un-enrolled in group(s), ID(s) are : %s ', 'buddyboss-app' ), $readable_item_ids ) );
	}

	/**
	 * Helper function to update users courses counts.
	 *
	 * @param int    $group_id Group id.
	 * @param int    $user_id  User id.
	 * @param string $action   Action.
	 */
	public function user_update_count( $group_id, $user_id, $action = 'plus' ) {
		$groups = get_user_meta( $user_id, '_learndash_inapp_purchase_enrolled_group_access_counter', true );

		if ( ! empty( $groups ) ) {
			$groups = maybe_unserialize( $groups );
		} else {
			$groups = array();
		}

		if ( isset( $groups[ $group_id ] ) ) {
			if ( 'plus' === $action ) {
				$groups[ $group_id ] += 1;
			} else {
				$groups[ $group_id ] -= 1;
			}
		} else {
			$groups[ $group_id ] = ( 'plus' === $action ) ? 1 : 0;
		}

		update_user_meta( $user_id, '_learndash_inapp_purchase_enrolled_group_access_counter', $groups );
	}

	/**
	 * Handle bbapp ajax iap_linking_options for this integration.
	 *
	 * @param array $results levels data.
	 *
	 * @return array|mixed
	 */
	public function iap_linking_options( $results ) {
		return \BuddyBossApp\Admin\InAppPurchases\Helpers::get_integration_items( 'groups' );
	}

	/**
	 * Handle bbapp ajax iap_integration_ids for this integration.
	 * it's return items label with id.
	 *
	 * @param array $results levels data.
	 * @param array $integration_ids selected integration id.
	 *
	 * @return array|mixed
	 */
	public function iap_integration_ids( $results, $integration_ids ) {
		return $results;
	}

	/**
	 * Get item edit link to show on order page.
	 *
	 * @param string $link URL.
	 * @param int    $item_id Item id.
	 *
	 * @return string
	 */
	public function item_id_permalink( $link, $item_id ) {
		return "post.php?post=$item_id&action=edit";
	}

	/**
	 * Check given post has given membership configured or not.
	 *
	 * @param bool $is_available Flag for product is available or not.
	 * @param int  $item_id Post id or membership id. it's directly or indirectly [ through integration's item id ] connected with bbapp product.
	 * @param int  $integration_item_id Integration item id which is connected with bbapp product.
	 *
	 * @return bool
	 */
	public function is_purchase_available( $is_available, $item_id, $integration_item_id ) {
		return learndash_group_has_course( $integration_item_id, $item_id );
	}

	/**
	 * Check given integration item has access.
	 *
	 * @param array $item_ids Array of item ID's.
	 * @param int   $user_id User ID.
	 *
	 * @return false
	 */
	public function has_access( $item_ids, $user_id ) {
		if ( is_admin() ) {
			return true;
		}

		$has_access = false;

		foreach ( $item_ids as $item_identifier ) {
			$split    = explode( ':', $item_identifier );
			$group_id = $split[0];
			if ( learndash_is_user_in_group( $user_id, $group_id ) ) {
				$has_access = true;
				break;
			}
		}

		return $has_access;
	}
}
