<?php
/**
 * Holds Learndash coruse related functionality for BuddyBossApp InAppPurchases.
 *
 * @package BuddyBossApp\Integrations\LearndashCourse
 */

namespace BuddyBossApp\Integrations\LearndashCourse;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\Admin\InAppPurchases\ProductHelper;
use BuddyBossApp\InAppPurchases\Controller;
use BuddyBossApp\InAppPurchases\IntegrationAbstract;
use BuddyBossApp\InAppPurchases\Orders;
use BuddyBossApp\Integrations\Learndash\Main;

/**
 * Learndash Integration for BuddyBossApp InAppPurchases.
 */
final class IAP extends IntegrationAbstract {

	/**
	 * Class instance.
	 *
	 * @var null $instance
	 */
	private static $instance = null;

	/**
	 * LearnDashCourseIntegration constructor.
	 */
	private function __construct() {
		// ... leave empty, see Singleton below
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return Controller|null
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Define all actions and filters here.
	 *
	 * @since BuddyBoss App 1.3.7
	 */
	public function hooks() {
		// Learndash integrated courses.
		add_filter( 'bbapp_iap_integrated_products', array( $this, 'bbapp_get_integrated_courses' ), 10, 3 );
	}

	/**
	 * Function to merge the iap product with course access.
	 *
	 * @since BuddyBoss App 1.3.7
	 *
	 * @param array  $integrated_products Integrated IAP products.
	 * @param int    $item_id             course id.
	 * @param string $integration_type    integration type.
	 *
	 * @return array
	 */
	public function bbapp_get_integrated_courses( $integrated_products, $item_id, $integration_type ) {
		global $wpdb;

		$global_prefix         = \bbapp_iap()->get_global_dbprefix();
		$iap_pluck_ids         = wp_list_pluck( $integrated_products, 'product_id' );
		$course_access_courses = Main::instance()->bbapp_get_course_cats_and_tags( $item_id );

		if ( empty( $course_access_courses ) ) {
			return $integrated_products;
		}

		$pluck_courses          = ! empty( $course_access_courses ) ? wp_list_pluck( $course_access_courses, 'id' ) : array();
		$merge_iaps             = array_unique( array_merge( $iap_pluck_ids, $pluck_courses ) );
		$diff_iaps              = array_diff( $merge_iaps, $iap_pluck_ids );
		$imploded_courses       = "'" . implode( "','", $diff_iaps ) . "'";
		$iaps                   = $wpdb->get_results( "SELECT * FROM {$global_prefix}bbapp_iap_products WHERE status = 'published' AND id IN ({$imploded_courses})" ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$course_access_products = array();

		if ( ! empty( $iaps ) ) {
			foreach ( $iaps as $iap ) {
				$misc_settings    = maybe_unserialize( $iap->misc_settings );
				$integration_data = maybe_unserialize( $iap->integration_data );

				if ( bbapp()->is_network_activated() ) {
					$misc_settings    = $misc_settings[ get_current_blog_id() ];
					$integration_data = $integration_data[ get_current_blog_id() ];
				}

				$integration_slug = $misc_settings['integration_type'];
				$integration_data = isset( $integration_data[ $integration_slug ] ) ? $integration_data[ $integration_slug ] : array();

				foreach ( $integration_data as $value ) {
					$store_data          = maybe_unserialize( $iap->store_data );
					$misc_settings       = \BuddyBossApp\Admin\InAppPurchases\Helpers::instance()->bbapp_iap_product_mics_setting( $iap );
					$integration_data    = \BuddyBossApp\Admin\InAppPurchases\Helpers::instance()->bbapp_iap_product_integration_data( $iap );
					$integration_type    = ( isset( $misc_settings['integration_type'] ) ? $misc_settings['integration_type'] : '' );
					$integrated_item_ids = \BuddyBossApp\Admin\InAppPurchases\Helpers::instance()->bbapp_iap_product_integration_ids( $integration_type, $integration_data );
					$ios                 = \BuddyBossApp\Admin\InAppPurchases\Helpers::instance()->bbapp_iap_ios_product_info( $store_data );
					$android             = \BuddyBossApp\Admin\InAppPurchases\Helpers::instance()->bbapp_iap_android_product_info( $store_data );
					$bbapp_product_type  = isset( $store_data['bbapp_product_type'] ) ? $store_data['bbapp_product_type'] : 'free';

					// Check Product is configure properly or not. If not it should not return in api response.
					if ( ( isset( $store_data['device_platforms'] ) && in_array( 'ios', $store_data['device_platforms'], true ) && ( empty( $ios['status'] ) || ( 'free' !== $bbapp_product_type && empty( $ios['store_product_id'] ) ) ) ) || ( isset( $store_data['device_platforms'] ) && in_array( 'android', $store_data['device_platforms'], true ) && ( empty( $android['status'] ) || ( 'free' !== $bbapp_product_type && empty( $android['store_product_id'] ) ) ) ) || empty( $integrated_item_ids ) ) {
						continue;
					}

					$has_access           = false;
					$group_active_product = 0;

					if ( is_user_logged_in() ) {
						// NOTE : Get user_id and check into bbapp_orders if this bbapp.
						$has_access_order = ProductHelper::has_active_order( $iap, get_current_user_id() );

						// Check Any order exist for same group product.
						$group_active_product = 0;

						if ( ! empty( $iap->iap_group ) ) {
							$group_active_product = ProductHelper::get_group_active_order_product_id( $iap->iap_group, get_current_user_id() );
						}
					}

					$course_access_products[] = array(
						'product_id'           => (int) $iap->id,
						'product_name'         => $iap->name,
						'product_tagline'      => $iap->tagline,
						'product_desc'         => $iap->description,
						'benefits'             => $misc_settings['benefits'],
						'global_subscription'  => $misc_settings['global_subscription'] ? true : false,
						'bbapp_product_type'   => $bbapp_product_type,
						'ios'                  => $ios,
						'android'              => $android,
						'has_access'           => $has_access_order,
						'group_active_product' => $group_active_product,
						'sort_order'           => (int) $iap->menu_order,
					);
				}
			}
		}

		return array_merge( $course_access_products, $integrated_products );
	}

	/**
	 * Overriding the parent(from base-class) function.
	 *
	 * @param string $integration_type  Integration type.
	 * @param string $integration_label Integration label.
	 */
	public function set_up( $integration_type, $integration_label ) {
		$this->integration_slug = Controller::$learndash_course_slug;

		parent::set_up( $integration_type, $integration_label );

		$this->item_label = __( 'Course', 'buddyboss-app' );

		// Register Instance.
		bbapp_iap()->integration[ $integration_type ] = $this::instance();
	}

	/**
	 * Below function get triggers when(hook) order is completed.
	 *
	 * @param array $item_ids Array of item ID's.
	 * @param array $order Order object.
	 *
	 * @return mixed
	 */
	public function on_order_completed( $item_ids, $order ) {
		foreach ( $item_ids as $item_identifier ) {
			$split               = explode( ':', $item_identifier );
			$id                  = $split[0];
			$readable_item_ids[] = "<a href=\"post.php?post=$id&action=edit\" target='_blank'>$id</a>";

			// Grant the course access.
			ld_update_course_access( $order->user_id, $id, false );
			// update user course count.
			$this->user_update_count( $id, $order->user_id, 'plus' );

		}

		$readable_item_ids = implode( ', ', $readable_item_ids );

		/* translators: %s: Item ids. */
		Orders::instance()->add_history( $order->id, 'info', sprintf( __( 'User enrolled in course(s), ID(s) are : %s', 'buddyboss-app' ), $readable_item_ids ) );

		Orders::instance()->update_meta( $order->id, '_learndash_course_ids', maybe_serialize( $item_ids ) );
	}

	/**
	 * Below function get triggers when(hook) order is activated.
	 *
	 * @param array $item_ids Array of item ID's.
	 * @param array $order Order object.
	 *
	 * @return mixed
	 */
	public function on_order_activate( $item_ids, $order ) {
		// NOTE : Similar to onOrderCompleted($order) until something needs to be changed?
		return $this->on_order_completed( $item_ids, $order );
	}

	/**
	 * Below function get triggers when(hook) order is expired.
	 *
	 * @param array $item_ids Array of item ID's.
	 * @param array $order Order object.
	 *
	 * @return mixed|void
	 */
	public function on_order_expired( $item_ids, $order ) {
		// NOTE : Similar to onOrderCancelled($order) until something needs to be changed?
		$this->on_order_cancelled( $item_ids, $order );
	}

	/**
	 * Below function get triggers when(hook) order is cancelled.
	 *
	 * @param array $item_ids Array of item ID's.
	 * @param array $order Order object.
	 *
	 * @return mixed|void
	 */
	public function on_order_cancelled( $item_ids, $order ) {

		foreach ( $item_ids as $item_identifier ) {
			$split               = explode( ':', $item_identifier );
			$id                  = $split[0];
			$readable_item_ids[] = "<a href=\"post.php?post=$id&action=edit\" target='_blank'>$id</a>";

			// Revoke the course access.
			ld_update_course_access( $order->user_id, $id, true );
			// update user course count.
			$this->user_update_count( $id, $order->user_id, 'minus' );

		}

		$readable_item_ids = implode( ', ', $readable_item_ids );

		/* translators: %s: Item id. */
		Orders::instance()->add_history( $order->id, 'info', sprintf( __( 'User un-enrolled in course(s), ID(s) are : %s ', 'buddyboss-app' ), $readable_item_ids ) );
	}

	/**
	 * Helper function to update users courses counts.
	 *
	 * @param int    $course_id Course id.
	 * @param int    $user_id   User id.
	 * @param string $action    Action.
	 */
	public function user_update_count( $course_id, $user_id, $action = 'plus' ) {
		$courses = get_user_meta( $user_id, '_learndash_inapp_purchase_enrolled_courses_access_counter', true );

		if ( ! empty( $courses ) ) {
			$courses = maybe_unserialize( $courses );
		} else {
			$courses = array();
		}

		if ( isset( $courses[ $course_id ] ) ) {
			if ( 'plus' === $action ) {
				$courses[ $course_id ] += 1;
			} else {
				$courses[ $course_id ] -= 1;
			}
		} else {
			$courses[ $course_id ] = ( 'plus' === $action ) ? 1 : 0;
		}

		update_user_meta( $user_id, '_learndash_inapp_purchase_enrolled_courses_access_counter', $courses );
	}

	/**
	 * Handle bbapp ajax iap_linking_options for this integration.
	 *
	 * @param array $results levels data.
	 *
	 * @return array|mixed
	 */
	public function iap_linking_options( $results ) {
		return \BuddyBossApp\Admin\InAppPurchases\Helpers::get_integration_items( 'sfwd-courses' );
	}

	/**
	 * Handle bbapp ajax iap_integration_ids for this integration.
	 * it's return items label with id.
	 *
	 * @param array $results levels data.
	 * @param array $integration_ids selected integration id.
	 *
	 * @return array|mixed
	 */
	public function iap_integration_ids( $results, $integration_ids ) {
		return $results;
	}

	/**
	 * Get item edit link to show on order page.
	 *
	 * @param string $link URL.
	 * @param int    $item_id Item id.
	 *
	 * @return string
	 */
	public function item_id_permalink( $link, $item_id ) {
		return "post.php?post=$item_id&action=edit";
	}

	/**
	 * Check given post has given membership configured or not.
	 *
	 * @param bool $is_available Flag for product is available or not.
	 * @param int  $item_id Post id or membership id. it's directly or indirectly [ through integration's item id ] connected with bbapp product.
	 * @param int  $integration_item_id Integration item id which is connected with bbapp product.
	 *
	 * @return bool
	 */
	public function is_purchase_available( $is_available, $item_id, $integration_item_id ) {
		return $item_id === $integration_item_id;
	}

	/**
	 * Check given integration item has access.
	 *
	 * @param array $item_ids Array of item ID's.
	 * @param int   $user_id User ID.
	 *
	 * @return false
	 */
	public function has_access( $item_ids, $user_id ) {
		if ( is_admin() ) {
			return true;
		}

		$has_access = false;

		foreach ( $item_ids as $item_identifier ) {
			$split    = explode( ':', $item_identifier );
			$level_id = $split[0];
			if ( sfwd_lms_has_access( $level_id, $user_id ) ) {
				$has_access = true;
				break;
			}
		}

		return $has_access;
	}
}
