<?php
/**
 * Holds Learndash related features/hooks/filters.
 *
 * @package BuddyBossApp\Integrations\Learndash
 */

namespace BuddyBossApp\Integrations\Learndash;

use BuddyBossApp\Admin\InAppPurchases\Helpers;
use BuddyBossApp\Api\LearnDash\V1\Course\CoursesRest;
use BuddyBossApp\App\App;
use BuddyBossApp\Helpers\PostCover;

/**
 * Learndash related features/hooks/filters.
 */
class Main {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Main constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return mixed
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Load the class required code.
	 */
	public function load() {

		add_action( 'init', array( $this, 'learndash_init' ) );
	}

	/**
	 * Load the integrations.
	 */
	public function learndash_init() {
		if ( ! bbapp_is_learndash_enabled() ) {
			return;
		}

		add_filter( 'bbapp_quick_link_user_links_items', array( $this, 'learndash_bbapp_quick_link_user_links_items' ), 10, 1 );
		add_filter( 'bbapp_quick_link_deeplink_links', array( $this, 'learndash_bbapp_quick_link_deeplink_links' ), 10, 2 );
		add_filter( 'bbapp_user_segment_integrations', array( $this, 'load_segment' ), 11 );
		$this->local_features();

		// Add Course cover image if BuddyBoss theme is not activated.
		if ( ! function_exists( 'buddyboss_theme' ) && class_exists( 'BuddyBossApp\Helpers\PostCover' ) ) {
			new PostCover(
				array(
					'label'     => __( 'Cover Photo', 'buddyboss-app' ),
					'id'        => 'course-cover-image',
					'post_type' => 'sfwd-courses',
				)
			);
		}
		add_filter( 'bbapp_core_profile_tabs', array( $this, 'bbapp_core_profile_learndash_tabs' ), 10, 1 );

		add_filter( 'bbapp_is_product_skip', array( $this, 'bbapp_is_course_product_skip' ), 10, 2 );
		add_filter( 'learndash-cpt-options', array( $this, 'add_assignment_comment_option' ), 10, 2 );

		add_filter( 'bbapp_block_filter_options', array( $this, 'add_filter_options' ), 10, 2 );
		add_filter( 'bbapp_block_filter_list', array( $this, 'add_filter_list' ), 10, 2 );
		add_filter( 'bbapp_block_query_args', array( $this, 'add_query_args' ), 10, 2 );
	}

	/**
	 * Add location features related to social features.
	 */
	public function local_features() {
		$is_enabled = bbapp_is_learndash_enabled();

		// Learndash Support.
		App::instance()->add_local_feature(
			'learndash',
			array(
				'is_enabled_android' => $is_enabled,
				'is_enabled_ios'     => $is_enabled,
			)
		);
	}

	/**
	 * Load Learndash related user segments
	 *
	 * @param array $integrations List of integration.
	 *
	 * @return mixed
	 */
	public function load_segment( $integrations ) {
		$integrations['learndash'] = new Segment();

		return $integrations;
	}

	/**
	 * Quick link widget user link.
	 *
	 * @param array $quick_link_user_links_items Array of courses.
	 *
	 * @return mixed
	 */
	public function learndash_bbapp_quick_link_user_links_items( $quick_link_user_links_items ) {
		if ( function_exists( 'bp_ld_sync' ) && bp_ld_sync()->settings->get( 'course.courses_visibility' ) ) {
			$slug         = apply_filters( 'bp_learndash_profile_courses_slug', \LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Section_Permalinks', 'courses' ) );
			$name         = \LearnDash_Custom_Label::get_label( 'courses' );
			$item_courses = array(
				'ID'    => $slug,
				'title' => $name,
				'icon'  => array(
					'id'   => 'course',
					'type' => 'buddyboss',
				),
			);

			$quick_link_user_links_items[] = $item_courses;
		}

		/**
		 * Course certificate support added.
		 */
		if ( function_exists( 'bp_core_learndash_certificates_enables' ) && bp_core_learndash_certificates_enables() ) {

			$item_certificates = array(
				'ID'    => apply_filters( 'bp_learndash_profile_certificates_slug', 'certificates' ),
				'title' => apply_filters( 'bp_learndash_profile_certificates_tab_name', __( 'Certificates', 'buddyboss-app' ) ),
				'icon'  => array(
					'id'   => 'certificate',
					'type' => 'buddyboss',
				),
			);

			$quick_link_user_links_items[] = $item_certificates;

		}

		return $quick_link_user_links_items;
	}

	/**
	 * Quick link deeplink url.
	 *
	 * @param string $link URL.
	 * @param string $link_slug URL slug.
	 *
	 * @return string
	 */
	public function learndash_bbapp_quick_link_deeplink_links( $link, $link_slug ) {
		if ( function_exists( 'bp_ld_sync' ) && bp_ld_sync()->settings->get( 'course.courses_visibility' ) ) {
			$slug = apply_filters( 'bp_learndash_profile_courses_slug', \LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Section_Permalinks', 'courses' ) );
			if ( $slug === $link_slug ) {
				return trailingslashit( bp_loggedin_user_domain() . $slug );
			}
		}

		/**
		 * Course certificate support added.
		 */
		if ( function_exists( 'bp_core_learndash_certificates_enables' ) && bp_core_learndash_certificates_enables() ) {
			$course_slug = apply_filters( 'bp_learndash_profile_courses_slug', \LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Section_Permalinks', 'courses' ) );
			if ( apply_filters( 'bp_learndash_profile_certificates_slug', 'certificates' ) === $link_slug ) {
				if ( ! empty( $course_slug ) ) {
					return trailingslashit( bp_loggedin_user_domain() . $course_slug . '/' . $link_slug );
				} else {
					return trailingslashit( bp_loggedin_user_domain() . $link_slug );
				}
			}
		}

		return $link;
	}

	/**
	 * Get learndash slugs.
	 *
	 * @param array $nav_items Menus array.
	 *
	 * @return mixed
	 */
	public function bbapp_core_profile_learndash_tabs( $nav_items ) {
		$course_name          = \LearnDash_Custom_Label::get_label( 'courses' );
		$course_slug          = bb_learndash_profile_courses_slug();
		$my_course_name       = sprintf( '%s %s', __( 'My', 'buddyboss-app' ), $course_name );
		$my_course_slug       = apply_filters( 'bp_learndash_profile_courses_slug', 'my-courses' );
		$my_certificates_name = sprintf( '%s %s', __( 'My', 'buddyboss-app' ), $course_name );
		$my_certificates_slug = apply_filters( 'bp_learndash_profile_certificates_slug', 'certificates' );

		$nav_items[ $course_slug ]          = $course_name;
		$nav_items[ $my_course_slug ]       = $my_course_name;
		$nav_items[ $my_certificates_slug ] = $my_certificates_name;

		return $nav_items;
	}

	/**
	 * Function to get the course categories and tags.
	 *
	 * @since 1.4.0
	 *
	 * @param int $item_id Categorirs ID.
	 *
	 * @return mixed
	 */
	public function bbapp_get_course_cats_and_tags( $item_id = 0 ) {

		if ( empty( $item_id ) ) {
			return array();
		}

		$course_cats = array();
		$course_tags = array();

		if ( method_exists( 'LearnDash_Settings_Section', 'get_section_setting' ) && 'yes' === \LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Taxonomies', 'ld_course_category' ) ) {
			$course_cats = wp_get_post_terms( $item_id, 'ld_course_category', array( 'fields' => 'ids' ) );
		}

		if ( method_exists( 'LearnDash_Settings_Section', 'get_section_setting' ) && 'yes' === \LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Taxonomies', 'ld_course_tag' ) ) {
			$course_tags = wp_get_post_terms( $item_id, 'ld_course_tag', array( 'fields' => 'ids' ) );
		}

		$iap_ids = $this->bbapp_iap_get_courses_iap_products( array( $item_id ), $course_cats, $course_tags );

		return $iap_ids;
	}

	/**
	 * Function to get the LD course IAP.
	 *
	 * @since 1.4.0
	 *
	 * @param array $course_ids     specific course ids.
	 * @param array $course_cat_ids specific course category ids.
	 * @param array $course_tag_ids specific course tag ids.
	 *
	 * @return array|object
	 */
	public function bbapp_iap_get_courses_iap_products( $course_ids = array(), $course_cat_ids = array(), $course_tag_ids = array() ) {
		global $wpdb;

		$select = "SELECT DISTINCT {$wpdb->prefix}bbapp_iap_products.id FROM {$wpdb->prefix}bbapp_iap_products INNER JOIN {$wpdb->prefix}bbapp_iap_productmeta iapm1 ON ({$wpdb->prefix}bbapp_iap_products.id = iapm1.iap_id ) INNER JOIN {$wpdb->prefix}bbapp_iap_productmeta iapm2 ON ({$wpdb->prefix}bbapp_iap_products.id = iapm2.iap_id)";

		$where = array();

		$where['all_courses'] = "(iapm1.meta_key='course_access_type' AND iapm1.meta_value='all-courses')";

		if ( ! empty( $course_ids ) ) {
			$imploded_course_ids       = "'" . implode( "','", array_map( 'intval', $course_ids ) ) . "'";
			$where['specific_courses'] = "(iapm1.meta_key='course_access_type' AND iapm1.meta_value='specific-course' AND iapm2.meta_key='course_access_id' AND iapm2.meta_value IN ({$imploded_course_ids}))";
		}

		if ( ! empty( $course_cat_ids ) ) {
			$imploded_cat_ids       = "'" . implode( "','", array_map( 'intval', $course_cat_ids ) ) . "'";
			$where['specific_cats'] = "(iapm1.meta_key='course_access_type' AND iapm1.meta_value='specific-category-courses' AND iapm2.meta_key='course_access_cat_id' AND iapm2.meta_value IN ({$imploded_cat_ids}))";
		}

		if ( ! empty( $course_tag_ids ) ) {
			$imploded_tag_ids       = "'" . implode( "','", array_map( 'intval', $course_tag_ids ) ) . "'";
			$where['specific_tags'] = "(iapm1.meta_key='course_access_type' AND iapm1.meta_value='specific-tag-courses' AND iapm2.meta_key='course_access_tag_id' AND iapm2.meta_value IN ({$imploded_tag_ids}))";
		}

		$imploded_where = '(' . implode( ' OR ', $where ) . ')';
		$published_iap  = "{$wpdb->prefix}bbapp_iap_products.status='published'";

		$sql = $select . ' WHERE ' . $published_iap . ' AND ' . $imploded_where;

		$results = $wpdb->get_results( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared

		return ! empty( $results ) ? $results : array();
	}

	/**
	 * Function to add logic for the course product skip.
	 *
	 * @param bool   $flag        skip flag.
	 * @param object $product_obj product object.
	 * @since 1.8.50
	 *
	 * @return boolean
	 */
	public function bbapp_is_course_product_skip( $flag, $product_obj ) {

		$misc_settings    = Helpers::bbapp_iap_product_mics_setting( $product_obj );
		$integration_data = Helpers::bbapp_iap_product_integration_data( $product_obj );
		$integration_type = isset( $misc_settings['integration_type'] ) ? $misc_settings['integration_type'] : '';

		if ( ! empty( $integration_type ) && 'learndash-course' === $integration_type ) {
			$integrated_item_ids = Helpers::bbapp_iap_product_integration_ids( $integration_type, $integration_data );
			$course_status       = bbapp_ld_get_course_status( $integrated_item_ids[0] );

			if ( isset( $course_status['is_seats_available'] ) && false === $course_status['is_seats_available'] ) {
				$flag = true;
			}
		}

		return $flag;
	}

	/**
	 * Add assignment comment option.
	 *
	 * @param array  $args      Arguments.
	 * @param string $post_type Post type.
	 *
	 * @since 2.1.70
	 * @return array
	 */
	public function add_assignment_comment_option( $args, $post_type ) {
		$request_uri = ( ! empty( $_SERVER['REQUEST_URI'] ) ) ? bbapp_input_clean( sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) ) : '';

		if ( 'sfwd-assignment' === $post_type && is_rest() && false !== strpos( $request_uri, 'buddyboss-app/learndash/v1/assignments' ) ) {
			$args['show_in_rest'] = true;
		}

		return $args;
	}

	/**
	 * Add filter options.
	 *
	 * @param array  $filter_options Filter options.
	 * @param string $post_type      Post type.
	 *
	 * @since 2.3.40
	 * @return array
	 */
	public function add_filter_options( $filter_options, $post_type ) {
		if ( 'sfwd-courses' === $post_type ) {
			$filter_options[] = array(
				'name'  => 'my_courses',
				'label' => __( 'My Courses', 'buddyboss-app' ),
			);
		}

		return $filter_options;
	}

	/**
	 * Add filter list.
	 *
	 * @param array  $filter_list Filter list.
	 * @param string $post_type   Post type.
	 *
	 * @since 2.3.40
	 * @return array
	 */
	public function add_filter_list( $filter_list, $post_type ) {
		if ( 'sfwd-courses' === $post_type ) {
			$my_courses_options = array(
				array(
					'label' => __( 'Yes', 'buddyboss-app' ),
					'key'   => true
				),
				array(
					'label' => __( 'No', 'buddyboss-app' ),
					'key'   => false
				)
			);

			$filter_list['my_courses'] = $my_courses_options;
		}

		return $filter_list;
	}

	/**
	 * Add query args.
	 *
	 * @param array  $query_args Query args.
	 * @param string $request  Request.
	 *
	 * @since 2.3.40
	 * @return array
	 */
	public function add_query_args( $query_args, $request ) {
		if ( 'sfwd-courses' === $query_args['post_type'] ) {
			if ( isset( $query_args['my_courses'] ) && $query_args['my_courses'] ) {
				$user_id      = get_current_user_id();
				$mycourse_ids = ld_get_mycourses( $user_id, array() );

				if ( ! empty( $mycourse_ids ) && ! is_wp_error( $mycourse_ids ) ) {
					$query_args['post__in'] = ! empty( $query_args['post__in'] ) ? array_intersect( $mycourse_ids, $query_args['post__in'] ) : $mycourse_ids;
				}

				/*
				 * If we intersected, but there are no post ids in common,
				 * WP_Query won't return "no posts" for post__in = array()
				 * so we have to fake it a bit.
				*/
				if ( ! $query_args['post__in'] ) {
					$query_args['post__in'] = array( 0 );
				}

				unset( $query_args['my_courses'] );
			}
		}

		return $query_args;
	}
}
