<?php
/**
 * Holds deepliking functionality related LearnDash.
 *
 * @package BuddyBossApp\Integrations\Learndash
 */

namespace BuddyBossApp\Integrations\Learndash;

use BuddyBossApp\DeepLinking\Type\TypeAbstract;
use WP_Post;

/**
 * Deeplink class for LearnDash.
 */
class DeepLinking extends TypeAbstract {

	/**
	 * Post types array.
	 *
	 * @var array
	 */
	public $arr_post_types = array();

	/**
	 * Array post type mapping.
	 *
	 * @var array
	 */
	public $arr_post_types_mapping = array();

	/**
	 * Taxonomy array.
	 *
	 * @var array
	 */
	public $arr_taxonomy = array();

	/**
	 * Construct function.
	 */
	public function __construct() {
		parent::__construct();
		$this->init();
		add_filter( 'bbapp_deeplinking_cpt_action', array( $this, 'update_cpt_action' ), 10, 2 );
		add_filter( 'bbapp_deeplinking_cpt_namespace', array( $this, 'update_cpt_namespace' ), 10, 2 );
		add_filter( 'bbapp_deeplinking_term_namespace', array( $this, 'update_term_namespace' ), 10, 2 );
		add_filter( 'bbapp_deeplinking_taxonomy_namespace', array( $this, 'update_taxonomy_namespace' ), 10, 2 );
		add_filter( 'bbapp_deeplinking_cpt_type', array( $this, 'update_cpt_object' ), 10, 3 );
	}

	/**
	 * Init all LearnDash maps.
	 */
	public function init() {
		$this->arr_post_types_mapping = array(
			'sfwd-courses'      => 'sfwd-courses',
			'sfwd-lessons'      => 'sfwd-lessons',
			'sfwd-topic'        => 'sfwd-topic',
			'sfwd-quiz'         => 'sfwd-quiz',
			'sfwd-question'     => 'question',
			'sfwd-certificates' => 'certificate',
			'sfwd-essays'       => 'essay',
			'sfwd-assignment'   => 'assignment',
			'groups'            => 'group',
		);
		$this->arr_post_types         = array_keys( $this->arr_post_types_mapping );
		$this->arr_taxonomy           = array(
			'ld_course_category',
			'ld_course_tag',
			'ld_lesson_category',
			'ld_lesson_tag',
			'ld_topic_category',
			'ld_topic_tag',
			'ld_quiz_category',
			'ld_quiz_tag',
			'ld_question_category',
			'ld_question_tag',
		);
	}

	/**
	 * Add item embed with response
	 *
	 * @param array   $response Response.
	 * @param WP_Post $post     Post type object.
	 * @param string  $url URL.
	 *
	 * @since 1.8.30 Added the `$url ` parameter.
	 * @return mixed
	 */
	public function update_cpt_object( $response, $post, $url ) {

		if ( 'sfwd-courses' === $post->post_type ) {
			$response['_link_action'] = 'ld_course';
		}

		if ( 'sfwd-lessons' === $post->post_type ) {
			$response['_link_action'] = 'ld_lesson';
		}

		if ( 'sfwd-topic' === $post->post_type ) {
			$response['_link_action'] = 'ld_topic';
		}

		if ( 'sfwd-quiz' === $post->post_type ) {
			$response['_link_action'] = 'ld_quiz';
		}

		if ( 'groups' === $post->post_type ) {
			$response['_link_action'] = 'ld_group';
		}

		return $response;
	}

	/**
	 * Update LearnDash's CPT action
	 *
	 * @param string  $action Action.
	 * @param WP_Post $post   Post object.
	 *
	 * @return string
	 */
	public function update_cpt_action( $action, $post ) {
		if ( $post instanceof WP_Post && in_array( $post->post_type, $this->arr_post_types, true ) ) {
			return 'open_' . $this->arr_post_types_mapping[ $post->post_type ];
		}

		return $action;
	}

	/**
	 * Update LearnDash's cpts namespace
	 *
	 * @param object $cpt_namespace Namespace obeject.
	 * @param string $post_type Post type.
	 *
	 * @return string
	 */
	public function update_cpt_namespace( $cpt_namespace, $post_type ) {
		if ( isset( $post_type ) && in_array( $post_type, $this->arr_post_types, true ) ) {
			return 'bbapp_lms';
		}

		return $cpt_namespace;
	}

	/**
	 * Update LearnDash's terms namespace
	 *
	 * @param object $term_namespace Namespace object.
	 * @param object $term      Term object.
	 *
	 * @return string
	 */
	public function update_term_namespace( $term_namespace, $term ) {
		if ( $term instanceof \WP_Term && in_array( $term->taxonomy, $this->arr_taxonomy, true ) ) {
			return 'bbapp_lms';
		}

		return $term_namespace;
	}

	/**
	 * Update LearnDash's taxonomy namespace.
	 *
	 * @param object $taxonomy_namespace Name space object.
	 * @param object $taxonomy  Taxonomy object.
	 *
	 * @return string
	 */
	public function update_taxonomy_namespace( $taxonomy_namespace, $taxonomy ) {
		if ( $taxonomy instanceof \WP_Taxonomy && in_array( $taxonomy->name, $this->arr_taxonomy, true ) ) {
			return 'bbapp_lms';
		}

		return $taxonomy_namespace;
	}

	/**
	 * Function to parse URL.
	 *
	 * @param string $url URL to parse.
	 *
	 * @return mixed|void
	 */
	public function parse( $url ) {
	}
}
