<?php
/**
 * Holds LearnDash groups functionality related Access Controls.
 *
 * @package BuddyBossApp\Integrations\Learndash\AccessControls
 */

namespace BuddyBossApp\Integrations\Learndash\AccessControls;

use BuddyBossApp\AccessControls\TmpItems;
use BuddyBossApp\Integrations\Learndash\AccessControls;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * LearnDash group class for Access Controls.
 */
class LDGroup {

	/**
	 * Condition name.
	 *
	 * @var string $condition_name condition name.
	 */
	public $condition_name = 'learndash-groups';

	/**
	 * Class instance.
	 *
	 * @var bool
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return LDGroup
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Class construct method.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Hooks here.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function load() {

	}

	/**
	 * Items callback method.
	 *
	 * @param string $search Search the condition.
	 * @param int    $page   Page number.
	 * @param int    $limit  Limit the items to be fetched.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function learndash_group_items_callback( $search = '', $page = 1, $limit = 20 ) {

		$results = new \WP_Query(
			array(
				'post_status'    => 'publish',
				'post_type'      => 'groups',
				'paged'          => $page,
				'posts_per_page' => $limit,
				's'              => $search,
			)
		);

		$list = array();

		if ( ! empty( $results->get_posts() ) ) {
			foreach ( $results->get_posts() as $item ) {
				$list[ $item->ID ] = array(
					'id'   => $item->ID,
					'name' => $item->post_title,
					'link' => admin_url( "post.php?post={$item->ID}&action=edit" ),
				);
			}
		}

		return $list;
	}

	/**
	 * Item callback method.
	 *
	 * @param int $item_value Item value of condition.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array|false
	 */
	public function learndash_group_item_callback( $item_value ) {
		$group = get_post( $item_value );

		if ( empty( $group ) || ! $group instanceof \WP_Post ) {
			return false;
		}

		return array(
			'id'   => $group->ID,
			'name' => $group->post_title,
			'link' => admin_url( "post.php?post={$group->ID}&action=edit" ),
		);
	}


	/**
	 * Users callback method.
	 *
	 * @param array $data     condition data.
	 * @param int   $page     current page number.
	 * @param int   $per_page limit.
	 *
	 * @since 1.5.2.1
	 * @return array
	 */
	public function learndash_group_users_callback( $data, $page = 1, $per_page = 10 ) {
		$user_ids      = array();
		$sub_condition = ( ! empty( $data['sub_condition'] ) ) ? $data['sub_condition'] : '';
		$item_value    = ( ! empty( $data['item_value'] ) ) ? $data['item_value'] : '';
		$group_id      = ( ! empty( $data['group_id'] ) ) ? $data['group_id'] : '';
		$rounds_count  = ( ! empty( $data['rounds_count'] ) ) ? $data['rounds_count'] : '';
		$transient_key = "access_ctrls_ld_group_condition_jb_data_{$group_id}";

		if ( 'specific' === $sub_condition ) {
			if ( empty( $item_value ) ) {
				return $user_ids;
			}

			// Check if it's a first round.
			if ( 1 === $rounds_count ) {
				$condition_data = array(
					'prepare_page'     => 1,
					'is_data_prepared' => false,
				);
			} else {
				$condition_data = get_transient( $transient_key );
			}

			/**
			 * Preparing the data into tmp items collections.
			 */
			if ( ! $condition_data['is_data_prepared'] ) {
				$ld_group_users = $this->learndash_get_all_group_users_ids( $item_value );

				if ( ! empty( $ld_group_users ) ) {
					$user_ids = ! empty( $ld_group_users ) ? AccessControls::instance()->paginate_users_list( array_unique( $ld_group_users ), $condition_data['prepare_page'], 500 ) : array();

					if ( ! empty( $user_ids ) ) {
						$items = array();

						foreach ( $user_ids as $user_id ) {
							$items[] = array(
								'item_type'         => "ld_groups_{$group_id}",
								'item_id'           => $user_id,
								'secondary_item_id' => $item_value,
							);
						}

						$insert = TmpItems::instance()->insert_items( $items );
						$condition_data['prepare_page'] ++;

						if ( empty( $insert ) ) {
							AccessControls::instance()->return_error( __( 'Error while preparing members data for LearnDash.', 'buddyboss-app' ) );
						}
					} else {
						$condition_data['is_data_prepared'] = true;
					}

					set_transient( $transient_key, $condition_data, DAY_IN_SECONDS );

					return AccessControls::instance()->return_wait( __( 'Preparing LD Members Data', 'buddyboss-app' ) );
				} else {
					return AccessControls::instance()->return_error( __( 'Error while getting users results from LearnDash function.', 'buddyboss-app' ) );
				}
			}
		} elseif ( 'any' === $sub_condition ) {

			// Check if it's a first round.
			if ( 1 === $rounds_count ) {
				$condition_data                  = array(
					'prepare_page'      => 1,
					'is_data_prepared'  => false,
					'pending_group_ids' => $this->learndash_get_all_group_ids(),
				);
				$condition_data['current_group'] = array_key_first( $condition_data['pending_group_ids'] );
			} else {
				$condition_data = get_transient( $transient_key );
			}

			/**
			 * Preparing the data into tmp items collections.
			 */
			if ( ! $condition_data['is_data_prepared'] ) {

				$ld_group_users = $this->learndash_get_all_group_users_ids( $condition_data['current_group'] );
				$user_ids       = ! empty( $ld_group_users ) ? AccessControls::instance()->paginate_users_list( array_unique( $ld_group_users ), $condition_data['prepare_page'], 500 ) : array();

				if ( ! empty( $user_ids ) ) {
					$items = array();

					foreach ( $user_ids as $user_id ) {
						$items[] = array(
							'item_type'         => "ld_groups_{$group_id}",
							'item_id'           => $user_id,
							'secondary_item_id' => $condition_data['current_group'],
						);
					}

					$insert = TmpItems::instance()->insert_items( $items );

					$condition_data['prepare_page'] ++;

					if ( empty( $insert ) ) {
						return AccessControls::instance()->return_error( __( 'Error while preparing members data for LearnDash.', 'buddyboss-app' ) );
					}
				} else { // when all users are processed.

					unset( $condition_data['pending_group_ids'][ $condition_data['current_group'] ] ); // remove from pending group id.

					if ( ! empty( $condition_data['pending_group_ids'] ) ) {
						$condition_data['current_group'] = array_key_first( $condition_data['pending_group_ids'] ); // get next group to process.
						$condition_data['prepare_page']  = 1; // reset pagination to 1.
					} else {
						// If pending groups found empty finish the job.
						$condition_data['is_data_prepared'] = true;
					}
				}

				set_transient( $transient_key, $condition_data, DAY_IN_SECONDS );

				return AccessControls::instance()->return_wait( __( 'Preparing LD Members Data', 'buddyboss-app' ) );
			}
		}

		/**
		 * Start Sending Users Data Once Tmp Collection Preparations has done.
		 * Once the Preparations for any of specific groups are done above the execution will reach here.
		 */

		$items = TmpItems::instance()->get_items(
			array(
				'include_item_types' => array( "ld_groups_{$group_id}" ),
				'per_page'           => $per_page,
				'page'               => $page,
			)
		);

		if ( ! empty( $items ) ) {
			$user_ids = array();

			foreach ( $items['results'] as $item ) {
				$user_ids[] = $item->item_id;
			}

			return AccessControls::instance()->return_users( $user_ids );
		}

		return AccessControls::instance()->return_error( __( 'Invalid sub condition.', 'buddyboss-app' ) );
	}

	/**
	 * Get all users within LearnDash Group.
	 *
	 * @param int $group_id LearnDash group id.
	 *
	 * @since 1.5.2.1
	 * @return array
	 */
	public function learndash_get_all_group_users_ids( $group_id ) {
		$ld_group_leader = learndash_get_groups_administrator_ids( $group_id );
		$ld_group_users  = learndash_get_groups_user_ids( $group_id );

		return array_merge( $ld_group_leader, $ld_group_users );
	}

	/**
	 * Return all group IDs
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed
	 */
	public function learndash_get_all_group_ids() {
		$query_args = array(
			'post_type'   => 'groups',
			'post_status' => 'publish',
			'fields'      => 'ids',
			'orderby'     => 'title',
			'order'       => 'ASC',
			'nopaging'    => true,    // Turns OFF paging logic to get ALL groups.
		);

		$query = new \WP_Query( $query_args );
		if ( $query instanceof \WP_Query ) {
			$ids    = $query->posts;
			$return = array();
			foreach ( $ids as $id ) {
				$return[ $id ] = $id;
			}

			return $return;
		}

		return array();
	}

	/**
	 * Function to check if user has any condition.
	 *
	 * @param int $user_id User id to check.
	 *
	 * @since 1.5.2.1
	 *
	 * @return bool
	 */
	public function has_any_items_callback( $user_id ) {
		$user_groups = learndash_get_users_group_ids( $user_id );

		return ( ! empty( $user_groups ) ) ? true : false;
	}

	/**
	 * This function add user to access control.
	 *
	 * @param int $user_id  User ID.
	 * @param int $group_id Group ID.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_ld_added_group_access( $user_id, $group_id ) {
		if ( empty( $user_id ) || empty( $group_id ) ) {
			return;
		}
		AccessControls::instance()->condition_add_user( $user_id, $this->condition_name, $group_id );
	}

	/**
	 * This function remove user from access control.
	 *
	 * @param int $user_id  User ID.
	 * @param int $group_id Group ID.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_ld_removed_group_access( $user_id, $group_id ) {
		if ( empty( $user_id ) || empty( $group_id ) ) {
			return;
		}
		AccessControls::instance()->condition_remove_user( $user_id, $this->condition_name, $group_id );
	}

	/**
	 * Function will remove member/course group from access groups when remove course group from admin.
	 *
	 * @param string   $new_status New post status.
	 * @param string   $old_status Old post status.
	 * @param \WP_Post $post       Post object.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_remove_ld_group_access_control_data( $new_status, $old_status, $post ) {
		if ( 'publish' !== $new_status && 'groups' === $post->post_type ) {
			AccessControls::instance()->condition_item_deleted( $this->condition_name, $post->ID );
			AccessControls::instance()->condition_remove_all_users( $this->condition_name, $post->ID );
			bb_access_recalculate_member_for_has_any_membership_group( $this->condition_name );
		}
	}
}
