<?php
/**
 * Holds LearnDash functioanality related Access Controls.
 *
 * @package BuddyBossApp\Integrations\Learndash\AccessControls
 */

namespace BuddyBossApp\Integrations\Learndash\AccessControls;

use BuddyBossApp\AccessControls\TmpItems;
use BuddyBossApp\Integrations\Learndash\AccessControls;
use BuddyBossApp\Jobs;

/**
 * LearnDash Access Controls class.
 */
class LDCourse {

	/**
	 * Conditiona name.
	 *
	 * @var string $condition_name condition name.
	 */
	public $condition_name = 'learndash-course';

	/**
	 * Class instance.
	 *
	 * @var bool
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return LDCourse
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Construct function.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function __construct() {
	}

	/**
	 * Hooks can be added here.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function load() {
	}

	/**
	 * Load screens settings classes.
	 *
	 * @since 1.5.2.1
	 */
	public function load_screens_settings() {
		AccessControls::instance()->register_settings_screen(
			array(
				'id'              => Course::instance()->screen_name,
				'menu_title'      => __( 'Courses', 'buddyboss-app' ),
				'page_title'      => __( 'Courses', 'buddyboss-app' ),
				'render_callback' => array( Course::instance(), 'render_callback' ),
				'options'         => array(
					'tutorial_link' => admin_url( 'admin.php?page=bbapp-help&article=125675' ),
					'sub_tab'       => array(
						'courses'          => __( 'Courses', 'buddyboss-app' ),
						'cat_tag'          => __( 'Categories & Tags', 'buddyboss-app' ),
						'default_settings' => __( 'Default Settings', 'buddyboss-app' ),
					),
				),
			)
		);
	}

	/**
	 * Items callback method.
	 *
	 * @param string $search Search the condition.
	 * @param int    $page   Page number.
	 * @param int    $limit  Limit the items to be fetched.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function learndash_course_items_callback( $search = '', $page = 1, $limit = 20 ) {
		$results = new \WP_Query(
			array(
				'post_status'    => 'publish',
				'post_type'      => 'sfwd-courses',
				'paged'          => $page,
				'posts_per_page' => $limit,
				's'              => $search,
			)
		);

		$list = array();

		if ( ! empty( $results->get_posts() ) ) {
			foreach ( $results->get_posts() as $item ) {
				$list[ $item->ID ] = array(
					'id'   => $item->ID,
					'name' => $item->post_title,
					'link' => admin_url( "post.php?post={$item->ID}&action=edit" ),
				);
			}
		}

		return $list;
	}

	/**
	 * Item callback method.
	 *
	 * @param int $item_value Item value of condition.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array|false
	 */
	public function learndash_course_item_callback( $item_value ) {
		$course = get_post( $item_value );

		if ( empty( $course ) || ! $course instanceof \WP_Post || 'trash' === $course->post_status ) {
			return false;
		}

		return array(
			'id'   => $course->ID,
			'name' => $course->post_title,
			'link' => admin_url( "post.php?post={$course->ID}&action=edit" ),
		);
	}

	/**
	 * Users callback method.
	 *
	 * @param array $data     condition data.
	 * @param int   $page     current page number.
	 * @param int   $per_page limit.
	 *
	 * @since 1.5.2.1
	 * @return array|string
	 */
	public function learndash_course_users_callback( $data, $page = 1, $per_page = 10 ) {
		$sub_condition = ( ! empty( $data['sub_condition'] ) ) ? $data['sub_condition'] : '';
		$item_value    = ( ! empty( $data['item_value'] ) ) ? $data['item_value'] : '';
		$group_id      = ( ! empty( $data['group_id'] ) ) ? $data['group_id'] : 0;
		$rounds_count  = ( ! empty( $data['rounds_count'] ) ) ? $data['rounds_count'] : 0;

		if ( 'specific' === $sub_condition ) {

			if ( empty( $item_value ) ) {
				return AccessControls::instance()->return_users( array() );
			}

			/**
			 * Preparing the data into tmp items collections.
			 */
			$query_args = array(
				'number' => $per_page,
				'paged'  => $page,
			);

			$ld_members = learndash_get_users_for_course( $item_value, $query_args, false );
			if ( ( ( $ld_members instanceof \WP_User_Query ) && ( property_exists( $ld_members, 'results' ) ) ) || is_array( $ld_members ) ) {

				// If user not set on list it's return blank array.
				if ( is_array( $ld_members ) ) {
					$user_ids = ! empty( $ld_members ) ? array_unique( $ld_members ) : array();
				} else {
					$user_ids = $ld_members->get_results();
				}

				return AccessControls::instance()->return_users( $user_ids );

			} else {
				return AccessControls::instance()->return_error( __( 'Error while getting users results from LearnDash function.', 'buddyboss-app' ) );
			}
		} elseif ( 'any' === $sub_condition ) {
			$transient_key = "access_ctrls_ld_course_condition_jb_data_{$group_id}";
			// Check if it's a first round.
			if ( 1 === $rounds_count ) {
				$condition_data                   = array(
					'prepare_page'       => 1,
					'is_data_prepared'   => false,
					'pending_course_ids' => $this->learndash_get_all_course_ids(),
				);
				$condition_data['current_course'] = array_key_first( $condition_data['pending_course_ids'] );
			} else {
				$condition_data = get_transient( $transient_key );
			}
			/**
			 * Preparing the data into tmp items collections.
			 */
			if ( ! $condition_data['is_data_prepared'] ) {

				$query_args = array(
					'number' => $per_page,
					'paged'  => ( ! empty( $condition_data['prepare_page'] ) ) ? $condition_data['prepare_page'] : 1,
				);
				$ld_members = learndash_get_users_for_course( $condition_data['current_course'], $query_args, false );
				if ( ( ( $ld_members instanceof \WP_User_Query ) && ( property_exists( $ld_members, 'results' ) ) ) || is_array( $ld_members ) ) {

					// If user not set on list it's return blank array.
					if ( is_array( $ld_members ) ) {
						$user_ids = ! empty( $ld_members ) ? array_unique( $ld_members ) : array();
					} else {
						$user_ids = $ld_members->get_results();
					}

					if ( ! empty( $user_ids ) ) {
						$items = array();

						foreach ( $user_ids as $user_id ) {
							$items[] = array(
								'item_type'         => "ld_courses_{$group_id}",
								'item_id'           => $user_id,
								'secondary_item_id' => $condition_data['current_course'],
							);
						}

						$insert = TmpItems::instance()->insert_items( $items );

						$condition_data['prepare_page'] ++;

						if ( empty( $insert ) ) {
							return AccessControls::instance()->return_error( __( 'Error while preparing members data for LearnDash.', 'buddyboss-app' ) );
						}
					} else { // when all users are processed.

						unset( $condition_data['pending_course_ids'][ $condition_data['current_course'] ] ); // remove from pending course id.

						if ( ! empty( $condition_data['pending_course_ids'] ) ) {
							$condition_data['current_course'] = array_key_first( $condition_data['pending_course_ids'] ); // get next course to process.
							$condition_data['prepare_page']   = 1; // reset pagination to 1.
						} else {
							// If pending courses found empty finish the job.
							$condition_data['is_data_prepared'] = true;
						}
					}

					set_transient( $transient_key, $condition_data, DAY_IN_SECONDS );

					return AccessControls::instance()->return_wait( __( 'Preparing LD Members Data', 'buddyboss-app' ) );

				} else {
					return AccessControls::instance()->return_error( __( 'Error while getting users results from LearnDash function.', 'buddyboss-app' ) );
				}
			}

			/**
			 * Start Sending Users Data Once Tmp Collection Preparations has done.
			 * Once the Preparations for any of specific courses are done above the execution will reach here.
			 */

			$items = TmpItems::instance()->get_items(
				array(
					'include_item_types' => array( "ld_courses_{$group_id}" ),
					'per_page'           => $per_page,
					'page'               => $page,
				)
			);

			$user_ids = array();

			foreach ( $items['results'] as $item ) {
				$user_ids[] = $item->item_id;
			}

			return AccessControls::instance()->return_users( $user_ids );
		}

		return AccessControls::instance()->return_error( __( 'Invalid sub condition.', 'buddyboss-app' ) );
	}

	/**
	 * Return all courses IDs
	 *
	 * Reference - https://developers.learndash.com/snippet/get-all-course-ids/
	 *
	 * @return mixed
	 */
	public function learndash_get_all_course_ids() {
		$query_args = array(
			'post_type'   => 'sfwd-courses',
			'post_status' => 'publish',
			'fields'      => 'ids',
			'orderby'     => 'title',
			'order'       => 'ASC',
			'nopaging'    => true,    // Turns OFF paging logic to get ALL courses.
		);

		$query = new \WP_Query( $query_args );
		if ( $query instanceof \WP_Query ) {
			$ids    = $query->posts;
			$return = array();
			foreach ( $ids as $id ) {
				$return[ $id ] = $id;
			}

			return $return;
		}

		return array();
	}

	/**
	 * Function to check if user has any condition.
	 *
	 * @param int $user_id User id to check.
	 *
	 * @since 1.5.2.1
	 *
	 * @return bool
	 */
	public function has_any_items_callback( $user_id ) {
		$course_ids = learndash_get_user_course_access_list( $user_id );

		return ( ! empty( $course_ids ) ) ? true : false;
	}

	/**
	 * This function to add/remove user from access control.
	 *
	 * @param int     $user_id            User ID.
	 * @param int     $course_id          Course ID.
	 * @param string  $course_access_list A comma-separated list of user IDs used for the course_access_list field.
	 * @param boolean $remove             Whether to remove course access from the user.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_learndash_update_course_access( $user_id, $course_id, $course_access_list, $remove ) {
		if ( empty( $user_id ) || empty( $course_id ) ) {
			return;
		}

		if ( learndash_get_post_type_slug( 'course' ) === get_post_type( $course_id ) ) {
			$settings = learndash_get_setting( $course_id );
			if ( ! isset( $settings['course_price_type'] ) ) {
				if ( ! defined( 'LEARNDASH_DEFAULT_COURSE_PRICE_TYPE' ) ) {
					$settings['course_price_type'] = LEARNDASH_DEFAULT_COURSE_PRICE_TYPE;
				} else {
					$settings['course_price_type'] = '';
				}
			}
			if ( isset( $settings['course_price_type'] ) && 'open' !== $settings['course_price_type'] ) {
				if ( ! $remove ) {
					AccessControls::instance()->condition_add_user( $user_id, $this->condition_name, $course_id );
				} else {
					AccessControls::instance()->condition_remove_user( $user_id, $this->condition_name, $course_id );
				}
			}
		}
	}

	/**
	 * Function will remove member/course from access groups when remove course from admin.
	 *
	 * @param string   $new_status New post status.
	 * @param string   $old_status Old post status.
	 * @param \WP_Post $post       Post object.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_remove_learndash_access_control_data( $new_status, $old_status, $post ) {
		if ( 'publish' !== $new_status && 'sfwd-courses' === $post->post_type ) {
			AccessControls::instance()->condition_item_deleted( $this->condition_name, $post->ID );
			AccessControls::instance()->condition_remove_all_users( $this->condition_name, $post->ID );
			bb_access_recalculate_member_for_has_any_membership_group( $this->condition_name );
		}
	}

	/**
	 * Fires at the end of the user activation process.
	 *
	 * @param int $user_id ID of the user being checked.
	 *
	 * @since 1.5.2.1
	 */
	public function ld_register_user_success( $user_id ) {
		if ( function_exists( 'bp_get_user_meta' ) && ! bp_get_user_meta( $user_id, 'activation_key', true ) ) {
			$enrolled_courses = learndash_user_get_enrolled_courses( $user_id );
			update_user_meta( $user_id, 'bb_ac_enrolled_courses_queue', $enrolled_courses );
			$this->bb_start_user_enrolled_for_open_courses_job( $user_id );
		}
	}

	/**
	 * Fires at the end of the user activation process.
	 *
	 * @param int    $user_id ID of the user being checked.
	 * @param string $key     Activation key.
	 * @param array  $user    Array of user data.
	 *
	 * @since 1.5.2.1
	 */
	public function core_activated_user( $user_id, $key, $user ) {
		$enrolled_courses = learndash_user_get_enrolled_courses( $user_id );
		update_user_meta( $user_id, 'bb_ac_enrolled_courses_queue', $enrolled_courses );
		$this->bb_start_user_enrolled_for_open_courses_job( $user_id );
	}

	/**
	 * Function to start Enrolled users from open course job.
	 *
	 * @param int $user_id User id.
	 *
	 * @since 1.5.2.1
	 * @return bool
	 */
	public function bb_start_user_enrolled_for_open_courses_job( $user_id ) {

		// Add the queue job to run on background.
		$bbapp_queue = Jobs::instance();
		$bbapp_queue->add(
			'ld_enrolled_course',
			array(
				'user_id' => $user_id,
			),
			1
		); // job type allows only 20 char.

		return $bbapp_queue->start();
	}

	/**
	 * Function run when run the background job for processing enrolled courses.
	 *
	 * @param object $task BG job task name.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function user_enrolled_for_open_courses_job( $task ) {
		$task_data        = maybe_unserialize( $task->data );
		$user_id          = $task_data['user_id'];
		$enrolled_courses = get_user_meta( $user_id, 'bb_ac_enrolled_courses_queue', true );
		$i                = 1;
		// Add job stat.
		if ( ! empty( $enrolled_courses ) ) {
			foreach ( $enrolled_courses as $key => $course_id ) {
				if ( $i > 20 ) {
					update_user_meta( $user_id, 'bb_ac_enrolled_courses_queue', $enrolled_courses );
					$this->bb_start_user_enrolled_for_open_courses_job( $user_id );
					break;
				}
				AccessControls::instance()->condition_add_user( $user_id, $this->condition_name, $course_id );
				unset( $enrolled_courses[ $key ] );
				$i ++;
			}
		}
	}

	/**
	 * Function run when run the background job for processing groups.
	 *
	 * @param int    $meta_id     ID of the metadata entry to update.
	 * @param int    $object_id   ID of the object metadata is for.
	 * @param string $meta_key    Metadata key.
	 * @param mixed  $_meta_value Metadata value.
	 *
	 * @since 1.5.2.1
	 */
	public function course_price_type_update( $meta_id, $object_id, $meta_key, $_meta_value ) {
		if ( '_sfwd-courses' === $meta_key ) {
			$_meta_value = maybe_unserialize( $_meta_value );
			if ( ! empty( $_meta_value ) && isset( $_meta_value['sfwd-courses_course_price_type'] ) ) {
				AccessControls::instance()->condition_recalculate( $this->condition_name, $object_id );

				/* //phpcs:ignore Squiz.PHP.CommentedOutCode.Found
				$open_courses = learndash_get_open_courses();
				if ( ! empty( $open_courses ) && count( $open_courses ) > 1 && in_array( $object_id, $open_courses, true ) ) {
					// AccessControls::instance()->condition_recalculate_any( $this->_condition_name );.
				}
				*/
			}
		}
	}
}
