<?php
/**
 * Holds LearnDash category rule.
 *
 * @package BuddyBossApp\Integrations\Learndash\AccessControls
 */

namespace BuddyBossApp\Integrations\Learndash\AccessControls;

use BuddyBossApp\AccessControls\AccessRule;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

if ( ! class_exists( '\WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * App Learndash category rule custom list.
 */
class LDCategoryRuleList extends \WP_List_Table {

	/**
	 * Class constructor
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {

		parent::__construct(
			array(
				'singular' => __( 'Category & Tag', 'buddyboss-app' ), // Singular name of the listed records.
				'plural'   => __( 'Categories & Tags', 'buddyboss-app' ), // Plural name of the listed records.
				'ajax'     => false, // should this table support ajax?
				'screen'   => 'ld-category-tag-list',
			)
		);
	}

	/**
	 * Return the column available to this table list.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function get_columns() {
		return array(
			'reorder' => '',
			'cb'      => '<input type="checkbox" />',
			'name'    => sprintf( '<strong class="bb-bold-text">%1$s</strong>', _x( 'Name', 'access controls category settings', 'buddyboss-app' ) ),
			'type'    => sprintf( '<strong class="bb-bold-text">%1$s</strong>', _x( 'Type ', 'access controls category settings', 'buddyboss-app' ) ),
			'allow'   => sprintf( '<strong class="bb-bold-text">%1$s</strong>', _x( 'Allow ', 'access controls category settings', 'buddyboss-app' ) ),
			'manage'  => sprintf( '<strong class="bb-bold-text">%1$s</strong>', _x( 'Manage ', 'access controls category settings', 'buddyboss-app' ) ),
		);
	}

	/**
	 * No groups found
	 *
	 * @since 1.5.2.1
	 */
	public function no_items() {
		esc_html_e( 'No access rules found.', 'buddyboss-app' );
	}

	/**
	 * Access Groups column default values.
	 *
	 * @param object $item        Group item.
	 * @param string $column_name Column name.
	 *
	 * @since 1.5.2.1
	 * @return string|void
	 */
	public function column_default( $item, $column_name ) {

		switch ( $column_name ) {
			case 'name':
				$title = $item->name;
				if ( empty( $title ) ) {
					$title = esc_html__( '(no title)', 'buddyboss-app' );
				}
				printf( '<a href="%1$s"><strong>%2$s</strong></a>', '#', esc_html( $title ) );
				break;
			case 'type':
				$taxonomy_type_name = $this->get_taxonomy_type_name( $item );
				echo esc_html( $taxonomy_type_name );
				break;
			case 'allow':
				$rule_data = AccessRule::instance()->get_access_rule( $item->term_id, LDCategoryRule::instance()->item_type );
				$rule_data = ! empty( $rule_data['rule_data'] ) ? $rule_data['rule_data'] : array();
				echo wp_kses_post( bb_access_controls_rule_preview( $rule_data ) );
				break;
			case 'manage':
				$delete_rules_args = array(
					'page'         => 'bbapp-access-controls',
					'tab'          => 'courses',
					'sub_tab'      => 'cat_tag',
					'action'       => 'delete',
					'cat_rule_ids' => $item->term_id,
					'item_type'    => LDCategoryRule::instance()->item_type,
					'_wpnonce'     => wp_create_nonce( 'acc-delete-cat-rule' ),
				);
				printf(
					'<div class="action-div-wrp submitbox"><span class="edit"><a href="javascript:void(0);" class="bb-access-control-edit-tax-rule" data-tax-id="%1$s" data-tax-type="%2$s">%3$s</a> | </span><span class="delete"><a class="submitdelete bb-dlt-cat-rule" href="%4$s" onclick="return showNotice.warn();">%5$s</a></span> </div>',
					esc_attr( $item->term_id ),
					esc_html( LDCategoryRule::instance()->item_type ),
					esc_html__( 'Edit Rule', 'buddyboss-app' ),
					esc_url( add_query_arg( $delete_rules_args, bbapp_get_admin_url( 'admin.php' ) ) ),
					esc_html__( 'Remove Rule', 'buddyboss-app' )
				);
				break;
			default:
				echo 'N/A';
				break;
		}
	}

	/**
	 * Checkbox for bulk items.
	 *
	 * @param object $item Item row.
	 *
	 * @since 1.5.2.1
	 *
	 * @return string|void
	 */
	public function column_cb( $item ) {
		if ( current_user_can( 'manage_options' ) ) {
			return sprintf( '<input type="checkbox" name="cat_rule_ids[]" value="%s" />', $item->term_id );
		}

		return '';
	}

	/**
	 * Checkbox for bulk items.
	 *
	 * @param object $item Item row.
	 *
	 * @return string
	 */
	public function column_reorder( $item ) {
		return sprintf( '<span class="bbapp-cat-reorder-rule"><i class="bb-icon-l bb-icon-bars"></i></span>' );
	}

	/**
	 * Handles data query and filter, sorting, and pagination.
	 *
	 * @since 1.5.2.1
	 */
	public function prepare_items() {
		/**
		 * Init column headers.
		 */
		$this->_column_headers = array( $this->get_columns(), array(), $this->get_sortable_columns() );

		/** Process bulk action */
		$this->process_bulk_action();

		$this->items = LDCategoryRule::instance()->get_stored_category_access_rules();
	}

	/**
	 * Generates content for a single row of the table
	 *
	 * @param object $item The current item.
	 *
	 * @since 1.5.2.1
	 */
	public function single_row( $item ) {
		static $count = 1;
		echo '<tr id="cat-id-' . esc_attr( $item->term_id ) . '" data-rule-id=' . esc_attr( $item->term_id ) . '>';
		$this->single_row_columns( $item );
		echo '</tr>';
		$count ++;
	}

	/**
	 * Display the table
	 *
	 * @since 1.5.2.1
	 */
	public function display() {
		$singular = $this->_args['singular'];

		$this->display_tablenav( 'top' );

		$this->screen->render_screen_reader_content( 'heading_list' );
		?>
		<table class="wp-list-table <?php echo implode( ' ', array_map( 'esc_attr', $this->get_table_classes() ) ); ?>">
			<thead>
			<tr>
				<?php $this->print_column_headers(); ?>
			</tr>
			</thead>

			<tbody id="the-list"
				<?php
				if ( $singular ) {
					echo " data-wp-lists='list:" . esc_attr( $singular ) . "'";
				}
				?>
			>
			<?php $this->display_rows_or_placeholder(); ?>
			</tbody>

			<tfoot>
			<tr>
				<?php $this->print_column_headers( false ); ?>
			</tr>
			</tfoot>
		</table>
		<?php
	}

	/**
	 * Bulk action.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function get_bulk_actions() {
		$actions['bulk-delete'] = __( 'Delete', 'buddyboss-app' );

		return $actions;
	}

	/**
	 * Bulk action process.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function process_bulk_action() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( 'You don\'t have permission to access this page.', 'buddyboss-app' );
		}

		$request_uri = ( ! empty( $_SERVER['REQUEST_URI'] ) ) ? bbapp_input_clean( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$redirect_to = remove_query_arg(
			array(
				'action',
				'action2',
				'cat_rule_ids',
				'deleted',
				'error',
				'updated',
				'success_new',
				'error_new',
				'success_modified',
				'error_modified',
				'_wpnonce',
				'_wp_http_referer',
				'paged',
			),
			$request_uri
		);

		$action = $this->current_action();

		switch ( $action ) {
			case 'delete':
			case 'bulk-delete':
				if ( ! empty( $_REQUEST['cat_rule_ids'] ) ) {
					$nonce_key = ( 'bulk-delete' === $action ) ? 'bulk-' . sanitize_key( $this->_args['plural'] ) : 'acc-delete-cat-rule';
					check_admin_referer( $nonce_key );

					$cat_rule_ids = wp_parse_id_list( map_deep( $_REQUEST['cat_rule_ids'], 'absint' ) );
					$item_type    = ( ! empty( $_REQUEST['item_type'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['item_type'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

					$access_rules = AccessRule::instance()->get_access_rules(
						array(
							'include_item_types' => $item_type,
							'include_item_id'    => $cat_rule_ids,
						)
					);
					$count        = 0;
					if ( ! empty( $access_rules['count'] ) ) {
						foreach ( $access_rules['results'] as $access_rule ) {
							$deleted = AccessRule::instance()->delete_access_rule( $access_rule['id'] );
							if ( ! is_wp_error( $deleted ) ) {
								$item_type = LDCategoryRule::instance()->item_type;
								if ( ! empty( $access_rule['item_id'] ) && delete_term_meta( $access_rule['item_id'], "bb_access_control_{$item_type}_menu_order" ) ) {
									$count ++;
								}
							}
						}
					}
					$redirect_to = add_query_arg(
						array(
							'deleted' => $count,
							'ids'     => $cat_rule_ids,
						),
						$redirect_to
					);

					wp_safe_redirect( $redirect_to );
					exit();
				}
				break;
		}
	}

	/**
	 * Taxonomy type name.
	 *
	 * @param object $item Taxonomy term object.
	 *
	 * @since 1.5.2.1
	 * @return string
	 */
	public function get_taxonomy_type_name( $item ) {
		$taxonomy_details = get_taxonomy( $item->taxonomy );

		return esc_html( $taxonomy_details->labels->singular_name );
	}
}
