<?php

namespace BuddyBossApp\Integrations\H5P;

use H5P_Plugin_Admin;
use H5PContentQuery;
use WP_REST_Controller;
use WP_REST_Server;

class RestAPI extends WP_REST_Controller {

	protected $namespace = "buddyboss-app/h5p/v1";
	protected $rest_base = 'items';
	protected static $instance;

	/**
	 * RestAPI constructor.
	 */
	public function __construct() {
		/** Nothing here */
	}

	/**
	 * @return mixed
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
			self::$instance->hooks(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Load hooks.
	 */
	public function hooks() {
		add_action( 'init', array( $this, 'init' ) );

		// only on rest API.
		if ( strpos( $_SERVER['REQUEST_URI'], "/wp-json/" ) !== false ) {
			add_filter( 'the_content', array( $this, 'parse_h5p_content' ), 10 );
		}

	}

	/**
	 * Init method.
	 */
	public function init() {
		if ( class_exists( 'H5P_Plugin' ) ) {
			add_action( 'rest_api_init', array( $this, "register_routes" ), 99 );
		}
	}

	/**
	 * Parse the H5p shortcode to have Iframe URL in SRC.
	 * @param $content
	 *
	 * @return mixed
	 */
	public function parse_h5p_content( $content ) {
		if( class_exists('\H5P_Plugin') ) {
			$pattern = get_shortcode_regex();
			$attr = array();
			$searchArray = array();
			$replaceArray = array();

			$core = \H5P_Plugin::get_instance()->get_h5p_instance('core');
			if (   preg_match_all( '/'. $pattern .'/s', $content, $matches )
			       && array_key_exists( 2, $matches )
			       && in_array( 'h5p', $matches[2] )
			) {
				foreach( (array) $matches[2] as $key => $value )
				{
					if( 'h5p' === $value ){
						$attr[] = shortcode_parse_atts( $matches[3][$key] );
						$id = shortcode_parse_atts( $matches[3][$key] )['id'];
						$loadContent = $core->loadContent($id);

						if (!$loadContent) {
							$replaceArray[] = '<div class="h5p-iframe-wrapper">' . sprintf(__('Cannot find H5P content with id: %d.', 'buddyboss-app' ), $id ) . '</div>';
						} else {
							$replaceArray[] = '<div class="h5p-iframe-wrapper"><iframe id="h5p-iframe-' . $id. '" class="h5p-iframe" data-content-id="' . $id . '"  src="' . admin_url( 'admin-ajax.php?action=h5p_embed&id='. $id ) . '" frameBorder="0" scrolling="no"></iframe></div>';
						}
						$searchArray[] = $matches[0][$key];
					}

				}
				$content = str_replace( $searchArray , $replaceArray, $content );
			}

		}

		return $content;
	}

	/**
	 * Register routes.
	 */
	public function register_routes() {
		if ( class_exists( 'H5P_Plugin' ) ) {
			register_rest_route( $this->namespace, $this->rest_base, array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'args'                => array(),
					'permission_callback' => array( $this, 'rest_api_permission' ),
				),
			) );
		}
	}

	/**
	 * Api permission.
	 *
	 * @param $request
	 *
	 * @return mixed|void
	 */
	public function rest_api_permission( $request ) {
		$retval = true;

		/**
		 * Filter the course H5P list `get_items` permissions check.
		 *
		 * @since 0.1.0
		 */
		return apply_filters( 'bbapp_h5p_content_list_permissions_check', $retval, $request );
	}

	/**
	 * Get H5P items list for H5P block select option.
	 *
	 * @param $request
	 *
	 * @return array|\WP_Error|\WP_HTTP_Response|\WP_REST_Response
	 */
	public function get_items( $request ) {

		$h5p_items = $this->h5p_content_list();

		$response = array();
		if ( ! empty( $h5p_items ) ) {
			foreach ( $h5p_items as $h5p_item ) {

				$data       = $this->prepare_item_for_response( $h5p_item, $request );
				$response[] = $this->prepare_response_for_collection( $data );
			}
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param $item
	 * @param $request
	 *
	 * @return mixed|void
	 */
	public function prepare_item_for_response( $item, $request ) {
		// Base fields for every post.
		$data = array(
			'id'           => $item->id,
			'user_id'      => $item->user_id,
			'title'        => $item->title,
			'library_id'   => $item->library_id,
			'slug'         => $item->slug,
			'content_type' => $item->content_type_id,
		);


		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		return apply_filters( 'bbapp_rest_prepare_h5p_item', $response, $item, $request );
	}

	/**
	 * Fetch H5P Content List.
	 *
	 * @return array
	 * @throws \Exception
	 */
	public function h5p_content_list() {
		// Load input vars.
		$admin = H5P_Plugin_Admin::get_instance();
		list( $offset, $limit, $sort_by, $sort_dir, $filters, $facets ) = $admin->get_data_view_input();


		$fields = array( 'title', 'content_type', 'user_name', 'tags', 'updated_at', 'id', 'user_id', 'content_type_id' );


		// Add filters to data query
		$conditions = array();
		if ( isset( $filters[0] ) ) {
			$conditions[] = array( 'title', $filters[0], 'LIKE' );
		}

		// Limit query to content types that user is allowed to view
		if ( current_user_can( 'view_others_h5p_contents' ) == false ) {
			array_push( $conditions, array( 'user_id', get_current_user_id(), '=' ) );
		}

		if ( $facets !== null ) {
			$facetmap = array(
				'content_type' => 'content_type_id',
				'user_name'    => 'user_id',
				'tags'         => 'tags',
			);
			foreach ( $facets as $field => $value ) {
				if ( isset( $facetmap[ $fields[ $field ] ] ) ) {
					$conditions[] = array( $facetmap[ $fields[ $field ] ], $value, '=' );
				}
			}
		}

		// Create new content query
		$content_query = new H5PContentQuery( $fields, $offset, null, $fields[ $sort_by ], $sort_dir, $conditions );

		return $content_query->get_rows();
	}

}