<?php

namespace BuddyBossApp\Integrations\GamiPress;

use WP_Error as WP_Error;

class RestAPIRank extends \WP_REST_Controller {

	protected $namespace = "buddyboss-app/gamipress/v1";
	protected $rest_base = 'ranks';
	protected static $instance;

	/**
	 * RestAPIRank constructor.
	 */
	public function __construct() {
		/** Nothing here */
	}

	/**
	 * @return mixed
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
			self::$instance->hooks(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 *
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, "registerRoutes" ), 99 );
	}

	/**
	 *
	 */
	public function registerRoutes() {
		register_rest_route( $this->namespace, '/' . $this->rest_base, array(
			array(
				'methods'             => \WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_items' ),
				'permission_callback' => array( $this, 'get_items_permissions_check' ),
				'args'                => $this->get_collection_params(),
			),
			'schema' => array( $this, 'get_public_item_schema' ),
		) );
		$this->register_gamipress_users_fields();
	}

	/**
	 * Register Rest field
	 */
	public function register_gamipress_users_fields() {

		register_rest_field( 'user',
			'user_ranks',
			array(
				'get_callback'    => array( $this, "get_user_ranks" ),
				'update_callback' => null,
				'schema'          => null,
			)
		);

		register_rest_field( 'bp_members',
			'user_ranks',
			array(
				'get_callback'    => array( $this, "get_user_ranks" ),
				'update_callback' => null,
				'schema'          => null,
			)
		);
	}

	/**
	 * Get user all type of ranks
	 *
	 * @param $object
	 * @param $field_name
	 * @param $request
	 *
	 * @return array
	 */
	public function get_user_ranks( $object, $field_name, $request ) {
		$ranks            = array();
		$rank_types       = gamipress_get_rank_types();
		$ranks_type_slugs = gamipress_get_rank_types_slugs();
		$context          = ( ! empty( $request['screen'] ) && 'reply' === $request['screen'] ) ? 'forum' : 'member';
		$display_options  = \BuddyBossApp\Integrations\GamiPress\RestAPI::instance()->gamipress_display_options( 'rank', $context );

		if ( empty( $display_options ) ) {
			return $ranks;
		}

		// Get ranks display settings
		$rank_types_to_show = ! empty( $display_options['rank_types_to_show'] ) ? $display_options['rank_types_to_show'] : array();

		if ( empty( $rank_types_to_show ) ) {
			return $ranks;
		}

		$is_thumbnail_enable = (bool) $display_options['rank_types_thumbnail'];
		$show_title          = (bool) $display_options['rank_types_title'];
		$show_link           = (bool) $display_options['rank_types_link'];
		$show_label          = (bool) $display_options['rank_types_label'];
		$rank_types_hide     = (bool) $display_options['rank_types_hide'];

		foreach ( $rank_types_to_show as $rank_type_to_show ) {
			// If rank type not registered, skip.
			if ( ! in_array( $rank_type_to_show, $ranks_type_slugs ) ) {
				continue;
			}

			$_user_rank = gamipress_get_user_rank( $object["id"], $rank_type_to_show );

			// Skip rank if not exists.
			if ( ! gamipress_get_post( $_user_rank->ID ) ) {
				continue;
			}

            $user_rank_is_lowest = gamipress_is_lowest_priority_rank( $_user_rank->ID );

			if ( ! class_exists( 'GamiPress_BuddyBoss' ) && $rank_types_hide && $user_rank_is_lowest ) {
				continue;
			}

			$ranks[] = array(
				'rank'          => array(
					'thumbnail_url' => ( true === $is_thumbnail_enable ) ? RestAPI::gamipress_get_post_thumbnail_url( $_user_rank->ID, 'rank-type', 'rank' ) : false,
					'link'          => ( true === $show_link ) ? get_permalink( $_user_rank->ID ) : '',
					'title'         => ( true === $show_title ) ? get_the_title( $_user_rank->ID ) : '',
				),
				'label'         => ( true === $show_label ) ? $rank_types[ $rank_type_to_show ]['singular_name'] : '',
				'singular_name' => ( true === $show_label ) ? $rank_types[ $rank_type_to_show ]['singular_name'] : '',
				'plural_name'   => ( true === $show_label ) ? $rank_types[ $rank_type_to_show ]['plural_name'] : '',
				'image'         => ( true === $is_thumbnail_enable ) ? RestAPI::gamipress_get_post_thumbnail_url( $rank_types[ $rank_type_to_show ]['ID'], 'rank-type', 'rank', 'gamipress-achievement', false ) : false,
			);
		}

		return array_values( $ranks );
	}


	/**
	 * @param $request
	 *
	 * @return array
	 * @api            {GET} /wp-json/buddyboss-app/gamipress/v1/ranks Ranks
	 * @apiName        GetGamiPressRanks
	 * @apiGroup       GamiPress
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get ranks
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {String=view,edit,embed} [context=view]
	 * @apiParam {Number} user_id Limit result set to specific user.
	 */
	public function get_items( $request ) {

		if ( ! empty( $request['user_id'] ) ) {
			$user_id = (int) $request['user_id'];
		} else {
			$user_id = get_current_user_id();
		}

		$ranks               = array();
		$rank_types          = gamipress_get_rank_types();
		$ranks_types_to_show = gamipress_bp_tab_get_ranks_types();
		$rank_types_hide     = (bool) gamipress_bp_get_option( 'tab_ranks_types_hide', false );

		foreach ( $ranks_types_to_show as $rank_type_slug ) {

			// Skip if rank not exists.
			if ( ! isset( $rank_types[ $rank_type_slug ] ) ) {
				continue;
			}
			
			$user_rank           = gamipress_get_user_rank( $user_id, $rank_type_slug );
			$user_rank_is_lowest = gamipress_is_lowest_priority_rank( $user_rank->ID );

			if ( ! class_exists( 'GamiPress_BuddyBoss' ) && $rank_types_hide && $user_rank_is_lowest ) {
				continue;
			}

			$allow_with_points     = 'on' === gamipress_get_post_meta( $user_rank->ID, '_gamipress_unlock_with_points' );
			$obj_rank              = new \stdClass();
			$obj_rank->ID          = $user_rank->ID;
			$obj_rank->title       = get_the_title( $user_rank->ID );
			$obj_rank->link        = get_permalink( $user_rank->ID );
			$obj_rank->content     = ! empty( $user_rank->post_excerpt ) ? $user_rank->post_excerpt : $user_rank->post_content;
			$obj_rank->points      = true === $allow_with_points ? gamipress_get_post_meta( $user_rank->ID, '_gamipress_points_to_unlock', true ) : 0;
			$obj_rank->points_type = true === $allow_with_points ? gamipress_get_post_meta( $user_rank->ID, '_gamipress_points_type_to_unlock', true ) : '';
			$obj_rank->image       = RestAPI::gamipress_get_post_thumbnail_url( $user_rank->ID, 'rank-type', 'rank' );
			$obj_rank->settings    = array(
				'allow_with_points' => $allow_with_points,
			);

			$data    = $this->prepare_item_for_response( $obj_rank, $request );
			$ranks[] = $this->prepare_response_for_collection( $data );
		}

		$response = rest_ensure_response( $ranks );

		return $response;
	}

	/**
	 * @param $request
	 *
	 * @return mixed
	 */
	public function get_items_permissions_check( $request ) {
		return apply_filters( 'bbapp_gamipress_ranks_items_permissions', true, $request );
	}

	/**
	 * Get the Post's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {

		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'gamipress_rank',
			'type'       => 'object',
			/*
				 * Base properties for every Post.
			*/
			'properties' => array(
				'id'       => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title'    => array(
					'description' => __( 'The title for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'content'  => array(
					'description' => __( 'The content for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Content for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML content for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit' ),
						),
					),
				),
				'link'     => array(
					'description' => __( 'URL to the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'image'    => array(
					'description' => __( 'URL to the image object.', 'buddyboss-app' ),
					'type'        => 'image',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'settings' => array(
					'description' => __( 'Settings of the object.', 'buddyboss-app' ),
					'type'        => 'array',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
			),
		);

		$schema['properties']['points'] = array(
			'description' => __( 'Point for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['points_type'] = array(
			'description' => __( 'Point type for the object', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		$params['context']['default'] = 'view';

		$params['user_id'] = array(
			'description'       => __( 'Limit result set to specific user.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param WP_Post         $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_Error $data
	 */
	public function prepare_item_for_response( $post, $request ) {
		$data = array();

		if ( isset( $post->ID ) ) {
			$data['id'] = $post->ID;
		} else {
			return new WP_Error( 'gamipress_json_internal_error', __( 'Required field "ID" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$schema = $this->get_item_schema();

		if ( ! empty( $schema['properties']['title'] ) ) {
			if ( isset( $post->title ) ) {
				$data['title'] = array(
					'raw'      => $post->title,
					'rendered' => $post->title,
				);
			} else {
				return new WP_Error( 'gamipress_json_internal_error', __( 'Required field "Title" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['content'] ) ) {
			if ( isset( $post->content ) ) {
				$data['content'] = array(
					'raw'      => $post->content,
					/** This filter is documented in wp-includes/post-template.php */
					'rendered' => apply_filters( 'the_content', $post->content ),
				);
			} else {
				return new WP_Error( 'gamipress_json_internal_error', __( 'Required field "Content" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['points'] ) ) {
			$data['points'] = (int) $post->points;
		}

		if ( ! empty( $schema['properties']['points_type'] ) ) {
			$data['points_type'] = $post->points_type;
		}

		if ( ! empty( $schema['properties']['link'] ) ) {
			$data['link'] = $post->link;
		}

		if ( ! empty( $schema['properties']['image'] ) ) {
			$data['image'] = $post->image;
		}

		if ( ! empty( $schema['properties']['settings'] ) ) {
			$data['settings'] = $post->settings;
		}

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = $this->add_additional_fields_to_object( $data, $request );
		$data    = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		/**
		 * Filter the post data for a response.
		 *
		 * The dynamic portion of the hook name, $this->post_type, refers to post_type of the post being
		 * prepared for the response.
		 *
		 * @param WP_REST_Response $response The response object.
		 * @param WP_Post          $post     Post object.
		 * @param WP_REST_Request  $request  Request object.
		 */
		return apply_filters( "rest_prepare_gamipress_ranks", $response, $post, $request );
	}


}