<?php

namespace BuddyBossApp\Integrations\GamiPress;

use WP_Post;
use WP_REST_Request;
use WP_REST_Response;

class RestAPIPoint extends \WP_REST_Controller {

	protected $namespace = "buddyboss-app/gamipress/v1";
	protected $rest_base = 'points';
	protected static $instance;

	/**
	 * RestAPIPoint constructor.
	 */
	public function __construct() {
		/** Nothing here */
	}

	/**
	 * @return mixed
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
			self::$instance->hooks(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 *
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, "registerRoutes" ), 99 );
	}

	/**
	 *
	 */
	public function registerRoutes() {
		register_rest_route( $this->namespace, '/' . $this->rest_base, array(
			array(
				'methods'             => \WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_items' ),
				'permission_callback' => array( $this, 'get_items_permissions_check' ),
				'args'                => $this->get_collection_params(),
			),
			'schema' => array( $this, 'get_public_item_schema' ),
		) );

		$this->register_learner_fields();
	}

	/**
	 * Register Rest field
	 */
	public function register_learner_fields() {

		register_rest_field( 'user',
			'user_points',
			array(
				'get_callback'    => array( $this, "get_user_points" ),
				'update_callback' => null,
				'schema'          => null,
			)
		);

		register_rest_field( 'bp_members',
			'user_points',
			array(
				'get_callback'    => array( $this, "get_user_points" ),
				'update_callback' => null,
				'schema'          => null,
			)
		);
	}

	/**
	 * Get user all type of points
	 *
	 * @param $object
	 * @param $field_name
	 * @param $request
	 *
	 * @return array
	 */
	public function get_user_points( $object, $field_name, $request ) {
		$points             = array();
		$points_types       = gamipress_get_points_types();
		$points_types_slugs = gamipress_get_points_types_slugs();
		$context            = ( ! empty( $request['screen'] ) && 'reply' === $request['screen'] ) ? 'forum' : 'member';
		$display_options    = \BuddyBossApp\Integrations\GamiPress\RestAPI::instance()->gamipress_display_options( 'point', $context );

		if ( empty( $display_options ) ) {
			return $points;
		}

		$points_types_to_show = !empty( $display_options['points_types_to_show'] ) ? $display_options['points_types_to_show'] : array();

		if ( empty( $points_types_to_show ) ) {
			return $points;
		}

		$is_thumbnail_enable = (bool) $display_options['points_types_thumbnail'];
		$show_label          = (bool) $display_options['points_types_label'];
		$points_types_hide   = (bool) $display_options['points_types_hide'];

		foreach ( $points_types_to_show as $points_type_to_show ) {
			// If points type not registered, skip
			if ( ! in_array( $points_type_to_show, $points_types_slugs ) ) {
				continue;
			}

			$_point   = gamipress_get_user_points( $object["id"], $points_type_to_show );

			if ( ! class_exists( 'GamiPress_BuddyBoss' ) && $points_types_hide && $_point < 1 ) {
				continue;
			}

			$points[] = array(
				'point'          => gamipress_format_amount( $_point, $points_type_to_show ),
				'label'          => ( true === $show_label ) ? ( $_point > 1 ) ? $points_types[ $points_type_to_show ]['plural_name'] : $points_types[ $points_type_to_show ]['singular_name'] : '',
				'singular_name'  => ( true === $show_label ) ? $points_types[ $points_type_to_show ]['singular_name'] : '',
				'plural_name'    => ( true === $show_label ) ? $points_types[ $points_type_to_show ]['plural_name'] : '',
				'label_position' => gamipress_get_points_type_label_position( $points_type_to_show ),
				'image'          => ( true === $is_thumbnail_enable ) ? RestAPI::gamipress_get_post_thumbnail_url( $points_types[ $points_type_to_show ]['ID'], 'achievement-type', 'achievement', 'gamipress-achievement', false ) : false,
			);
		}

		return array_values( $points );
	}

	/**
	 * @param $request
	 *
	 * @return array
	 * @api            {GET} /wp-json/buddyboss-app/gamipress/v1/points Points
	 * @apiName        GetGamiPressPoints
	 * @apiGroup       GamiPress
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get points
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {String=view,edit,embed} [context=view]
	 * @apiParam {Number} user_id Limit result set to specific user.
	 */
	public function get_items( $request ) {

		if ( ! empty( $request['user_id'] ) ) {
			$user_id = (int) $request['user_id'];
		} else {
			$user_id = get_current_user_id();
		}

		$points               = array();
		$points_types         = gamipress_get_points_types();
		$points_types_to_show = gamipress_bp_tab_get_points_types();
		$points_types_hide    = (bool) gamipress_bp_get_option( 'tab_points_types_hide', false );

		foreach ( $points_types_to_show as $points_type_slug ) {

			// Skip if rank not exists
			if ( ! isset( $points_types[ $points_type_slug ] ) ) {
				continue;
			}

			$_point = gamipress_get_user_points( $user_id, $points_type_slug );

			if ( ! class_exists( 'GamiPress_BuddyBoss' ) && $points_types_hide && $_point < 1 ) {
				continue;
			}

			$obj_point                = new \stdClass();
			$obj_point->title         = ( $_point > 1 ) ? $points_types[ $points_type_slug ]['plural_name'] : $points_types[ $points_type_slug ]['singular_name'];
			$obj_point->singular_name = $points_types[ $points_type_slug ]['singular_name'];
			$obj_point->plural_name   = $points_types[ $points_type_slug ]['plural_name'];
			$obj_point->points        = gamipress_format_amount( $_point, $points_type_slug );
			$obj_point->image         = RestAPI::gamipress_get_post_thumbnail_url( $points_types[ $points_type_slug ]['ID'], 'achievement-type', 'achievement', 'gamipress-achievement', false );
			$obj_point->slug          = $points_type_slug;
			$data                     = $this->prepare_item_for_response( $obj_point, $request );
			$points[]                 = $this->prepare_response_for_collection( $data );
		}

		$response = rest_ensure_response( $points );

		return $response;
	}

	/**
	 * @param $request
	 *
	 * @return mixed
	 */
	public function get_items_permissions_check( $request ) {
		return apply_filters( 'bbapp_gamipress_points_items_permissions', true, $request );
	}

	/**
	 * Get the Post's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {

		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'gamipress_point',
			'type'       => 'object',
			/*
				 * Base properties for every Post.
			*/
			'properties' => array(
				'title' => array(
					'description' => __( 'The title for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'image' => array(
					'description' => __( 'URL to the image object.', 'buddyboss-app' ),
					'type'        => 'image',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
			),
		);

		$schema['properties']['singular_name'] = array(
			'description' => __( 'Point name for the object', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['plural_name'] = array(
			'description' => __( 'Point name for the object', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['label_position'] = array(
			'description' => __( 'Point label position for the object', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['points'] = array(
			'description' => __( 'Point for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		$params['context']['default'] = 'view';

		$params['user_id'] = array(
			'description'       => __( 'Limit result set to specific user.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param object          $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response( $post, $request ) {
		$data   = array();
		$schema = $this->get_item_schema();

		if ( ! empty( $schema['properties']['title'] ) ) {
			if ( isset( $post->title ) ) {
				$data['title'] = array(
					'raw'      => $post->title,
					'rendered' => $post->title,
				);
			} else {
				return rest_ensure_response( new \WP_Error( 'gamipress_json_internal_error', __( 'Required field "Title" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) ) );
			}
		}

		if ( ! empty( $schema['properties']['singular_name'] ) ) {
			$data['singular_name'] = $post->singular_name;
		}

		if ( ! empty( $schema['properties']['plural_name'] ) ) {
			$data['plural_name'] = $post->plural_name;
		}

		if ( ! empty( $schema['properties']['image'] ) ) {
			$data['image'] = $post->image;
		}

		if ( ! empty( $schema['properties']['label_position'] ) ) {
			$data['label_position'] = gamipress_get_points_type_label_position( $post->slug );
		}

		if ( ! empty( $schema['properties']['points'] ) ) {
			$data['points'] = $post->points;
		}

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = $this->add_additional_fields_to_object( $data, $request );
		$data    = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		/**
		 * Filter the post data for a response.
		 *
		 * The dynamic portion of the hook name, $this->post_type, refers to post_type of the post being
		 * prepared for the response.
		 *
		 * @param WP_REST_Response $response The response object.
		 * @param WP_Post          $post     Post object.
		 * @param WP_REST_Request  $request  Request object.
		 */
		return apply_filters( 'rest_prepare_gamipress_points', $response, $post, $request );
	}
}