<?php

namespace BuddyBossApp\Integrations\GamiPress;

use WP_Error as WP_Error;

class RestAPIAchievement extends \WP_REST_Controller {

	protected $namespace = "buddyboss-app/gamipress/v1";
	protected $rest_base = 'achievements';
	protected static $instance;

	/**
	 * RestAPIAchievement constructor.
	 */
	public function __construct() {
		/** Nothing here */
	}

	/**
	 * @return mixed
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
			self::$instance->hooks(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 *
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, "registerRoutes" ), 99 );
	}

	/**
	 *
	 */
	public function registerRoutes() {
		register_rest_route( $this->namespace, '/' . $this->rest_base, array(
			array(
				'methods'             => \WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_items' ),
				'permission_callback' => array( $this, 'get_items_permissions_check' ),
				'args'                => $this->get_collection_params(),
			),
			'schema' => array( $this, 'get_public_item_schema' ),
		) );
		$this->register_gamipress_users_fields();
	}

	/**
	 * Register Rest field
	 */
	public function register_gamipress_users_fields() {

		register_rest_field('user',
			'user_achievements',
			array(
				'get_callback' => array($this, "get_user_achievements"),
				'update_callback' => null,
				'schema' => null,
			)
		);

		register_rest_field('bp_members',
			'user_achievements',
			array(
				'get_callback' => array($this, "get_user_achievements"),
				'update_callback' => null,
				'schema' => null,
			)
		);
	}

	/**
	 * Get user all type of achievements
	 * @param $object
	 * @param $field_name
	 * @param $request
	 * @return array
	 */
	public function get_user_achievements( $object, $field_name, $request ) {
		$achievements            = array();
		$achievement_types       = gamipress_get_achievement_types();
		$achievement_types_slugs = gamipress_get_achievement_types_slugs();
		$context                 = ( ! empty( $request['screen'] ) && 'reply' === $request['screen'] ) ? 'forum' : 'member';
		$display_options         = \BuddyBossApp\Integrations\GamiPress\RestAPI::instance()->gamipress_display_options( 'achievement', $context );

		if ( empty( $display_options ) ) {
			return $achievements;
		}

		// Get achievements display settings
		$achievement_types_to_show = !empty( $display_options['achievement_types_to_show'] ) ? $display_options['achievement_types_to_show'] : array();

		if ( empty( $achievement_types_to_show ) ) {
			return $achievements;
		}

		$is_thumbnail_enable = (bool) $display_options['achievement_types_thumbnail'];
		$show_title          = (bool) $display_options['achievement_types_title'];
		$show_link           = (bool) $display_options['achievement_types_link'];
		$show_label          = (bool) $display_options['achievement_types_label'];
		$limit               = (int) $display_options['achievement_types_limit'];
		foreach ( $achievement_types_to_show as $achievement_type_to_show ) {
			// If achievements type not registered, skip
			if ( ! in_array( $achievement_type_to_show, $achievement_types_slugs ) ) {
				continue;
			}

			$args               = array(
				'user_id'          => $object["id"], // The given user's ID
				'achievement_type' => $achievement_type_to_show, // A specific achievement type
				'groupby'          => 'achievement_id',
				'limit'            => $limit,
			);
			$_user_achievements = gamipress_get_user_achievements( $args );
			$_achievement       = array();
			foreach ( $_user_achievements as $_user_achievement ) {
				// Skip achievement if not exists
				if ( ! gamipress_get_post( $_user_achievement->ID ) ) {
					continue;
				}

				$_achievement[] = array(
					'thumbnail_url' => ( true === $is_thumbnail_enable ) ? RestAPI::gamipress_get_post_thumbnail_url( $_user_achievement->ID ) : false,
					'link'          => ( true === $show_link ) ? get_permalink( $_user_achievement->ID ) : '',
					'title'         => ( true === $show_title ) ? get_the_title( $_user_achievement->ID ) : '',
				);
			}
			$achievements[] = array(
				'achievement'   => $_achievement,
				'label'         => ( true === $show_label ) ? ( count( $_achievement ) > 1 ) ? $achievement_types[ $achievement_type_to_show ]['plural_name'] : $achievement_types[ $achievement_type_to_show ]['singular_name'] : '',
				'singular_name' => ( true === $show_label ) ? $achievement_types[ $achievement_type_to_show ]['singular_name'] : '',
				'plural_name'   => ( true === $show_label ) ? $achievement_types[ $achievement_type_to_show ]['plural_name'] : '',
				'image'         => ( true === $is_thumbnail_enable ) ? RestAPI::gamipress_get_post_thumbnail_url( $achievement_types[ $achievement_type_to_show ]['ID'], 'achievement-type', 'achievement', 'gamipress-achievement', false ) : false,
			);
		}

		return array_values( $achievements );
	}

	/**
	 * @param $request
	 *
	 * @return array
	 * @api            {GET} /wp-json/buddyboss-app/gamipress/v1/achievements Achievements
	 * @apiName        GetGamiPressAchievements
	 * @apiGroup       GamiPress
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Get achievements
	 * @apiHeader {String} accessToken Auth token
	 * @apiParam {String=view,edit,embed} [context=view]
	 * @apiParam {Number} user_id Limit result set to specific user.
	 */
	public function get_items( $request ) {

		if ( ! empty( $request['user_id'] ) ) {
			$user_id = (int) $request['user_id'];
		} else {
			$user_id = get_current_user_id();
		}
		$achievements = array();

		$achivement_types = gamipress_get_achievement_types();
		$achivement_types = apply_filters( 'bbapp_gamipress_achievements_types', $achivement_types );

		$achivement_types_slugs = gamipress_get_achievement_types_slugs();
		$gamipress_settings     = ( $exists = get_option( 'gamipress_settings' ) ) ? $exists : array();
		$achivement_types_order = isset( $gamipress_settings['bp_members_achivement_types_order'] ) ? $gamipress_settings['bp_members_achivement_types_order'] : $achivement_types_slugs;

		foreach ( $achivement_types_order as $points_type_slug ) {

			// Skip if rank not exists
			if ( ! isset( $achivement_types[ $points_type_slug ] ) ) {
				continue;
			}

			$earned_ids = gamipress_get_user_earned_achievement_ids( $user_id, $points_type_slug );

			if ( empty( $earned_ids ) ) {
				$earned_ids = array( 0 );
			}

			$args = array(
				'post_type'   => $points_type_slug,
				'post_status' => 'publish',
				'post__in'    => $earned_ids,
				'order'       => isset( $request['order'] ) ? $request['order'] : 'Desc',
				'orderby'     => isset( $request['orderby'] ) ? $request['order'] : 'date',
			);

			// Loop Achievements.
			$achievement_posts = new \WP_Query( $args );

			if ( $achievement_posts->have_posts() ) {
				foreach ( $achievement_posts->posts as $achievement ) {

					$obj_achievement              = new \stdClass();
					$obj_achievement->ID          = $achievement->ID;
					$obj_achievement->title       = get_the_title( $achievement->ID );
					$obj_achievement->link        = get_permalink( $achievement->ID );
					$obj_achievement->content     = ! empty( $achievement->post_excerpt ) ? $achievement->post_excerpt : $achievement->post_content;
					$obj_achievement->points      = gamipress_get_post_meta( $achievement->ID, '_gamipress_points', true );
					$obj_achievement->points_type = gamipress_get_points_amount_label( floatval( $obj_achievement->points ), gamipress_get_post_meta( $achievement->ID, '_gamipress_points_type', true ), true );
					$obj_achievement->image       = RestAPI::gamipress_get_post_thumbnail_url( $achievement->ID );

					$data           = $this->prepare_item_for_response( $obj_achievement, $request );
					$achievements[] = $this->prepare_response_for_collection( $data );
				}
			}
		}
		$response = rest_ensure_response( $achievements );

		return $response;
	}

	/**
	 * @param $request
	 *
	 * @return mixed
	 */
	public function get_items_permissions_check( $request ) {
		return apply_filters( 'bbapp_gamipress_achievements_items_permissions', true, $request );
	}

	/**
	 * Get the Post's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {

		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'gamipress_achievement',
			'type'       => 'object',
			/*
				 * Base properties for every Post.
			*/
			'properties' => array(
				'id'      => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title'   => array(
					'description' => __( 'The title for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'content' => array(
					'description' => __( 'The content for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Content for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML content for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit' ),
						),
					),
				),
				'link'    => array(
					'description' => __( 'URL to the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'image'   => array(
					'description' => __( 'URL to the image object.', 'buddyboss-app' ),
					'type'        => 'image',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
			),
		);

		$schema['properties']['points'] = array(
			'description' => __( 'Point for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit' ),
		);

		$schema['properties']['points_type'] = array(
			'description' => __( 'Point type for the object', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		$params['context']['default'] = 'view';

		$params['user_id'] = array(
			'description'       => __( 'Limit result set to specific user.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['order']   = array(
			'description'       => __( 'Order sort attribute ascending or descending.' ),
			'type'              => 'string',
			'default'           => 'desc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby'] = array(
			'description'       => __( 'Sort collection by object attribute.' ),
			'type'              => 'string',
			'default'           => 'date',
			'enum'              => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}

	/**
	 * Prepare a single post output for response.
	 *
	 * @param WP_Post         $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response|\WP_Error $data
	 */
	public function prepare_item_for_response( $post, $request ) {
		$data = array();

		if ( isset( $post->ID ) ) {
			$data['id'] = $post->ID;
		} else {
			return new WP_Error( 'gamipress_json_internal_error', __( 'Required field "ID" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$schema = $this->get_item_schema();

		if ( ! empty( $schema['properties']['title'] ) ) {
			if ( isset( $post->title ) ) {
				$data['title'] = array(
					'raw'      => $post->title,
					'rendered' => $post->title,
				);
			} else {
				return new WP_Error( 'gamipress_json_internal_error', __( 'Required field "Title" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['content'] ) ) {
			if ( isset( $post->content ) ) {
				$data['content'] = array(
					'raw'      => $post->content,
					/** This filter is documented in wp-includes/post-template.php */
					'rendered' => apply_filters( 'the_content', $post->content ),
				);
			} else {
				return new WP_Error( 'gamipress_json_internal_error', __( 'Required field "Content" missing by add-on plugin', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		}

		if ( ! empty( $schema['properties']['points'] ) ) {
			$data['points'] = (int) $post->points;
		}

		if ( ! empty( $schema['properties']['points_type'] ) ) {
			$data['points_type'] = $post->points_type;
		}

		if ( ! empty( $schema['properties']['link'] ) ) {
			$data['link'] = $post->link;
		}

		if ( ! empty( $schema['properties']['image'] ) ) {
			$data['image'] = $post->image;
		}

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = $this->add_additional_fields_to_object( $data, $request );
		$data    = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		/**
		 * Filter the post data for a response.
		 *
		 * The dynamic portion of the hook name, $this->post_type, refers to post_type of the post being
		 * prepared for the response.
		 *
		 * @param WP_REST_Response $response The response object.
		 * @param WP_Post          $post     Post object.
		 * @param WP_REST_Request  $request  Request object.
		 */
		return apply_filters( "rest_prepare_gamipress_achievements", $response, $post, $request );
	}


}