<?php

namespace BuddyBossApp\Integrations\GamiPress;

use BuddyBossApp\App\App;

class RestAPI {

	protected static $instance;

	/**
	 * RestAPI constructor.
	 */
	public function __construct() {
		/** Nothing here */
	}

	/**
	 * @return mixed
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
			self::$instance->hooks(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 *
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'rest_api_init' ) );
		add_filter( 'bbapp_profile_tab', array( $this, 'updateAppProfileTab' ), 10, 2 );
		add_filter( 'bp_rest_profile_tab', array( $this, 'updateAppProfileTab' ), 10, 2 );
		add_filter( 'bbapp_core_profile_tabs', array( $this, 'bbapp_core_profile_gamipress_tabs' ), 10, 1 );
	}

	/**
	 *
	 */
	public function rest_api_init() {
		if ( class_exists( 'GamiPress' ) ) {
			add_filter( 'bbapp_settings_rest_response', array( $this, 'addSettings' ) );
			add_filter( 'gamipress_bp_user_details_activity_display_options', array(
				$this,
				'gamipress_bp_user_details_activity_remove_options'
			), 10, 2 );
			App::instance()->add_local_feature( "gamipress", array(
				"is_enabled_android" => true,
				"is_enabled_ios"     => true,
			) );

			RestAPIPoint::instance();
			RestAPIAchievement::instance();
			RestAPIRank::instance();
		}
	}

	/**
	 * @param $response
	 *
	 * @return array
	 */
	public function addSettings( $response ) {
		/**
		 * Placement
		 * "":     Means not to display
		 * 'Top':  Means display on profile header
		 * 'Tab':  Means display in profile tabs
		 * 'both': Means Display on Profile header as well as in Profile tab.
		 */

		if ( ! function_exists( 'gamipress_bp_get_option' ) ) {
			$response["badges"] = array(
				'points_placement'       => '',
				'ranks_placement'        => '',
				'achievements_placement' => ''
			);

			return $response;
		}

		$points_placement       = \gamipress_bp_get_option( 'points_placement', '' );
		$ranks_placement        = \gamipress_bp_get_option( 'ranks_placement', '' );
		$achievements_placement = \gamipress_bp_get_option( 'achievements_placement', '' );

		$response["badges"] = array(
			'points_placement'       => $points_placement,
			'ranks_placement'        => $ranks_placement,
			'achievements_placement' => $achievements_placement
		);

		/**
		 * Pass lable and slug for the points, achievments and ranks based on gamipress settings.
		 */
		$points_tab_title = \gamipress_bp_get_option( 'points_tab_title', __( 'Points', 'buddyboss-app' ) );
		$points_tab_slug  = $this->get_points_tab_slug( $points_tab_title );

		$achievements_tab_title = \gamipress_bp_get_option( 'achievements_tab_title', __( 'Achievements', 'buddyboss-app' ) );
		$achievements_tab_slug  = $this->get_achievements_tab_slug( $achievements_tab_title );

		$ranks_tab_title = \gamipress_bp_get_option( 'ranks_tab_title', __( 'Ranks', 'buddyboss-app' ) );
		$ranks_tab_slug  = $this->get_ranks_tab_slug( $ranks_tab_title );

		$response["gamipress"] = array(
			'labels'     => array(
				'points'       => $points_tab_title,	
				'achievements' => $achievements_tab_title,	
				'ranks'        => $ranks_tab_title,				
			),
			'permalinks' => array(
				'points'       => strtolower( $points_tab_slug ),	
				'achievements' => strtolower( $achievements_tab_slug ),	
				'ranks'        => strtolower( $ranks_tab_slug ),			
			)
		);

		return $response;
	}

	/**
	 * @param $tab
	 * @param $nav
	 *
	 * @return mixed
	 */
	public function updateAppProfileTab( $tab, $nav ) {
		if ( ! class_exists( 'GamiPress' ) ) {
			return $tab;
		}

		if ( in_array( $nav->screen_function, array(
			'gamipress_bp_achievements_tab',
			'gamipress_bp_points_tab',
			'gamipress_bp_ranks_tab',
		) ) ) {
			$tab['id'] = 'gamipress_' . strtolower( $tab['id'] );
		}

		return $tab;
	}

	/**
	 * Get the thubnail url from achievement and rank.
	 *
	 * Ref_Files: wp-content/plugins/gamipress/includes/functions/achievements.php:814
	 * Ref_Funcation: gamipress_get_achievement_post_thumbnail
	 *
	 * Ref_Files: wp-content/plugins/gamipress/includes/functions/ranks.php:935
	 * Ref_Funcation: gamipress_default_rank_post_thumbnail
	 *
	 * @param int    $post_id
	 * @param string $image_size
	 * @param bool   $default
	 *
	 * @return mixed
	 */
	public static function gamipress_get_post_thumbnail_url( $post_id = 0, $type = 'achievement-type', $mode = 'achievement', $image_size = 'gamipress-achievement', $default = true ) {
		// Get our badge thumbnail
		$image_url = get_the_post_thumbnail_url( $post_id, $image_size );
		// If we don't have an image...
		if ( ! $image_url && $default ) {

			// Grab our gamipress type's post thumbnail
			$type_object = get_page_by_path( gamipress_get_post_type( $post_id ), OBJECT, $type );
			$image       = is_object( $type_object ) ? get_the_post_thumbnail_url( $type_object->ID, $image_size ) : false;

			// If we still have no image, use one from Credly
			if ( ! $image ) {
				// Available filter: 'badgeos_default_achievement_post_thumbnail','badgeos_default_rank_post_thumbnail'
				$image_url = apply_filters( "gamipress_default_{$mode}_post_thumbnail", '', $type_object );

			}
		}

		// Finally, return our image tag
		return $image_url;
	}

	/**
	 * Remove activity gamipress user details.
	 *
	 * @param $display_options
	 * @param $user_id
	 *
	 * @return mixed
	 */
	public function gamipress_bp_user_details_activity_remove_options( $display_options, $user_id ) {

		$response = $this->addSettings( array() );

		if ( isset( $display_options['show_points'] ) && 1 == $display_options['show_points'] ) {
			if ( ! empty( $response['badges']['points_placement'] ) && in_array( 'activity', $response['badges']['points_placement'] ) ) {
				$display_options['show_points'] = 0;
			}
		}
		if ( isset( $display_options['show_ranks'] ) && 1 == $display_options['show_ranks'] ) {
			if ( ! empty( $response['badges']['ranks_placement'] ) && in_array( 'activity', $response['badges']['ranks_placement'] ) ) {
				$display_options['show_ranks'] = 0;
			}
		}
		if ( isset( $display_options['show_achievements'] ) && 1 == $display_options['show_achievements'] ) {
			if ( ! empty( $response['badges']['achievements_placement'] ) && in_array( 'activity', $response['badges']['achievements_placement'] ) ) {
				$display_options['show_achievements'] = 0;
			}
		}

		return $display_options;
	}

	/**
	 * Gamipress Buddyboss integration settings options.
	 *
	 * @param string $display
	 * @param string $context
	 *
	 * @since 2.3.30 Added '$context' parameter.
	 *
	 * @return array
	 */
	public function gamipress_display_options( $display = '', $context = '' ) {
		$display_options = array();
		if ( 'member' === $context && function_exists( 'gamipress_bp_get_option' ) ) {
			// Points
			if ( 'point' === $display ) {
				$display_options['points_types_to_show']        = function_exists( 'gamipress_bp_members_get_points_types' ) ? \gamipress_bp_members_get_points_types() : array();
				$display_options['points_types_thumbnail']      = (bool) \gamipress_bp_get_option( 'members_points_types_top_thumbnail', false );
				$display_options['points_types_thumbnail_size'] = (int) \gamipress_bp_get_option( 'members_points_types_top_thumbnail_size', 25 );
				$display_options['points_types_label']          = (bool) \gamipress_bp_get_option( 'members_points_types_top_label', false );
				$display_options['points_types_hide']           = (bool) \gamipress_bp_get_option( 'members_points_types_hide', false );
			}

			// Achievements
			if ( 'achievement' === $display ) {
				$display_options['achievement_types_to_show']        = function_exists( 'gamipress_bp_members_get_achievements_types' ) ? \gamipress_bp_members_get_achievements_types() : array();
				$display_options['achievement_types_thumbnail']      = (bool) \gamipress_bp_get_option( 'members_achievements_top_thumbnail', false );
				$display_options['achievement_types_thumbnail_size'] = (int) \gamipress_bp_get_option( 'members_achievements_top_thumbnail_size', 25 );
				$display_options['achievement_types_title']          = (bool) \gamipress_bp_get_option( 'members_achievements_top_title', false );
				$display_options['achievement_types_link']           = (bool) \gamipress_bp_get_option( 'members_achievements_top_link', false );
				$display_options['achievement_types_label']          = (bool) \gamipress_bp_get_option( 'members_achievements_top_label', false );
				$display_options['achievement_types_limit']          = (int) \gamipress_bp_get_option( 'members_achievements_top_limit', 10 );
			}

			// Ranks
			if ( 'rank' === $display ) {
				$display_options['rank_types_to_show']        = function_exists( 'gamipress_bp_members_get_ranks_types' ) ? \gamipress_bp_members_get_ranks_types() : array();
				$display_options['rank_types_thumbnail']      = (bool) \gamipress_bp_get_option( 'members_ranks_top_thumbnail', false );
				$display_options['rank_types_thumbnail_size'] = (int) \gamipress_bp_get_option( 'members_ranks_top_thumbnail_size', 25 );
				$display_options['rank_types_title']          = (bool) \gamipress_bp_get_option( 'members_ranks_top_title', false );
				$display_options['rank_types_link']           = (bool) \gamipress_bp_get_option( 'members_ranks_top_link', false );
				$display_options['rank_types_label']          = (bool) \gamipress_bp_get_option( 'members_ranks_top_label', false );
				$display_options['rank_types_hide']           = (bool) \gamipress_bp_get_option( 'members_ranks_types_hide', false );
			}
		}

		if ( 'forum' === $context && function_exists( 'gamipress_bbp_get_option' ) ) {
			// Points
			if ( 'point' === $display ) {
				$display_options['points_types_to_show']        = \gamipress_bbp_get_option( 'points_types', array() );
				$display_options['points_types_thumbnail']      = (bool) \gamipress_bbp_get_option( 'points_types_thumbnail', false );
				$display_options['points_types_thumbnail_size'] = (int) \gamipress_bbp_get_option( 'points_types_thumbnail_size', 25 );
				$display_options['points_types_label']          = (bool) \gamipress_bbp_get_option( 'points_types_label', false );
				$display_options['points_types_hide']           = (bool) \gamipress_bbp_get_option( 'points_types_hide', false );
			}

			// Achievements
			if ( 'achievement' === $display ) {
				$display_options['achievement_types_to_show']        = \gamipress_bbp_get_option( 'achievement_types', array() );
				$display_options['achievement_types_thumbnail']      = (bool) \gamipress_bbp_get_option( 'achievement_types_thumbnail', false );
				$display_options['achievement_types_thumbnail_size'] = (int) \gamipress_bbp_get_option( 'achievement_types_thumbnail_size', 25 );
				$display_options['achievement_types_title']          = (bool) \gamipress_bbp_get_option( 'achievement_types_title', false );
				$display_options['achievement_types_link']           = (bool) \gamipress_bbp_get_option( 'achievement_types_link', false );
				$display_options['achievement_types_label']          = (bool) \gamipress_bbp_get_option( 'achievement_types_label', false );
				$display_options['achievement_types_limit']          = (int) \gamipress_bbp_get_option( 'achievements_limit', 10 );
			}

			// Ranks
			if ( 'rank' === $display ) {
				$display_options['rank_types_to_show']        = \gamipress_bbp_get_option( 'rank_types', array() );
				$display_options['rank_types_thumbnail']      = (bool) \gamipress_bbp_get_option( 'rank_types_thumbnail', false );
				$display_options['rank_types_thumbnail_size'] = (int) \gamipress_bbp_get_option( 'rank_types_thumbnail_size', 25 );
				$display_options['rank_types_title']          = (bool) \gamipress_bbp_get_option( 'rank_types_title', false );
				$display_options['rank_types_link']           = (bool) \gamipress_bbp_get_option( 'rank_types_link', false );
				$display_options['rank_types_label']          = (bool) \gamipress_bbp_get_option( 'rank_types_label', false );
				$display_options['rank_types_hide']           = (bool) \gamipress_bbp_get_option( 'rank_types_hide', false );
			}
		}

		return $display_options;
	}

	/**
	 * Get point tab slug.
	 * @param $points_tab_title
	 *
	 * @return mixed|string
	 */
	public function get_points_tab_slug( $points_tab_title ) {
		$points_tab_slug = \gamipress_bp_get_option( 'points_tab_slug', '' );

		// If empty point slug then generate it from the title.
		if ( empty( $points_tab_slug ) ) {
			$points_tab_slug = sanitize_title( $points_tab_title );
		}

		return $points_tab_slug;
	}

	/**
	 * Get achievements tab slug.
	 *
	 * @param $achievements_tab_title
	 *
	 * @return mixed|string
	 */
	public function get_achievements_tab_slug( $achievements_tab_title ) {
		$achievements_tab_slug = \gamipress_bp_get_option( 'achievements_tab_slug', '' );

		// If empty achievements slug then generate it from the title.
		if ( empty( $achievements_tab_slug ) ) {
			$achievements_tab_slug = sanitize_title( $achievements_tab_title );
		}

		return $achievements_tab_slug;
	}

	/**
	 * Ranks tab slug.
	 *
	 * @param $ranks_tab_title
	 *
	 * @return mixed|string
	 */
	public function get_ranks_tab_slug( $ranks_tab_title ) {
		$ranks_tab_slug = \gamipress_bp_get_option( 'ranks_tab_slug', '' );

		// If empty ranks slug then generate it from the title.
		if ( empty( $ranks_tab_slug ) ) {
			$ranks_tab_slug = sanitize_title( $ranks_tab_title );
		}

		return $ranks_tab_slug;
	}

	/**
	 * Add core profile tabs.
	 *
	 * @param $nav_items
	 *
	 * @return mixed
	 */
	public function bbapp_core_profile_gamipress_tabs( $nav_items ) {

		if ( ! function_exists( 'gamipress_bp_get_option' ) ) {
			return $nav_items;
		}

		/**
		 * Pass lable and slug for the points, achievments and ranks based on gamipress settings.
		 */
		$points_tab_title = \gamipress_bp_get_option( 'points_tab_title', __( 'Points', 'buddyboss-app' ) );
		$points_tab_slug  = $this->get_points_tab_slug( $points_tab_title );

		$achievements_tab_title = \gamipress_bp_get_option( 'achievements_tab_title', __( 'Achievements', 'buddyboss-app' ) );
		$achievements_tab_slug  = $this->get_achievements_tab_slug( $achievements_tab_title );

		$ranks_tab_title                                    = \gamipress_bp_get_option( 'ranks_tab_title', __( 'Ranks', 'buddyboss-app' ) );
		$ranks_tab_slug                                     = $this->get_ranks_tab_slug( $ranks_tab_title );
		$nav_items[ 'gamipress_' . $points_tab_slug ]       = $points_tab_title;
		$nav_items[ 'gamipress_' . $achievements_tab_slug ] = $achievements_tab_title;
		$nav_items[ 'gamipress_' . $ranks_tab_slug ]        = $ranks_tab_title;

		// Get achievement our types
		$achievement_types = \gamipress_get_achievement_types();
		if ( ! empty( $achievement_types ) ) {
			foreach ( $achievement_types as $achievement_key => $achievement_type ) {
				$nav_items[ $achievement_key ] = $achievement_type['singular_name'];
			}
		}
		// Get point our types
		$points_types = \gamipress_get_points_types();
		if ( ! empty( $points_types ) ) {
			foreach ( $points_types as $points_key => $points_type ) {
				$nav_items[ $points_key ] = $points_type['singular_name'];
			}
		}
		// Get rank our types
		$rank_types = \gamipress_get_rank_types();
		if ( ! empty( $rank_types ) ) {
			foreach ( $rank_types as $rank_key => $rank_type ) {
				$nav_items[ $rank_key ] = $rank_type['singular_name'];
			}
		}

		return $nav_items;
	}
}