<?php

namespace BuddyBossApp\Integrations\GamiPress;

// Contains all bbpress related feature

class Main {

	private static $instance;

	/**
	 * Main constructor.
	 */
	public function __construct() {
	}

	/**
	 * @return mixed
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class;
			self::$instance->load(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 *
	 */
	public function load() {
		add_filter( 'bbapp_menu_item_dependency', array( $this, 'filter_app_menu_by_dependency' ), 10, 2 );
		if ( ! class_exists( 'GamiPress' ) ) {
			return;
		}
		RestAPI::instance();
		add_filter( 'bbapp_quick_link_user_links_items', array( $this, 'gamipress_quick_link_user_links_items' ), 10, 1 );
		add_filter( 'bbapp_quick_link_deeplink_links', array( $this, 'gamipress_bbapp_quick_link_deeplink_links' ), 10, 2 );
		add_action( 'updated_option', array( $this, 'purge_component_cache' ), 10, 3 );
		add_action( 'gamipress_update_user_rank', array( $this, 'gamipress_update_user_rank_clear_cache' ), 10, 5 );
		add_action( 'gamipress_update_user_points', array( $this, 'gamipress_update_user_points_clear_cache' ), 10, 8 );
		add_action( 'gamipress_insert_user_earning', array( $this, 'gamipress_insert_user_earning_clear_cache' ), 10, 4 );
		add_filter( 'bbapp_prepare_core_data', array( $this, 'filter_bbapp_prepare_core_data' ), 10, 2 );
		add_filter( 'bp_rest_notifications_prepare_value', array( $this, 'modify_notification_avatar_url' ), 10, 3 );
		add_filter( 'bp_rest_reply_prepare_links', array( $this, 'modify_reply_embeddable_link' ), 10, 2 );
	}

	/**
	 * Gamipress updated rank as user earning.
	 *
	 * @param integer $user_id        The user ID
	 * @param /WP_Post $new_rank       New rank post object
	 * @param /WP_Post $old_rank       Old rank post object
	 * @param integer $admin_id       An admin ID (if admin-awarded)
	 * @param integer $achievement_id Achievement that has been triggered it
	 */
	public function gamipress_update_user_rank_clear_cache( $user_id, $new_rank, $old_rank, $admin_id, $achievement_id ) {
		if ( method_exists( '\BuddyBoss\Performance\Cache', 'purge_by_user_id' ) ) {
			\BuddyBoss\Performance\Cache::instance()->purge_by_user_id( $user_id, 'bp-members' );
		}
	}

	/**
	 * Gamipress updated user points.
	 *
	 * @updated 1.3.6 Support for deduct/revoke points
	 *
	 * @param integer $user_id        The user ID
	 * @param integer $new_points     Points added/deducted to the user's total
	 * @param integer $total_points   The user's updated total points
	 * @param integer $admin_id       An admin ID (if admin-awarded)
	 * @param integer $achievement_id The associated achievement ID
	 * @param string  $points_type    The points type
	 * @param string  $reason         Custom reason to override default log pattern
	 * @param string  $log_type       Custom log type
	 */
	public function gamipress_update_user_points_clear_cache( $user_id, $new_points, $total_points, $admin_id, $achievement_id, $points_type = '', $reason = '', $log_type = '' ) {
		if ( method_exists( '\BuddyBoss\Performance\Cache', 'purge_by_user_id' ) ) {
			\BuddyBoss\Performance\Cache::instance()->purge_by_user_id( $user_id, 'bp-members' );
		}
	}

	/**
	 * Create a user earning achievement.
	 *
	 * @param int   $user_id The user ID
	 * @param array $data    User earning data
	 * @param array $meta    User earning meta data
	 *
	 * @return int                    The user earning ID of the newly created user earning entry
	 * @since  1.4.3
	 *
	 */
	public function gamipress_insert_user_earning_clear_cache( $user_earning_id, $data, $meta, $user_id ) {
		if ( method_exists( '\BuddyBoss\Performance\Cache', 'purge_by_user_id' ) ) {
			\BuddyBoss\Performance\Cache::instance()->purge_by_user_id( $user_id, 'bp-members' );
		}
	}

	/**
	 * Purge component cache by component setting enabled or disable.
	 *
	 * @param string $option    Option Name.
	 * @param string $old_value Option Old Value.
	 * @param array  $value     Option Updated Value.
	 */
	public function purge_component_cache( $option, $old_value, $value ) {

		if ( function_exists( 'bbapp_is_active' ) && ! bbapp_is_active( 'performance' ) ) {
			return;
		}

		if ( 'gamipress_settings' === $option ) {
			if ( ! isset( $value['bp_members_points_types_order'] ) || ! isset( $value['bp_members_achievements_types_order'] ) || ! isset( $value['bp_members_ranks_types_order'] ) ) {
				return;
			}

			if ( ! isset( $value['bp_tab_points_types_order'] ) || ! isset( $value['bp_tab_achievements_types_order'] ) || ! isset( $value['bp_tab_ranks_types_order'] ) ) {
				return;
			}
			\BuddyBoss\Performance\Cache::instance()->purge_by_component( 'bp-members' );
			\BuddyBoss\Performance\Cache::instance()->purge_by_component( 'bbapp-deeplinking' );
		}
	}

	/**
	 * Menu dependency added.
	 *
	 * @param $status
	 * @param $menu
	 *
	 * @return false
	 */
	public function filter_app_menu_by_dependency( $status, $menu ) {
		if ( 'gamipress_achievements' === $menu['object'] ) {
			return class_exists( 'GamiPress' );
		}

		return $status;
	}

	/**
	 * Gamipress support for quick link.
	 * @param $quick_link_user_links_items
	 *
	 * @return mixed
	 */
	public function gamipress_quick_link_user_links_items( $quick_link_user_links_items ) {

		// Requires GamiPress & GamiPress BuddyPress Plugin Active.
		if ( class_exists( 'GamiPress' ) && function_exists( 'gamipress_bp_get_option' ) ) {
			// Get stored version
			$stored_version = get_option( 'gamipress_buddyboss_integration_version', '1.0.0' );

			// GamiPress BuddyPress 1.1.2 upgrade
			if ( version_compare( $stored_version, '1.1.2', '<' ) ) {
				$points_placement             = gamipress_bp_get_option( 'points_placement', '' );
				$is_points_table_enable       = in_array( $points_placement, array( 'tab', 'both' ) );
				$achievements_placement       = gamipress_bp_get_option( 'achievements_placement', '' );
				$is_achievements_table_enable = in_array( $achievements_placement, array( 'tab', 'both' ) );
				$ranks_placement              = gamipress_bp_get_option( 'ranks_placement', '' );
				$is_ranks_table_enable        = in_array( $ranks_placement, array( 'tab', 'both' ) );
			} else {
				$gamipress_settings           = get_option( 'gamipress_settings' );
				$is_points_table_enable       = ( isset($gamipress_settings['bp_points_tab']) ) ? 'on' === $gamipress_settings['bp_points_tab'] : false;
				$is_achievements_table_enable = ( isset($gamipress_settings['bp_achievements_tab']) ) ? 'on' === $gamipress_settings['bp_achievements_tab'] : false;
				$is_ranks_table_enable        = ( isset($gamipress_settings['bp_ranks_tab']) ) ? 'on' === $gamipress_settings['bp_ranks_tab'] : false;
			}

			// Points.
			if ( $is_points_table_enable ) {
				$points_types_to_show = gamipress_bp_members_get_points_types();
				if ( ! empty( $points_types_to_show ) ) {
					$points_tab_title = gamipress_bp_get_option( 'points_tab_title', __( 'Points', 'gamipress-buddyboss-integration' ) );
					$points_tab_slug  = gamipress_bp_get_option( 'points_tab_slug', '' );

					// If empty slug generate it from the title.
					if ( empty( $points_tab_slug ) ) {
						$points_tab_slug = sanitize_title( $points_tab_title );
					}
					$quick_link_user_links_items[] = array(
						'ID'    => $points_tab_slug,
						'title' => $points_tab_title,
						'icon'  => array(
							'id'   => 'file-bookmark',
							'type' => 'buddyboss',
						),
					);
				}
			}

			// Achievements.
			if ( $is_achievements_table_enable ) {
				$achievements_types_to_show = gamipress_bp_members_get_achievements_types();

				if ( ! empty( $achievements_types_to_show ) ) {

					$achievements_tab_title = gamipress_bp_get_option( 'achievements_tab_title', __( 'Achievements', 'gamipress-buddyboss-integration' ) );
					$achievements_tab_slug  = gamipress_bp_get_option( 'achievements_tab_slug', '' );

					// If empty slug generate it from the title.
					if ( empty( $achievements_tab_slug ) ) {
						$achievements_tab_slug = sanitize_title( $achievements_tab_title );
					}
					$quick_link_user_links_items[] = array(
						'ID'    => $achievements_tab_slug,
						'title' => $achievements_tab_title,
						'icon'  => array(
							'id'   => 'file-bookmark',
							'type' => 'buddyboss',
						),
					);
				}
			}

			// Ranks.
			if ( $is_ranks_table_enable ) {
				$ranks_types_to_show = gamipress_bp_members_get_ranks_types();

				if ( ! empty( $ranks_types_to_show ) ) {
					$ranks_tab_title = gamipress_bp_get_option( 'ranks_tab_title', __( 'Ranks', 'gamipress-buddyboss-integration' ) );
					$ranks_tab_slug  = gamipress_bp_get_option( 'ranks_tab_slug', '' );

					// If empty slug generate it from the title.
					if ( empty( $ranks_tab_slug ) ) {
						$ranks_tab_slug = sanitize_title( $ranks_tab_title );
					}
					$quick_link_user_links_items[] = array(
						'ID'    => $ranks_tab_slug,
						'title' => $ranks_tab_title,
						'icon'  => array(
							'id'   => 'file-bookmark',
							'type' => 'buddyboss',
						),
					);
				}
			}
		}

		return $quick_link_user_links_items;
	}

	/**
	 * Quick link deeplink url.
	 *
	 * @param $link
	 * @param $link_slug
	 *
	 * @return string
	 */
	public function gamipress_bbapp_quick_link_deeplink_links( $link, $link_slug ) {
		// Requires GamiPress & GamiPress BuddyPress Plugin Active.
		if ( class_exists( 'GamiPress' ) && function_exists( 'gamipress_bp_get_option' ) ) {
			// Get stored version
			$stored_version = get_option( 'gamipress_buddyboss_integration_version', '1.0.0' );

			// GamiPress BuddyPress 1.1.2 upgrade
			if ( version_compare( $stored_version, '1.1.2', '<' ) ) {
				$points_placement             = gamipress_bp_get_option( 'points_placement', '' );
				$is_points_table_enable       = in_array( $points_placement, array( 'tab', 'both' ) );
				$achievements_placement       = gamipress_bp_get_option( 'achievements_placement', '' );
				$is_achievements_table_enable = in_array( $achievements_placement, array( 'tab', 'both' ) );
				$ranks_placement              = gamipress_bp_get_option( 'ranks_placement', '' );
				$is_ranks_table_enable        = in_array( $ranks_placement, array( 'tab', 'both' ) );
			} else {
				$gamipress_settings           = get_option( 'gamipress_settings' );
				$is_points_table_enable       = 'on' === $gamipress_settings['bp_points_tab'];
				$is_achievements_table_enable = 'on' === $gamipress_settings['bp_achievements_tab'];
				$is_ranks_table_enable        = 'on' === $gamipress_settings['bp_ranks_tab'];
			}

			// Points.
			if ( $is_points_table_enable ) {
				$points_types_to_show = gamipress_bp_members_get_points_types();

				if ( ! empty( $points_types_to_show ) ) {
					$points_tab_title = gamipress_bp_get_option( 'points_tab_title', __( 'Points', 'gamipress-buddyboss-integration' ) );
					$points_tab_slug  = gamipress_bp_get_option( 'points_tab_slug', '' );

					// If empty slug generate it from the title
					if ( empty( $points_tab_slug ) ) {
						$points_tab_slug = sanitize_title( $points_tab_title );
					}
					if ( $points_tab_slug === $link_slug ) {
						return trailingslashit( bp_loggedin_user_domain() . $points_tab_slug );
					}
				}
			}

			// Achievements.
			if ( $is_achievements_table_enable ) {
				$achievements_types_to_show = gamipress_bp_members_get_achievements_types();

				if ( ! empty( $achievements_types_to_show ) ) {

					$achievements_tab_title = gamipress_bp_get_option( 'achievements_tab_title', __( 'Achievements', 'gamipress-buddyboss-integration' ) );
					$achievements_tab_slug  = gamipress_bp_get_option( 'achievements_tab_slug', '' );

					// If empty slug generate it from the title
					if ( empty( $achievements_tab_slug ) ) {
						$achievements_tab_slug = sanitize_title( $achievements_tab_title );
					}
					if ( $achievements_tab_slug === $link_slug ) {
						return trailingslashit( bp_loggedin_user_domain() . $achievements_tab_slug );
					}
				}
			}

			// Ranks.
			if ( $is_ranks_table_enable ) {
				$ranks_types_to_show = gamipress_bp_members_get_ranks_types();

				if ( ! empty( $ranks_types_to_show ) ) {
					$ranks_tab_title = gamipress_bp_get_option( 'ranks_tab_title', __( 'Ranks', 'gamipress-buddyboss-integration' ) );
					$ranks_tab_slug  = gamipress_bp_get_option( 'ranks_tab_slug', '' );

					// If empty slug generate it from the title
					if ( empty( $ranks_tab_slug ) ) {
						$ranks_tab_slug = sanitize_title( $ranks_tab_title );
					}
					if ( $ranks_tab_slug === $link_slug ) {
						return trailingslashit( bp_loggedin_user_domain() . $ranks_tab_slug );
					}
				}
			}
		}

		return $link;
	}
	/**
	 * Deeplinking url class checking.
	 *
	 * @param $url_data
	 * @param $url_params
	 *
	 * @return mixed
	 */
	public function filter_bbapp_prepare_core_data( $url_data, $url_params ) {
		if ( 'open_screen' === $url_data['action'] && 'gamipress_achievements' === $url_data['item_id'] && ! class_exists( 'GamiPress' ) ) {
			$url_data['action']  = 'open_404';
			$url_data['item_id'] = '404';
		}

		return $url_data;
	}

	/**
	 * Gamipress notification used feature image so change avatar url to feature image.
	 *
	 * @param \WP_REST_Response              $response     The response data.
	 * @param \WP_REST_Request               $request      Full details about the request.
	 * @param \BP_Notifications_Notification $notification Notification object.
	 *
	 * @since 1.7.3
	 *
	 * @return \WP_REST_Response
	 */
	public function modify_notification_avatar_url( $response, $request, $notification ) {

		if ( 'gamipress_buddyboss_notifications' === $response->data['action'] ) {
			$post_id = $response->data['item_id'];
			$post    = gamipress_get_post( $response->data['item_id'] );
			if ( in_array( $post->post_type, gamipress_get_achievement_types_slugs(), true ) ) {

				// Grab our achievement type's post thumbnail.
				$achievement = get_page_by_path( gamipress_get_post_type( $post_id ), OBJECT, 'achievement-type' );
				$image_full  = is_object( $achievement ) ? get_the_post_thumbnail_url( $achievement->ID, 'full' ) : false;
				$image_thumb = is_object( $achievement ) ? get_the_post_thumbnail_url( $achievement->ID, 'gamipress-achievement' ) : false;

			} elseif ( 'step' === $post->post_type ) {
				// Step.
				$achievement = gamipress_get_step_achievement( $post->ID );

				if ( $achievement ) {
					$image_full  = is_object( $achievement ) ? get_the_post_thumbnail_url( $achievement->ID, 'full' ) : false;
					$image_thumb = is_object( $achievement ) ? get_the_post_thumbnail_url( $achievement->ID, 'gamipress-achievement' ) : false;
				}
			} elseif ( in_array( $post->post_type, array( 'points-award', 'points-deduct' ), true ) ) {

				// Points award and deduct.
				$points_type_id = gamipress_get_post_field( 'post_parent', $post->ID );
				$post_id        = 0;
				if ( gettype( $points_type_id ) === 'integer' ) {
					$post_id = $points_type_id;
				} elseif ( absint( $points_type_id ) !== 0 ) {
					$post_id = $points_type_id;
				} else {
					$points_types = gamipress_get_points_types();

					if ( isset( $points_types[ $points_type_id ] ) ) {
						$post_id = $points_types[ $points_type_id ]['ID'];
					}
				}
				// Return our image tag with custom size.
				$image_full  = get_the_post_thumbnail_url( $post_id, 'full' );
				$image_thumb = get_the_post_thumbnail_url( $post_id, 'gamipress-points' );

			} elseif ( in_array( $post->post_type, gamipress_get_rank_types_slugs(), true ) ) {
				// Grab our rank type's post thumbnail.
				$rank        = get_page_by_path( gamipress_get_post_type( $post_id ), OBJECT, 'rank-type' );
				$image_full  = is_object( $rank ) ? get_the_post_thumbnail_url( $rank->ID, 'full' ) : false;
				$image_thumb = is_object( $rank ) ? get_the_post_thumbnail_url( $rank->ID, 'gamipress-rank' ) : false;

			} elseif ( 'rank-requirement' === $post->post_type ) {
				// Rank requirement.
				$rank = gamipress_get_rank_requirement_rank( $post->ID );
				if ( $rank ) {
					$image_full  = is_object( $rank ) ? get_the_post_thumbnail_url( $rank->ID, 'full' ) : false;
					$image_thumb = is_object( $rank ) ? get_the_post_thumbnail_url( $rank->ID, 'gamipress-rank' ) : false;
				}
			}

			// Avatars.
			$response->data['avatar_urls'] = array(
				'full'  => ! empty( $image_full ) ? $image_full : '',
				'thumb' => ! empty( $image_thumb ) ? $image_thumb : '',
			);
			// Read only.
			$response->data['readonly'] = true;
		}

		return $response;
	}

	/**
	 * Add `screen` parameter to the existing embaddable user link.
	 *
	 * @param array   $links The prepared links of the REST response.
	 * @param WP_Post $post  Post object.
	 *
	 * @since 2.2.30
	 *
	 * @return array
	 */
	function modify_reply_embeddable_link( $links, $post ) {
		if ( class_exists( 'GamiPress' ) && is_array( $links ) && ! empty( $links['user'] ) && $links['user']['embeddable'] ) {
			$links['user']['href'] .= '?screen=reply';
		}

		return $links;
	}
}
