<?php
/**
 * Holds GamiPress Rank functionality for the Access Controls.
 *
 * @package BuddyBossApp\Integrations\GamiPress\AccessControls
 */

namespace BuddyBossApp\Integrations\GamiPress\AccessControls;

use BuddyBossApp\AccessControls\TmpItems;
use CT_Query;
use stdClass;
use BuddyBossApp\Integrations\GamiPress\AccessControls;
use WP_Post;
use WP_Query;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Class Course
 */
class Rank {
	/**
	 * Condition name.
	 *
	 * @var string $condition_name condition name.
	 */
	public $condition_name = 'gamipress-ranks';

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @since 1.7.6
	 * @return Rank
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Class construct method.
	 *
	 * @since 1.7.6
	 * @return void
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Items callback method.
	 *
	 * @param string $search Search the condition.
	 * @param int    $page   Page number.
	 * @param int    $limit  Limit the items to be fetched.
	 *
	 * @since 1.7.6
	 * @return array
	 */
	public function gamipress_rank_items_callback( $search = '', $page = 1, $limit = 1 ) {
		$list       = array();
		$rank_types = $this->get_rank_types();

		if ( empty( $rank_types ) ) {
			return $list;
		}

		$args = array(
			'fields'         => 'all',
			'post_status'    => array( 'publish' ),
			'post_type'      => $rank_types,
			'posts_per_page' => $limit,
			'paged'          => $page,
		);

		if ( ! empty( $search ) ) {
			$args['s'] = $search;
		}

		$results = $this->gamipress_get_all_ranks( $args );

		if ( ! empty( $results ) ) {
			foreach ( $results as $item ) {
				$list[ $item->ID ] = array(
					'id'   => $item->ID,
					'name' => esc_html( $item->post_title ),
					'link' => admin_url( "post.php?post={$item->ID}&action=edit" ),
				);
			}
		}

		return $list;
	}

	/**
	 * Item callback method.
	 *
	 * @param int $item_value Item value of condition.
	 *
	 * @since 1.7.6
	 * @return array|false
	 */
	public function gamipress_rank_item_callback( $item_value ) {
		$rank_type = get_post( $item_value );

		if ( empty( $rank_type ) && ! $rank_type instanceof WP_Post ) {
			return false;
		}

		return array(
			'id'   => $rank_type->ID,
			'name' => $rank_type->post_title,
			'link' => admin_url( "post.php?post={$rank_type->ID}&action=edit" ),
		);
	}

	/**
	 * Users callback method.
	 *
	 * @param array $data     condition data.
	 * @param int   $page     current page number.
	 * @param int   $per_page limit.
	 *
	 * @since 1.7.6
	 * @return array|string
	 */
	public function gamipress_rank_users_callback( $data, $page = 1, $per_page = 10 ) {
		$user_ids      = array();
		$sub_condition = ( ! empty( $data['sub_condition'] ) ) ? $data['sub_condition'] : '';
		$item_value    = ( ! empty( $data['item_value'] ) ) ? $data['item_value'] : '';
		$group_id      = ( ! empty( $data['group_id'] ) ) ? $data['group_id'] : 0;
		$rounds_count  = ( ! empty( $data['rounds_count'] ) ) ? $data['rounds_count'] : 0;

		if ( 'specific' === $sub_condition ) {
			if ( empty( $item_value ) ) {
				return $user_ids;
			}

			ct_setup_table( 'gamipress_user_earnings' );

			$query_args = array(
				'paged'          => $page,
				'items_per_page' => $per_page,
				'post_id'        => $item_value,
			);
			$ct_query   = new CT_Query( $query_args );
			$results    = $ct_query->get_results();

			if ( ! empty( $results ) ) {
				$user_ids = wp_list_pluck( $results, 'user_id' );
			}

			ct_reset_setup_table();

			return AccessControls::instance()->return_users( $user_ids );
		} elseif ( 'any' === $sub_condition ) {
			$transient_key = "access_ctrls_gamipress_rank_condition_jb_data_{$group_id}";

			// Check if it's a first round.
			if ( 1 === $rounds_count ) {
				$all_ranks = $this->gamipress_get_all_ranks( array( 'fields' => 'ids' ) );
				$rank_ids  = array();

				if ( ! empty( $all_ranks ) ) {
					foreach ( $all_ranks as $id ) {
						$rank_ids[ $id ] = $id;
					}
				}

				$condition_data                 = array(
					'prepare_page'     => 1,
					'is_data_prepared' => false,
					'pending_rank_ids' => $rank_ids,
				);
				$condition_data['current_rank'] = array_key_first( $condition_data['pending_rank_ids'] );
			} else {
				$condition_data = get_transient( $transient_key );
			}

			/**
			 * Preparing the data into tmp  items collections.
			 */
			if ( ! $condition_data['is_data_prepared'] ) {
				ct_setup_table( 'gamipress_user_earnings' );

				$query_args = array(
					'paged'          => $condition_data['prepare_page'],
					'items_per_page' => $per_page,
					'post_id'        => $condition_data['current_rank'],
				);
				$ct_query   = new CT_Query( $query_args );
				$results    = $ct_query->get_results();

				if ( ! empty( $results ) ) {
					$user_ids = wp_list_pluck( $results, 'user_id' );
				}

				ct_reset_setup_table();

				if ( ! empty( $user_ids ) ) {
					$items = array();

					foreach ( $user_ids as $user_id ) {
						$items[] = array(
							'item_type'         => "gp_ranks_{$group_id}",
							'item_id'           => $user_id,
							'secondary_item_id' => $condition_data['current_rank'],
						);
					}

					$insert = TmpItems::instance()->insert_items( $items );

					$condition_data['prepare_page'] ++;

					if ( empty( $insert ) ) {
						return AccessControls::instance()->return_error( __( 'Error while preparing members data for GamiPress Rank..', 'buddyboss-app' ) );
					}
				} else { // when all users are processed.
					unset( $condition_data['pending_rank_ids'][ $condition_data['current_rank'] ] ); // remove from pending rank id.

					if ( ! empty( $condition_data['pending_rank_ids'] ) ) {
						$condition_data['current_rank'] = array_key_first( $condition_data['pending_rank_ids'] ); // get next rank to process.
						$condition_data['prepare_page'] = 1; // reset pagination to 1.
					} else {
						// If pending memberships found empty finish the job.
						$condition_data['is_data_prepared'] = true;
					}
				}

				set_transient( $transient_key, $condition_data, DAY_IN_SECONDS );

				return AccessControls::instance()->return_wait( __( 'Preparing GamiPress rank Members Data', 'buddyboss-app' ) );
			}

			$items = TmpItems::instance()->get_items(
				array(
					'include_item_types' => array( "gp_ranks_{$group_id}" ),
					'per_page'           => $per_page,
					'page'               => $page,
				)
			);

			$user_ids = array();

			foreach ( $items['results'] as $item ) {
				$user_ids[] = $item->item_id;
			}

			return AccessControls::instance()->return_users( $user_ids );
		}

		return AccessControls::instance()->return_error( __( 'Invalid sub condition.', 'buddyboss-app' ) );
	}

	/**
	 * Function to check if user has any condition.
	 *
	 * @param int $user_id User id to check.
	 *
	 * @since 1.7.6
	 * @return bool
	 */
	public function has_any_items_callback( $user_id ) {
		$rank_types = $this->get_rank_types();

		if ( ! empty( $rank_types ) ) {
			foreach ( $rank_types as $rank_type ) {
				$ranks_id = gamipress_get_user_rank_id( $user_id, $rank_type );

				ct_setup_table( 'gamipress_user_earnings' );

				$query_args = array(
					'post_id' => $ranks_id,
				);
				$ct_query   = new CT_Query( $query_args );
				$results    = $ct_query->get_results();

				if ( ! empty( $results ) ) {
					$user_ids = wp_list_pluck( $results, 'user_id' );
				}

				ct_reset_setup_table();

				if ( ! empty( $user_ids ) && in_array( (int) $user_id, array_map( 'intval', $user_ids ), true ) ) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Get registered rank types.
	 *
	 * @since 1.7.6
	 * @return array|stdClass[]
	 */
	public function get_rank_types() {
		$args = array(
			'posts_per_page' => - 1,
			'post_status'    => array( 'publish' ),
			'post_type'      => 'rank-type',
		);

		$ranks      = new WP_Query( $args );
		$rank_types = ( ! empty( $ranks->get_posts() ) ) ? wp_list_pluck( $ranks->get_posts(), 'post_name' ) : array();

		return $rank_types;
	}

	/**
	 * Get all ranks.
	 *
	 * @param array $args Arguments.
	 *
	 * @since 1.7.6
	 * @return array|int[]|WP_Post[]
	 */
	public function gamipress_get_all_ranks( $args = array() ) {
		$defaults = array(
			'posts_per_page' => - 1,
			'paged'          => 1,
			'fields'         => 'all',
			'post_status'    => array( 'publish' ),
			'post_type'      => $this->get_rank_types(),
			's'              => '',
		);
		$args     = wp_parse_args( $args, $defaults );
		$query    = new \WP_Query( $args );

		return $query->posts;
	}

	/**
	 * Add/Removes user from rank on rank update.
	 *
	 * @param int     $user_id  User id.
	 * @param WP_Post $new_rank New rank data.
	 *
	 * @since 1.7.6
	 * @return void
	 */
	public function update_user_rank( $user_id, $new_rank ) {
		// Remove user from old rank.
		$registered_rank_types = self::instance()->get_rank_types();

		// Add user to new rank.
		if ( ! empty( $new_rank->ID ) && ! empty( $registered_rank_types ) && in_array( $new_rank->post_type, $registered_rank_types, true ) ) {
			AccessControls::instance()->condition_add_user( $user_id, self::instance()->condition_name, $new_rank->ID );
		}
	}
}
