<?php
/**
 * Holds GamiPress Achievement functionality related Access Controls.
 *
 * @package BuddyBossApp\Integrations\GamiPress\AccessControls
 */

namespace BuddyBossApp\Integrations\GamiPress\AccessControls;

use BuddyBossApp\AccessControls\TmpItems;
use BuddyBossApp\Integrations\GamiPress\AccessControls;
use CT_Query;
use stdClass;
use WP_Post;
use WP_Query;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * GamiPress Achievement class for Access Controls.
 */
class Achievement {
	/**
	 * Condition name.
	 *
	 * @var string $condition_name condition name.
	 */
	public $condition_name = 'gp-achievements';

	/**
	 * Class instance.
	 *
	 * @var bool
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return Achievement
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Class construct method.
	 *
	 * @since 1.7.6
	 * @return void
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Items callback method.
	 *
	 * @param string $search Search the condition.
	 * @param int    $page   Page number.
	 * @param int    $limit  Limit the items to be fetched.
	 *
	 * @since 1.7.6
	 * @return array
	 */
	public function gamipress_achievement_items_callback( $search = '', $page = 1, $limit = 20 ) {
		$list              = array();
		$achievement_types = $this->get_achievement_types();

		if ( empty( $achievement_types ) ) {
			return $list;
		}

		$args = array(
			'fields'         => 'all',
			'post_status'    => array( 'publish' ),
			'post_type'      => $achievement_types,
			'posts_per_page' => $limit,
			'paged'          => $page,
		);

		if ( ! empty( $search ) ) {
			$args['s'] = $search;
		}

		$results = $this->gamipress_get_all_achievements( $args );

		if ( ! empty( $results ) ) {
			foreach ( $results as $item ) {
				$list[ $item->ID ] = array(
					'id'   => $item->ID,
					'name' => esc_html( $item->post_title ),
					'link' => admin_url( "post.php?post={$item->ID}&action=edit" ),
				);
			}
		}

		return $list;
	}

	/**
	 * Item callback method.
	 *
	 * @param int $item_value Item value of condition.
	 *
	 * @since 1.7.6
	 * @return array|false
	 */
	public function gamipress_achievement_item_callback( $item_value ) {
		$achievement_type = get_post( $item_value );

		if ( empty( $achievement_type ) && ! $achievement_type instanceof WP_Post ) {
			return false;
		}

		return array(
			'id'   => $achievement_type->ID,
			'name' => $achievement_type->post_title,
			'link' => admin_url( "post.php?post={$achievement_type->ID}&action=edit" ),
		);
	}

	/**
	 * Users callback method.
	 *
	 * @param array $data     condition data.
	 * @param int   $page     current page number.
	 * @param int   $per_page limit.
	 *
	 * @since 1.7.6
	 * @return array
	 */
	public function gamipress_achievement_users_callback( $data, $page = 1, $per_page = 10 ) {
		$user_ids      = array();
		$sub_condition = ( ! empty( $data['sub_condition'] ) ) ? $data['sub_condition'] : '';
		$item_value    = ( ! empty( $data['item_value'] ) ) ? $data['item_value'] : '';
		$group_id      = ( ! empty( $data['group_id'] ) ) ? $data['group_id'] : 0;
		$rounds_count  = ( ! empty( $data['rounds_count'] ) ) ? $data['rounds_count'] : 0;

		if ( 'specific' === $sub_condition ) {
			if ( empty( $item_value ) ) {
				return $user_ids;
			}

			ct_setup_table( 'gamipress_user_earnings' );

			$query_args = array(
				'paged'          => $page,
				'items_per_page' => $per_page,
				'post_id'        => $item_value,
			);
			$ct_query   = new CT_Query( $query_args );
			$results    = $ct_query->get_results();

			if ( ! empty( $results ) ) {
				$user_ids = wp_list_pluck( $results, 'user_id' );
			}

			ct_reset_setup_table();

			return AccessControls::instance()->return_users( $user_ids );
		} elseif ( 'any' === $sub_condition ) {
			$transient_key    = "access_ctrls_gamipress_achievements_condition_jb_data_{$group_id}";
			$all_achievements = $this->gamipress_get_all_achievements( array( 'fields' => 'ids' ) );
			$achievements_ids = array();

			if ( ! empty( $all_achievements ) ) {
				foreach ( $all_achievements as $id ) {
					$achievements_ids[ $id ] = $id;
				}
			}

			// Check if it's a first round.
			if ( 1 === $rounds_count ) {
				$condition_data                        = array(
					'prepare_page'            => 1,
					'is_data_prepared'        => false,
					'pending_achievement_ids' => $achievements_ids,
				);
				$condition_data['current_achievement'] = array_key_first( $condition_data['pending_achievement_ids'] );
			} else {
				$condition_data = get_transient( $transient_key );
			}

			/**
			 * Preparing the data into tmp items collections.
			 */
			if ( ! $condition_data['is_data_prepared'] ) {
				$achievement_users = gamipress_get_achievement_earners( $condition_data['current_achievement'] );
				$achievement_users = ( ! empty( $achievement_users ) ) ? wp_list_pluck( $achievement_users, 'ID' ) : array();
				$user_ids          = ! empty( $achievement_users ) ? AccessControls::instance()->paginate_users_list( array_unique( $achievement_users ), $condition_data['prepare_page'], 500 ) : array();

				if ( ! empty( $user_ids ) ) {
					$items = array();

					foreach ( $user_ids as $user_id ) {
						$items[] = array(
							'item_type'         => "gp_achievements_{$group_id}",
							'item_id'           => $user_id,
							'secondary_item_id' => $condition_data['current_achievement'],
						);
					}

					$insert = TmpItems::instance()->insert_items( $items );

					$condition_data['prepare_page'] ++;

					if ( empty( $insert ) ) {
						return AccessControls::instance()->return_error( __( 'Error while preparing members data for GamiPress Achievement..', 'buddyboss-app' ) );
					}
				} else { // when all users are processed.
					unset( $condition_data['pending_achievement_ids'][ $condition_data['current_achievement'] ] ); // remove from pending membership id.

					if ( ! empty( $condition_data['pending_achievement_ids'] ) ) {
						$condition_data['current_achievement'] = array_key_first( $condition_data['pending_achievement_ids'] ); // get next membership to process.
						$condition_data['prepare_page']        = 1; // reset pagination to 1.
					} else {
						// If pending memberships found empty finish the job.
						$condition_data['is_data_prepared'] = true;
					}

					set_transient( $transient_key, $condition_data, DAY_IN_SECONDS );

					return AccessControls::instance()->return_wait( __( 'Preparing GamiPress achievement Members Data', 'buddyboss-app' ) );
				}
			}

			$items = TmpItems::instance()->get_items(
				array(
					'include_item_types' => array( "gp_achievements_{$group_id}" ),
					'per_page'           => $per_page,
					'page'               => $page,
				)
			);

			$user_ids = array();

			foreach ( $items['results'] as $item ) {
				$user_ids[] = $item->item_id;
			}

			return AccessControls::instance()->return_users( $user_ids );
		}

		return AccessControls::instance()->return_error( __( 'Invalid sub condition.', 'buddyboss-app' ) );
	}

	/**
	 * Function to check if user has any condition.
	 *
	 * @param int $user_id User id to check.
	 *
	 * @since 1.7.6
	 * @return bool
	 */
	public function has_any_items_callback( $user_id ) {
		return ! empty( gamipress_get_user_achievements( array( 'user_id' => $user_id ) ) );
	}

	/**
	 * Get registered achievement types.
	 *
	 * @since 1.7.6
	 * @return array|stdClass[]
	 */
	public function get_achievement_types() {
		$args = array(
			'posts_per_page' => - 1,
			'post_status'    => array( 'publish' ),
			'post_type'      => 'achievement-type',
		);

		$achievements = new WP_Query( $args );

		return ( ! empty( $achievements->get_posts() ) ) ? wp_list_pluck( $achievements->get_posts(), 'post_name' ) : array();
	}

	/**
	 * Get all achievements.
	 *
	 * @param array $args Arguments.
	 *
	 * @since 1.7.6
	 * @return array|int[]|WP_Post[]
	 */
	public function gamipress_get_all_achievements( $args = array() ) {
		$defaults = array(
			'posts_per_page' => - 1,
			'paged'          => 1,
			'fields'         => 'all',
			'post_status'    => array( 'publish' ),
			'post_type'      => $this->get_achievement_types(),
			's'              => '',
		);
		$args     = wp_parse_args( $args, $defaults );
		$query    = new WP_Query( $args );

		return $query->posts;
	}
}
