<?php
/**
 * Holds GamiPress functionality related Access Controls.
 *
 * @package BuddyBossApp\Integrations\GamiPress
 */

namespace BuddyBossApp\Integrations\GamiPress;

use BuddyBossApp\Integrations\GamiPress\AccessControls\Rank;
use BuddyBossApp\Integrations\GamiPress\AccessControls\Achievement;
use mysqli_result;
use WP_Post;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * GamiPress Access Controls class.
 */
class AccessControls extends \BuddyBossApp\AccessControls\Integration_Abstract {
	/**
	 * Function to set up the conditions.
	 *
	 * @since 1.7.6
	 *
	 * @return mixed|void
	 */
	public function setup() {
		// Register Conditions.
		$this->register_condition(
			array(
				'condition'              => Rank::instance()->condition_name,
				'items_callback'         => array( Rank::instance(), 'gamipress_rank_items_callback' ),
				'item_callback'          => array( Rank::instance(), 'gamipress_rank_item_callback' ),
				'users_callback'         => array( Rank::instance(), 'gamipress_rank_users_callback' ),
				'labels'                 => array(
					'condition_name'          => __( 'GamiPress Ranks', 'buddyboss-app' ),
					'item_singular'           => __( 'Rank', 'buddyboss-app' ),
					'member_of_specific_item' => __( 'Member received specific rank', 'buddyboss-app' ),
					'member_of_any_items'     => __( 'Member received any of the rank', 'buddyboss-app' ),
				),
				'support_any_items'      => true,
				'has_any_items_callback' => array( Rank::instance(), 'has_any_items_callback' ),
			)
		);

		$this->register_condition(
			array(
				'condition'              => Achievement::instance()->condition_name,
				'items_callback'         => array( Achievement::instance(), 'gamipress_achievement_items_callback' ),
				'item_callback'          => array( Achievement::instance(), 'gamipress_achievement_item_callback' ),
				'users_callback'         => array( Achievement::instance(), 'gamipress_achievement_users_callback' ),
				'labels'                 => array(
					'condition_name'          => __( 'GamiPress Achievements', 'buddyboss-app' ),
					'item_singular'           => __( 'Achievement', 'buddyboss-app' ),
					'member_of_specific_item' => __( 'Member received specific achievement', 'buddyboss-app' ),
					'member_of_any_items'     => __( 'Member received any of achievement', 'buddyboss-app' ),
				),
				'support_any_items'      => true,
				'has_any_items_callback' => array( Achievement::instance(), 'has_any_items_callback' ),
			)
		);

		$this->load_hooks();
	}

	/**
	 * Function to load all hooks of this condition.
	 *
	 * @since 1.7.6
	 */
	public function load_hooks() {
		// Common hooks.
		add_action( 'gamipress_award_achievement', array( $this, 'add_user_to_rank_or_achievement' ), PHP_INT_MAX, 2 );
		add_action(
			'gamipress_revoke_achievement_to_user_after',
			array(
				$this,
				'remove_user_from_rank_or_achievement',
			),
			PHP_INT_MAX,
			2
		);
		add_action(
			'transition_post_status',
			array(
				$this,
				'remove_rank_achievement_access_control_data',
			),
			PHP_INT_MAX,
			3
		);

		// Rank hooks.
		add_action( 'gamipress_update_user_rank', array( Rank::instance(), 'update_user_rank' ), PHP_INT_MAX, 2 );
	}

	/**
	 * Add user to rank.
	 *
	 * @param int $user_id             The given user's ID.
	 * @param int $rank_achievement_id The given rank OR achievement ID to award.
	 *
	 * @since  1.7.6
	 */
	public function add_user_to_rank_or_achievement( $user_id, $rank_achievement_id ) {
		$rank_type             = gamipress_get_post_type( $rank_achievement_id );
		$registered_rank_types = Rank::instance()->get_rank_types();

		if ( ! empty( $rank_type ) && ! empty( $registered_rank_types ) && in_array( $rank_type, $registered_rank_types, true ) ) {
			$this->condition_add_user( $user_id, Rank::instance()->condition_name, $rank_achievement_id );
		}

		$achievement_type             = gamipress_get_post_type( $rank_achievement_id );
		$registered_achievement_types = Achievement::instance()->get_achievement_types();

		if ( ! empty( $achievement_type ) && ! empty( $registered_achievement_types ) && in_array( $achievement_type, $registered_achievement_types, true ) ) {
			$this->condition_add_user( $user_id, Achievement::instance()->condition_name, $rank_achievement_id );
		}
	}

	/**
	 * Remove user from rank.
	 *
	 * @param int $user_id             User id.
	 * @param int $rank_achievement_id Rank OR Achievement id.
	 *
	 * @since 1.7.6
	 */
	public function remove_user_from_rank_or_achievement( $user_id, $rank_achievement_id ) {
		$rank_type             = gamipress_get_post_type( $rank_achievement_id );
		$registered_rank_types = Rank::instance()->get_rank_types();

		if ( ! empty( $rank_type ) && ! empty( $registered_rank_types ) && in_array( $rank_type, $registered_rank_types, true ) ) {
			$this->condition_remove_user( $user_id, Rank::instance()->condition_name, $rank_achievement_id );
		}

		$achievement_type             = gamipress_get_post_type( $rank_achievement_id );
		$registered_achievement_types = Achievement::instance()->get_achievement_types();

		if ( ! empty( $achievement_type ) && ! empty( $registered_achievement_types ) && in_array( $achievement_type, $registered_achievement_types, true ) ) {
			$this->condition_remove_user( $user_id, Achievement::instance()->condition_name, $rank_achievement_id );
		}
	}

	/**
	 * Function will remove member/course group from access groups when remove course group from admin.
	 *
	 * @param string  $new_status New post status.
	 * @param string  $old_status Old post status.
	 * @param WP_Post $post       Post object.
	 *
	 * @since 1.7.6
	 */
	public function remove_rank_achievement_access_control_data( $new_status, $old_status, $post ) {
		$registered_rank_types = Rank::instance()->get_rank_types();

		if ( ! empty( $new_status ) && ! empty( $registered_rank_types ) && 'publish' !== $new_status && in_array( $post->post_type, $registered_rank_types, true ) ) {
			$this->condition_item_deleted( Rank::instance()->condition_name, $post->ID );
			$this->condition_remove_all_users( Rank::instance()->condition_name, $post->ID );
			bb_access_recalculate_member_for_has_any_membership_group( Rank::instance()->condition_name );
		}

		$registered_achievement_types = Achievement::instance()->get_achievement_types();

		if ( ! empty( $new_status ) && ! empty( $registered_achievement_types ) && 'publish' !== $new_status && in_array( $post->post_type, $registered_achievement_types, true ) ) {
			$this->condition_item_deleted( Achievement::instance()->condition_name, $post->ID );
			$this->condition_remove_all_users( Achievement::instance()->condition_name, $post->ID );
			bb_access_recalculate_member_for_has_any_membership_group( Achievement::instance()->condition_name );
		}
	}
}
