<?php

namespace BuddyBossApp\Integrations\BuddyBossPlatform;

use BuddyBossApp\AppSettings;
use WP_REST_Server;

/**
 * Integration code related to BuddyBoss Platform.
 * Class Main
 *
 * @package BuddyBossApp\Integrations\BuddyBossPlatform
 */
class Rest {

	private static $instance;

	/**
	 * Rest constructor.
	 */
	public function __construct() {
	}

	/**
	 * @return Rest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 *
	 */
	public function load() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
		add_filter( 'bbapp_rest_profile_tab', array( $this, 'add_profile_tabs' ), 99 );
		add_filter( 'bp_rest_profile_tab', array( $this, 'bbapp_bp_rest_profile_tab' ), 10, 2 );
		add_filter( 'bp_rest_group_tabs', array( $this, 'bbapp_bp_rest_group_tabs' ), 99, 1 );
		add_filter( 'bbapp_settings_rest_response', array( $this, 'addSettings' ) );

		// Update the user active time when he loads the API.
		add_filter( 'rest_post_dispatch', array( $this, 'update_user_active_time' ), 10, 3 );

		// Get connected groups by course id.
		add_filter( 'bp_rest_groups_get_items_query_args', array( $this, 'bbapp_bp_rest_group_by_course_id' ), 10, 2 );
		add_filter( 'mepr_developer_tools_rest_item_response', array( $this, 'bbapp_bp_rest_has_connected_groups' ), 10, 2 );
	}

	/**
	 * Load the rest API.
	 */
	public function register_routes() {
	}

	/**
	 * Add the BuddyPress Profile Tabs.
	 *
	 * @param $tabs
	 *
	 * @return mixed
	 */
	public function add_profile_tabs( $tabs ) {

		if ( ! function_exists( 'buddypress' ) ) {
			return $tabs;
		}

		$buddyPressTabs = array();

		$navs = buddypress()->members->nav;

		// if it's nouveau then let it order the tabs.
		if ( function_exists( 'bp_nouveau_set_nav_item_order' ) ) {
			bp_nouveau_set_nav_item_order( $navs, bp_nouveau_get_appearance_settings( 'user_nav_order' ) );
		}

		$default_tab = 'profile';

		if ( function_exists( 'bp_nouveau_get_appearance_settings' ) ) {
			$tab         = bp_nouveau_get_appearance_settings( 'user_default_tab' );
			$default_tab = bp_is_active( $tab ) ? $tab : $default_tab;
		}

		/**
		 * ID map for rename for previous name support.
		 */
		$id_map = array(
			'activity' => 'activities',
			'profile'  => 'xprofile',
		);

		if ( ! empty( $navs->get_primary() ) ) {
			foreach ( $navs->get_primary() as $nav ) {

				$name = $nav['name'];

				// remove the notification numbers.
				$name = preg_replace( '/^(.*)(<(.*)<\/(.*)>)/', '$1', $name );

				$name = trim( $name );

				$id = $nav['slug'];

				if ( isset( $id_map[ $id ] ) ) {
					$id = $id_map[ $id ];
				}

				$tab = array(
					'id'                      => $id,
					'title'                   => $name,
					'default'                 => false,
					'show_for_displayed_user' => $nav['show_for_displayed_user'],
				);

				if ( $default_tab == $nav['slug'] ) {
					$tab['default'] = true;
				}

				$buddyPressTabs[] = apply_filters( 'bbapp_profile_tab', $tab, $nav );
			}
		}

		// NOTE : Must be merged NOT overridden and must avoid duplicates
		$tabs = array_merge( $tabs, $buddyPressTabs );
		$tabs = bbapp_unique_multidim_array( $tabs, 'id' );

		return $tabs;
	}

	/**
	 * Add new field in profile tabs.
	 *
	 * @param $tab
	 * @param $nav
	 *
	 * @return mixed
	 */
	public function bbapp_bp_rest_profile_tab( $tab, $nav ) {
		$hide_in_app        = $this->is_hide_in_app_profile_tab( false, $tab );
		$tab['hide_in_app'] = apply_filters( 'bbapp_profile_tab_visibility', $hide_in_app, $tab, $nav );
		$tab['tab_icon']    = apply_filters( 'bbapp_profile_tab_icon', null, $tab, $nav );

		if ( isset( $tab['children'] ) && ! empty( $tab['children'] ) ) {
			foreach ( $tab['children'] as $key => $children_tab ) {
				$hide_in_app                            = $this->is_hide_in_app_profile_tab( false, $children_tab );
				$tab['children'][ $key ]['hide_in_app'] = apply_filters( 'bbapp_profile_child_tab_visibility', $hide_in_app, $children_tab, $tab, $nav );
				$tab['children'][ $key ]['tab_icon']    = apply_filters( 'bbapp_profile_child_tab_icon', null, $children_tab, $tab, $nav );
			}
		}

		return $tab;
	}

	/**
	 * Add new fields in groups tabs.
	 *
	 * @param $navigations
	 *
	 * @return mixed
	 */
	public function bbapp_bp_rest_group_tabs( $navigations ) {
		if ( is_array( $navigations ) ) {
			$navigations = json_decode( wp_json_encode( $navigations ), true );
			foreach ( $navigations as $key => $navigation ) {
				$hide_in_app                        = $this->is_hide_in_app_group_tab( false, $navigation );
				$navigations[ $key ]['hide_in_app'] = apply_filters( 'bbapp_group_tab_visibility', $hide_in_app, $navigation );
				$navigations[ $key ]['tab_icon']    = apply_filters( 'bbapp_group_tab_icon', null, $navigation );
				if ( ! empty( $navigation['children'] ) ) {
					foreach ( $navigation['children'] as $navigation_key => $children_navigation ) {
						$hide_in_app = $this->is_hide_in_app_group_tab( false, $children_navigation );

						$navigations[ $key ]['children'][ $navigation_key ]['hide_in_app'] = apply_filters( 'bbapp_group_child_tab_visibility', $hide_in_app, $children_navigation, $navigation );
						$navigations[ $key ]['children'][ $navigation_key ]['tab_icon']    = apply_filters( 'bbapp_group_child_tab_icon', null, $children_navigation, $navigation );
					}
				}
			}
		}

		return $navigations;
	}

	/**
	 * Is hide in app group tabs.
	 *
	 * @param $status
	 * @param $navigation
	 *
	 * @return bool
	 */
	public function is_hide_in_app_group_tab( $status, $navigation ) {
		if ( isset( $navigation['id'] ) ) {
			$group_tabs = $this->get_core_group_tabs();
			if ( array_key_exists( $navigation['id'], $group_tabs ) ) {
				return false;
			}
		}

		return false === $status ? ! (bool) AppSettings::instance()->get_setting_value( 'compatibility.custom_tab.group' ) : true;
	}

	/**
	 * Core group tabs list.
	 *
	 * @return mixed|void
	 */
	public function get_core_group_tabs() {
		$forum_slug = get_option( '_bbp_forum_slug', 'forum' );

		$nav_items = array(
			'members'             => __( 'Members', 'buddyboss-app' ),
			'invite'              => __( 'Send Invites', 'buddyboss-app' ),
			'send-invites'        => __( 'Send Invites', 'buddyboss-app' ),
			'pending-invites'     => __( 'Pending Invites', 'buddyboss-app' ),
			'manage'              => __( 'Manage', 'buddyboss-app' ),
			'admin'               => __( 'Manage', 'buddyboss-app' ),
			'edit-details'        => __( 'Details', 'buddyboss-app' ),
			'group-settings'      => __( 'Settings', 'buddyboss-app' ),
			'group-avatar'        => __( 'Photo', 'buddyboss-app' ),
			'group-cover-image'   => __( 'Cover Photo', 'buddyboss-app' ),
			'manage-members'      => __( 'Members', 'buddyboss-app' ),
			'membership-requests' => __( 'Requests', 'buddyboss-app' ),
			'delete-group'        => __( 'Delete', 'buddyboss-app' ),
			'photos'              => __( 'Photos', 'buddyboss-app' ),
			'albums'              => __( 'Albums', 'buddyboss-app' ),
			'videos'              => __( 'Videos', 'buddyboss-app' ),
			$forum_slug           => __( 'Discussions', 'buddyboss-app' ),
			'subgroups'           => __( 'Subgroups', 'buddyboss-app' ),
			'courses'             => __( 'Courses', 'buddyboss-app' ),
			'reports'             => __( 'Reports', 'buddyboss-app' ),
			'activity'            => __( 'Feed', 'buddyboss-app' ),
			'messages'            => __( 'Send Messages', 'buddyboss-app' ),
			'documents'           => __( 'Documents', 'buddyboss-app' ),
		);

		return apply_filters( 'bbapp_core_group_tabs', $nav_items );
	}


	/**
	 * Is hide in app profile tabs.
	 *
	 * @param $status
	 * @param $tab
	 *
	 * @return bool
	 */
	public function is_hide_in_app_profile_tab( $status, $tab ) {
		if ( isset( $tab['id'] ) ) {

			$profile_tabs = $this->get_core_profile_tabs();

			if ( array_key_exists( $tab['id'], $profile_tabs ) ) {
				return false;
			}
		}

		return false === $status ? ! (bool) AppSettings::instance()->get_setting_value( 'compatibility.custom_tab.profile' ) : true;
	}

	/**
	 * Core profile tabs list.
	 *
	 * @return mixed|void
	 */
	public function get_core_profile_tabs() {
		$id_map = array(
			'activity' => 'activities',
			'profile'  => 'xprofile',
		);

		$nav_items = array(
			'public'             => __( 'View', 'buddyboss-app' ),
			'edit'               => __( 'Edit', 'buddyboss-app' ),
			'change-avatar'      => __( 'Profile Photo', 'buddyboss-app' ),
			'change-cover-image' => __( 'Cover Photo', 'buddyboss-app' ),
			'just-me'            => __( 'Personal', 'buddyboss-app' ),
			'favorites'          => __( 'Likes', 'buddyboss-app' ),
			'mentions'           => __( 'Mentions', 'buddyboss-app' ),
			'following'          => __( 'Following', 'buddyboss-app' ),
			'my-friends'         => __( 'All Connections', 'buddyboss-app' ),
			'mutual'             => __( 'Mutual Connections', 'buddyboss-app' ),
			'requests'           => __( 'Requests', 'buddyboss-app' ),
			'my-sites'           => __( 'My Sites', 'buddyboss-app' ),
			'my-document'        => __( 'Documents', 'buddyboss-app' ),
			'my-groups'          => __( 'My Groups', 'buddyboss-app' ),
			'invites'            => __( 'Invitations', 'buddyboss-app' ),
			'send-invites'       => __( 'Send Invites', 'buddyboss-app' ),
			'sent-invites'       => __( 'Sent Invites', 'buddyboss-app' ),
			'my-media'           => __( 'Photos', 'buddyboss-app' ),
			'albums'             => __( 'Albums', 'buddyboss-app' ),
			'inbox'              => __( 'Messages', 'buddyboss-app' ),
			'compose'            => __( 'New Message', 'buddyboss-app' ),
			'notices'            => __( 'Notices', 'buddyboss-app' ),
			'unread'             => __( 'Unread', 'buddyboss-app' ),
			'read'               => __( 'Read', 'buddyboss-app' ),
			'general'            => __( 'Login Information', 'buddyboss-app' ),
			'notifications'      => __( 'Email Preferences', 'buddyboss-app' ),
			'profile'            => __( 'Privacy', 'buddyboss-app' ),
			'export'             => __( 'Export Data', 'buddyboss-app' ),
			'blocked-members'    => __( 'Blocked Members', 'buddyboss-app' ),
			'capabilities'       => __( 'Capabilities', 'buddyboss-app' ),
			'delete-account'     => __( 'Delete Account', 'buddyboss-app' ),
			'my-video'           => __( 'Videos', 'buddyboss-app' ),
		);

		if ( function_exists( 'bp_is_active' ) ) {
			if ( function_exists( 'bp_get_profile_slug' ) ) {
				$profile_slug               = isset( $id_map[ bp_get_profile_slug() ] ) ? $id_map[ bp_get_profile_slug() ] : bp_get_profile_slug();
				$nav_items[ $profile_slug ] = __( 'Profile', 'buddyboss-app' );
			}
			if ( function_exists( 'bp_get_settings_slug' ) ) {
				$nav_items[ bp_get_settings_slug() ] = __( 'Account', 'buddyboss-app' );
			}
			if ( bp_is_active( 'activity' ) ) {
				$activity_slug               = isset( $id_map[ bp_get_activity_slug() ] ) ? $id_map[ bp_get_activity_slug() ] : bp_get_activity_slug();
				$nav_items[ $activity_slug ] = __( 'Timeline', 'buddyboss-app' );
			}
			if ( bp_is_active( 'friends' ) ) {
				$nav_items[ bp_get_friends_slug() ] = __( 'Connections', 'buddyboss-app' );
			}
			if ( bp_is_active( 'media' ) ) {
				if ( function_exists( 'bp_get_video_slug' ) ) {
					$nav_items[ bp_get_video_slug() ] = __( 'Videos', 'buddyboss-app' );
				}
				if ( function_exists( 'bp_get_media_slug' ) ) {
					$nav_items[ bp_get_media_slug() ] = __( 'Photos', 'buddyboss-app' );
				}
				if ( function_exists( 'bp_get_document_slug' ) ) {
					$nav_items[ bp_get_document_slug() ] = __( 'Documents', 'buddyboss-app' );
				}
			}
			if ( bp_is_active( 'invites' ) ) {
				$nav_items[ bp_get_invites_slug() ] = __( 'Email Invites', 'buddyboss-app' );
			}
			if ( bp_is_active( 'messages' ) ) {
				$nav_items[ bp_get_messages_slug() ] = __( 'Messages', 'buddyboss-app' );
			}
			if ( bp_is_active( 'notifications' ) ) {
				$nav_items[ bp_get_notifications_slug() ] = __( 'Notifications', 'buddyboss-app' );
			}
			if ( bp_is_active( 'groups' ) ) {
				$nav_items[ bp_get_groups_slug() ] = __( 'Groups', 'buddyboss-app' );
			}
			if ( bp_is_active( 'blogs' ) ) {
				$nav_items[ bp_get_blogs_slug() ] = __( 'Sites', 'buddyboss-app' );
			}
			if ( defined( 'BP_FORUMS_SLUG' ) ) {
				$nav_items[ BP_FORUMS_SLUG ] = __( 'Forums', 'buddyboss-app' );
			}
			if ( bp_is_active( 'forums' ) ) {
				$nav_items[ bbp_get_topic_archive_slug() ]      = ( bp_loggedin_user_id() === bp_displayed_user_id() ) ? __( 'My Discussions', 'buddyboss-app' ) : __( 'Discussions', 'buddyboss-app' );
				$nav_items[ bbp_get_reply_archive_slug() ]      = ( bp_loggedin_user_id() === bp_displayed_user_id() ) ? __( 'My Replies', 'buddyboss-app' ) : __( 'Replies', 'buddyboss-app' );
				$nav_items[ bbp_get_user_favorites_slug() ]     = ( bp_loggedin_user_id() === bp_displayed_user_id() ) ? __( 'My Favorites', 'buddyboss-app' ) : __( 'Favorites', 'buddyboss-app' );
				$nav_items[ bbp_get_user_subscriptions_slug() ] = __( 'Subscriptions', 'buddyboss-app' );
			}
		}

		return apply_filters( 'bbapp_core_profile_tabs', $nav_items );
	}



	/**
	 * @param $response
	 *
	 * @return array
	 */
	public function addSettings( $response ) {
		// Check Permalinks for platform bb core slugs.
		$permalinks_settings       = $this->bbapp_rest_get_bb_core_slugs_settings();
		$response['bb_core_slugs'] = apply_filters( 'bbapp_rest_bb_core_slugs_settings', $permalinks_settings );
		return $response;
	}

	/**
	 * Get bb core slugs settings.
	 *
	 * @return array
	 */
	public function bbapp_rest_get_bb_core_slugs_settings() {
		$results = array();

		if ( function_exists( 'bp_is_active' ) ) {
			if ( function_exists( 'bp_get_profile_slug' ) ) {
				$results['profile'] = bp_get_profile_slug();
			}
			if ( bp_is_active( 'settings' ) && function_exists( 'bp_get_settings_slug' ) ) {
				$results['settings'] = bp_get_settings_slug();
			}

			if ( bp_is_active( 'activity' ) ) {
				$results['activity'] = bp_get_activity_slug();
			}
			if ( bp_is_active( 'friends' ) ) {
				$results['friends'] = bp_get_friends_slug();
			}
			if ( bp_is_active( 'media' ) ) {
				if ( function_exists( 'bp_get_media_slug' ) ) {
					$results['media'] = bp_get_media_slug();
				}
				if ( function_exists( 'bp_get_video_slug' ) ) {
					$results['video'] = bp_get_video_slug();
				}
				if ( function_exists( 'bp_get_document_slug' ) ) {
					$results['document'] = bp_get_document_slug();
				}
			}
			if ( bp_is_active( 'invites' ) ) {
				$results['invites'] = bp_get_invites_slug();
			}
			if ( bp_is_active( 'messages' ) ) {
				$results['messages'] = bp_get_messages_slug();
			}
			if ( bp_is_active( 'notifications' ) ) {
				$results['notifications'] = bp_get_notifications_slug();
			}
			if ( bp_is_active( 'groups' ) ) {
				$results['groups'] = bp_get_groups_slug();
			}
			if ( bp_is_active( 'blogs' ) ) {
				$results['blogs'] = bp_get_blogs_slug();
			}
		}

		return $results;
	}

	/**
	 * Listener function for the logged-in user's 'last_activity' metadata on Rest API.
	 *
	 * @param \WP_REST_Response $result  Rest response.
	 * @param \WP_REST_Server   $server  Server instance.
	 * @param \WP_REST_Request  $request Request used to generate the response.
	 *
	 * @since 1.6.5
	 * @since 2.3.20 Added $server and $request parameter.
	 *
	 * @return mixed
	 */
	public function update_user_active_time( $result, $server, $request ) {
		if (
			! is_user_logged_in() ||
			! isset( $result->status ) ||
			200 !== $result->status ||
			(
				// If deleting own account then do not record last activity.
				WP_REST_Server::DELETABLE === $request->get_method() &&
				false !== strpos( $request->get_route(), 'buddyboss/v1/members' ) &&
				get_current_user_id() === (int) $request->get_param( 'id' ) &&
				! empty( $result->data['deleted'] )
			)
		) {
			return $result;
		}

		if ( function_exists( 'bp_core_record_activity' ) ) {
			bp_core_record_activity();
		}

		return $result;
	}

	/**
	 * Filter the group query args to order by course id.
	 *
	 * @param array            $args    Query args.
	 * @param \WP_REST_Request $request Request object.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function bbapp_bp_rest_group_by_course_id( $args, $request ) {
		if ( ! empty( $request['course_id'] ) && function_exists( 'bb_load_meprlms_group' ) ) {
			$connected_groups_data = bb_load_meprlms_group()->get(
				array(
					'course_id' => array( $request['course_id'] ),
				)
			);

			if ( ! empty( $connected_groups_data['courses'] ) ) {
				$group_ids = wp_list_pluck( $connected_groups_data['courses'], 'group_id' );

				if ( ! empty( $args['include'] ) ) {
					$args['include'] = array_merge( $args['include'], $group_ids );
				} else {
					$args['include'] = $group_ids;
				}
			}
		}

		return $args;
	}

	/**
	 * Filter the item response to add has connected groups.
	 *
	 * @param object $item     Post object.
	 * @param object $request  Request object.
	 *
	 * @since 2.2.80
	 * @return mixed
	 */
	public function bbapp_bp_rest_has_connected_groups( $item, $request ) {
		if ( 'mpcs-course' !== get_post_type( $item['id'] ) ) {
			return $item;
		}

		if ( function_exists( 'bb_load_meprlms_group' ) ) {
			$connected_groups_data = bb_load_meprlms_group()->get(
				array(
					'course_id' => array( $item['id'] ),
				)
			);

			$item['has_connected_groups'] = ! empty( $connected_groups_data['courses'] );
			$item['connected_groups']     = ! empty( $connected_groups_data['courses'] ) ? $connected_groups_data['courses'] : array();
		}
		return $item;
	}
}
