<?php
/**
 * Holds message related push notifications functionality.
 *
 * @package BuddyBossApp\Integrations\BuddyBossPlatform\Push
 */

namespace BuddyBossApp\Integrations\BuddyBossPlatform\Push;

use BuddyBossApp\Notification\IntegrationAbstract;

/**
 * Messages component push notification support.
 * Class MessagesNotification
 *
 * @package BuddyBossApp\Integrations\BuddyBossPlatform\Push
 */
class MessagesNotification extends IntegrationAbstract {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Load methods.
	 */
	public function load() {
		$this->push_types();
		$this->link_notifications();

	}

	/**
	 * Register Subscriptions Types
	 */
	public function push_types() {
		if ( bbapp_is_platform_notification_preferences_legacy_mode_enabled() ) {
			// Messages.
			$this->register_push_group( 'messages', __( 'Messages', 'buddyboss-app' ) );
			$this->register_push_type( 'bp_messages_new_message', __( 'A member receives a new message', 'buddyboss-app' ), __( 'A member sends you a new message', 'buddyboss-app' ), array( 'push_group' => 'messages' ) );
		}
	}

	/**
	 * Link Normal Notification to Push.
	 */
	public function link_notifications() {
		if ( bbapp_is_platform_notification_preferences_legacy_mode_enabled() ) {
			// Messages.
			$this->register_push_to_normal( 'messages', 'new_message', 'bp_messages_new_message' );

			// Modern notification.
			$this->register_push_to_normal( 'messages', 'bb_messages_new', 'bp_messages_new_message' );
		}
	}

	/**
	 * Function to change push notification format.
	 *
	 * @param string $content           Notification content.
	 * @param string $component_name    Component name.
	 * @param string $component_action  Component action name.
	 * @param int    $item_id           Item id.
	 * @param int    $secondary_item_id Secondary item id.
	 * @param int    $notification_id   Notification id.
	 * @param int    $total_items       Number of total items.
	 * @param string $format            Notification format.
	 * @param string $screen            Screen name.
	 *
	 * @since 1.4.7
	 * @return mixed|string
	 */
	public function format_notification_override( $content, $component_name, $component_action, $item_id, $secondary_item_id, $notification_id, $total_items, $format, $screen ) {
		if ( 'app_push' !== $screen ) {
			return $content;
		}

		$user_id = $secondary_item_id;

		switch ( $component_action ) {
			// Modern Notifications.
			case 'bb_messages_new':
				// Get message thread ID.
				$message      = new \BP_Messages_Message( $item_id );
				$media_ids    = bp_messages_get_meta( $item_id, 'bp_media_ids', true );
				$document_ids = bp_messages_get_meta( $item_id, 'bp_document_ids', true );
				$video_ids    = bp_messages_get_meta( $item_id, 'bp_video_ids', true );
				$gif_data     = bp_messages_get_meta( $item_id, '_gif_data', true );
				$excerpt      = wp_strip_all_tags( $message->message );

				if ( '&nbsp;' === $excerpt ) {
					$excerpt = '';
				} else {
					$excerpt = '"' . bp_create_excerpt(
						$excerpt,
						50,
						array(
							'ending' => __( '&hellip;', 'buddyboss-app' ),
						)
					) . '"';

					$excerpt = str_replace( '&hellip;"', '&hellip;', $excerpt );
				}

				if ( ! empty( $excerpt ) ) {
					/* translators: 1. except text */
					$text = sprintf( esc_html__( 'Sent you a message: %1$s', 'buddyboss-app' ), $excerpt );
				} elseif ( $media_ids ) {
					$media_ids = array_filter( explode( ',', $media_ids ) );

					if ( count( $media_ids ) > 1 ) {
						/* translators: 1. photos text */
						$text = sprintf( esc_html__( 'Sent you %1$s', 'buddyboss-app' ), esc_html__( 'some photos', 'buddyboss-app' ) );
					} else {
						/* translators: 1. user display name 2. photo text */
						$text = sprintf( esc_html__( 'Sent you %1$s', 'buddyboss-app' ), esc_html__( 'a photo', 'buddyboss-app' ) );
					}
				} elseif ( $document_ids ) {
					$document_ids = array_filter( explode( ',', $document_ids ) );

					if ( count( $document_ids ) > 1 ) {
						/* translators: 1. documents text */
						$text = sprintf( esc_html__( 'Sent you %1$s', 'buddyboss-app' ), esc_html__( 'some documents', 'buddyboss-app' ) );
					} else {
						/* translators: 1. document text */
						$text = sprintf( esc_html__( 'Sent you %1$s', 'buddyboss-app' ), esc_html__( 'a document', 'buddyboss-app' ) );
					}
				} elseif ( $video_ids ) {
					$video_ids = array_filter( explode( ',', $video_ids ) );
					if ( count( $video_ids ) > 1 ) {
						/* translators: 1. videos text */
						$text = sprintf( esc_html__( 'Sent you %1$s', 'buddyboss-app' ), esc_html__( 'some videos', 'buddyboss-app' ) );
					} else {
						/* translators: 1. video text */
						$text = sprintf( esc_html__( 'Sent you %1$s', 'buddyboss-app' ), esc_html__( 'a video', 'buddyboss-app' ) );
					}
				} elseif ( empty( $gif_data ) ) {
					/* translators: 1. gif text */
					$text = sprintf( esc_html__( 'Sent you %1$s', 'buddyboss-app' ), esc_html__( 'a gif', 'buddyboss-app' ) );
				} else {
					$text = esc_html__( 'Sent you a message', 'buddyboss-app' );
				}

				$content['title'] = bp_core_get_user_displayname( $user_id );
				if ( ! empty( $text ) ) {
					$content['text'] = $text;
				}
				break;
			// Legacy Notifications.
			case 'new_message':
				$content['title'] = bp_core_get_user_displayname( $user_id );
				break;
		}

		return $content;
	}
}
