<?php
/**
 * Holds functionality related group push notifications.
 *
 * @package BuddyBossApp\Integrations\BuddyBossPlatform\Push
 */

namespace BuddyBossApp\Integrations\BuddyBossPlatform\Push;

use BuddyBossApp\Notification\IntegrationAbstract;

/**
 * Groups component push notification support.
 * Class GroupsNotification
 *
 * @package BuddyBossApp\Integrations\BuddyBossPlatform\Push
 */
class GroupsNotification extends IntegrationAbstract {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Load methods.
	 */
	public function load() {
		$this->push_types();
		$this->link_notifications();
	}

	/**
	 * Register Subscriptions Types
	 */
	public function push_types() {
		if ( bbapp_is_platform_notification_preferences_legacy_mode_enabled() ) {
			// Groups.
			$this->register_push_group( 'groups', __( 'Social Groups', 'buddyboss-app' ) );
			$this->register_push_type( 'bp_groups_new_invite', __( 'A member is invited to join a group', 'buddyboss-app' ), __( 'A member invites you to join a group', 'buddyboss-app' ), array( 'push_group' => 'groups' ) );
			$this->register_push_type( 'bp_groups_admin_promotion', __( 'A member is promoted to a group organizer', 'buddyboss-app' ), __( 'You are promoted to a group organizer', 'buddyboss-app' ), array( 'push_group' => 'groups' ) );
			$this->register_push_type( 'bp_groups_admin_promotion_mod', __( 'A member is promoted to a moderator', 'buddyboss-app' ), __( 'You are promoted to a group moderator', 'buddyboss-app' ), array( 'push_group' => 'groups' ) );
			$this->register_push_type( 'bp_groups_new_membership_request', __( 'A member receives a request to join a private group they organize', 'buddyboss-app' ), __( 'A member requests to join a private group you organize', 'buddyboss-app' ), array( 'push_group' => 'groups' ) );
			$this->register_push_type( 'bp_groups_request_completed', __( "A member's request to join a group has been approved", 'buddyboss-app' ), __( 'Your request to join a group has been approved', 'buddyboss-app' ), array( 'push_group' => 'groups' ) );
			$this->register_push_type( 'bp_groups_request_denied', __( "A member's request to join a group has been denied", 'buddyboss-app' ), __( 'Your request to join a group has been denied', 'buddyboss-app' ), array( 'push_group' => 'groups' ) );

			if ( function_exists( 'bp_disable_group_messages' ) && true === bp_disable_group_messages() ) {
				$this->register_push_type( 'bp_groups_send_new_message', __( 'A member receives a new group message', 'buddyboss-app' ), __( 'A group sends you a new message', 'buddyboss-app' ), array( 'push_group' => 'groups' ) );
			}

			if ( function_exists( 'bb_platform_pro' ) ) {
				$this->register_push_type( 'bp_zoom_meeting_scheduled', __( 'A Zoom meeting has been scheduled in one of your groups', 'buddyboss-app' ), __( 'A Zoom meeting has been scheduled in one of your groups', 'buddyboss-app' ), array( 'push_group' => 'groups' ) );
				$this->register_push_type( 'bp_zoom_webinar_scheduled', __( 'A Zoom webinar has been scheduled in one of your groups', 'buddyboss-app' ), __( 'A Zoom webinar has been scheduled in one of your groups', 'buddyboss-app' ), array( 'push_group' => 'groups' ) );
			}

			if ( $this->new_is_groups_notifications_available() ) {
				if ( function_exists( 'bp_is_labs_notification_preferences_support_enabled' ) && bp_is_labs_notification_preferences_support_enabled() ) {
					$this->register_push_type( 'bb_groups_details_updated', __( 'A group\'s details are updated', 'buddyboss-app' ), __( 'The details of a group you manage are updated', 'buddyboss-app' ), array( 'push_group' => 'groups' ) );
					$this->register_push_type( 'bb_groups_promoted', __( 'A member is promoted in a group', 'buddyboss-app' ), __( 'You\'re promoted in a group', 'buddyboss-app' ), array( 'push_group' => 'groups' ) );
					$this->register_push_type( 'bb_groups_new_zoom', __( 'A Zoom meeting or webinar is scheduled in a group', 'buddyboss-app' ), __( 'New meeting or webinar is scheduled in one of your groups', 'buddyboss-app' ), array( 'push_group' => 'groups' ) );
				}

				$this->register_push_type( 'bb_groups_request_accepted', __( 'A member is accepted into a group', 'buddyboss-app' ), __( 'Your request to join a group is accepted', 'buddyboss-app' ), array( 'push_group' => 'groups' ) );
				$this->register_push_type( 'bb_groups_request_rejected', __( 'A member is rejected from joining a group', 'buddyboss-app' ), __( 'Your request to join a group is rejected', 'buddyboss-app' ), array( 'push_group' => 'groups' ) );

			}
		}
	}

	/**
	 * Tells if the new groups notifications are available or not.
	 */
	public function new_is_groups_notifications_available() {
		/**
		 * We can know if the notifications are available or not by checking lab function is available or not. because
		 * all new notifications were added in same time when this function was added.
		 */
		if ( function_exists( 'bp_is_labs_notification_preferences_support_enabled' ) ) {
			return true;
		} else {
			return false;
		}

	}

	/**
	 * Link Normal Notification to Push.
	 */
	public function link_notifications() {

		if ( bbapp_is_platform_notification_preferences_legacy_mode_enabled() ) {
			// Groups.
			$this->register_push_to_normal( 'groups', 'group_invite', 'bp_groups_new_invite' );
			$this->register_push_to_normal( 'groups', 'new_membership_request', 'bp_groups_new_membership_request' );
			$this->register_push_to_normal( 'groups', 'membership_request_accepted', 'bp_groups_request_completed' );
			$this->register_push_to_normal( 'groups', 'membership_request_rejected', 'bp_groups_request_denied' );
			$this->register_push_to_normal( 'groups', 'member_promoted_to_admin', 'bp_groups_admin_promotion' );
			$this->register_push_to_normal( 'groups', 'member_promoted_to_mod', 'bp_groups_admin_promotion_mod' );

			if ( function_exists( 'bp_disable_group_messages' ) && true === bp_disable_group_messages() ) {
				$this->register_push_to_normal( 'groups', 'new_message', 'bp_groups_send_new_message' );
			}

			if ( function_exists( 'bb_platform_pro' ) ) {
				$this->register_push_to_normal( 'groups', 'zoom_meeting_created', 'bp_zoom_meeting_scheduled' );
				$this->register_push_to_normal( 'groups', 'zoom_meeting_notified', 'bp_zoom_meeting_scheduled' );
				$this->register_push_to_normal( 'groups', 'zoom_webinar_created', 'bp_zoom_webinar_scheduled' );
				$this->register_push_to_normal( 'groups', 'zoom_webinar_notified', 'bp_zoom_webinar_scheduled' );
			}

			// Modern notification.
			$this->register_push_to_normal( 'groups', 'bb_groups_details_updated', 'bb_groups_details_updated' );
			$this->register_push_to_normal( 'groups', 'bb_groups_new_invite', 'bp_groups_new_invite' );
			$this->register_push_to_normal( 'groups', 'bb_groups_promoted', 'bb_groups_promoted' );
			$this->register_push_to_normal( 'groups', 'bb_groups_new_request', 'bp_groups_new_membership_request' );
			$this->register_push_to_normal( 'groups', 'bb_groups_request_accepted', 'bb_groups_request_accepted' );
			$this->register_push_to_normal( 'groups', 'bb_groups_request_rejected', 'bb_groups_request_rejected' );
			$this->register_push_to_normal( 'groups', 'bb_groups_new_message', 'bp_groups_send_new_message' );
			$this->register_push_to_normal( 'groups', 'bb_groups_new_zoom', 'bb_groups_new_zoom' );
		}
	}

	/**
	 * Function to change push notification format.
	 *
	 * @param string $content           Notification content.
	 * @param string $component_name    Component name.
	 * @param string $component_action  Component action name.
	 * @param int    $item_id           Item id.
	 * @param int    $secondary_item_id Secondary item id.
	 * @param int    $notification_id   Notification id.
	 * @param int    $total_items       Number of total items.
	 * @param string $format            Notification format.
	 * @param string $screen            Screen name.
	 *
	 * @since 1.4.7
	 * @return mixed|string
	 * @throws \Exception Exception throws.
	 */
	public function format_notification_override( $content, $component_name, $component_action, $item_id, $secondary_item_id, $notification_id, $total_items, $format, $screen ) {
		if ( 'app_push' !== $screen ) {
			return $content;
		}

		switch ( $component_action ) {
			// Modern Notifications.
			case 'bb_groups_details_updated':
				$group_id         = $item_id;
				$group_name       = bp_get_group_name( groups_get_group( $group_id ) );
				$content['title'] = $group_name;
				$content['text']  = esc_html__( 'The group details were updated', 'buddyboss-app' );

				break;
			case 'bb_groups_promoted':
				$group_id     = $item_id;
				$group        = groups_get_group( $group_id );
				$promote_text = bp_notifications_get_meta( $notification_id, 'promoted_to', true );

				if ( $promote_text ) {
					/* translators: %1$s: Promoted role */
					$text = sprintf( esc_html__( 'Your role was changed to "%1$s"', 'buddyboss-app' ), $promote_text );
				} else {
					/* translators: group name */
					$text = sprintf( esc_html__( 'Your role in "%1$s" was changed', 'buddyboss-app' ), $group->name );
				}

				$content['title'] = $group->name;
				$content['text']  = $text;

				break;
			case 'bb_groups_new_invite':
				$group_id = $item_id;
				$group    = groups_get_group( $group_id );

				$content['title'] = $group->name;
				$content['text']  = esc_html__( 'You\'ve been invited to join this group', 'buddyboss-app' );

				break;
			case 'bb_groups_new_request':
				$group_id      = $item_id;
				$group         = groups_get_group( $group_id );
				$user_fullname = bp_core_get_user_displayname( $secondary_item_id );
				/* translators: 1. user name. */
				$text             = sprintf( esc_html__( '%1$s has requested to join this group', 'buddyboss-app' ), $user_fullname );
				$content['title'] = $group->name;
				$content['text']  = $text;

				break;
			case 'bb_groups_request_accepted':
				$group_id = $item_id;
				$group    = groups_get_group( $group_id );

				$content['title'] = $group->name;
				$content['text']  = esc_html__( 'Your request to join has been approved', 'buddyboss-app' );

				break;
			case 'bb_groups_request_rejected':
				$group_id = $item_id;
				$group    = groups_get_group( $group_id );

				$content['title'] = $group->name;
				$content['text']  = esc_html__( 'Your request to join has been denied', 'buddyboss-app' );

				break;
			case 'bb_groups_new_message':
				// Get message thread ID.
				$message      = new \BP_Messages_Message( $item_id );
				$media_ids    = bp_messages_get_meta( $item_id, 'bp_media_ids', true );
				$document_ids = bp_messages_get_meta( $item_id, 'bp_document_ids', true );
				$video_ids    = bp_messages_get_meta( $item_id, 'bp_video_ids', true );
				$gif_data     = bp_messages_get_meta( $item_id, '_gif_data', true );

				$excerpt = bp_create_excerpt(
					wp_strip_all_tags( $message->message ),
					50,
					array(
						'ending' => __( '&hellip;', 'buddyboss-app' ),
					)
				);

				if ( '&nbsp;' === $excerpt ) {
					$excerpt = '';
				}

				if ( ! empty( $secondary_item_id ) && bp_is_active( 'groups' ) && true === bp_disable_group_messages() ) {

					$group_id     = bp_messages_get_meta( $item_id, 'group_id', true ); // group id.
					$message_from = bp_messages_get_meta( $item_id, 'message_from', true ); // group.
					$group_name   = bp_get_group_name( groups_get_group( $group_id ) );

					if ( empty( $message_from ) ) {
						/* translators: %1$s user display name */
						$text = sprintf( esc_html__( '%1$s sent a new message', 'buddyboss-app' ), bp_core_get_user_displayname( $secondary_item_id ) );
					} elseif ( 'group' === $message_from ) {
						if ( ! empty( $excerpt ) ) {
							/* translators: 1. user display name 2. excerpt */
							$text = sprintf( esc_html__( '%1$s sent a new message: "%2$s"', 'buddyboss-app' ), bp_core_get_user_displayname( $secondary_item_id ), $excerpt );
						} elseif ( $media_ids ) {
							$media_ids = array_filter( explode( ',', $media_ids ) );
							if ( count( $media_ids ) > 1 ) {
								/* translators: 1. user display name 2. photos text  */
								$text = sprintf( esc_html__( '%1$s sent %2$s', 'buddyboss-app' ), bp_core_get_user_displayname( $secondary_item_id ), esc_html__( 'some photos', 'buddyboss-app' ) );
							} else {
								/* translators: 1. user display name 2. photo text */
								$text = sprintf( esc_html__( '%1$s sent %2$s', 'buddyboss-app' ), bp_core_get_user_displayname( $secondary_item_id ), esc_html__( 'a photo', 'buddyboss-app' ) );
							}
						} elseif ( $document_ids ) {
							$document_ids = array_filter( explode( ',', $document_ids ) );
							if ( count( $document_ids ) > 1 ) {
								/* translators: 1. user display name 2. documents text */
								$text = sprintf( esc_html__( '%1$s sent %2$s', 'buddyboss-app' ), bp_core_get_user_displayname( $secondary_item_id ), esc_html__( 'some documents', 'buddyboss-app' ) );
							} else {
								/* translators: 1. user display name 2. document text */
								$text = sprintf( esc_html__( '%1$s sent %2$s', 'buddyboss-app' ), bp_core_get_user_displayname( $secondary_item_id ), esc_html__( 'a document', 'buddyboss-app' ) );
							}
						} elseif ( $video_ids ) {
							$video_ids = array_filter( explode( ',', $video_ids ) );
							if ( count( $video_ids ) > 1 ) {
								/* translators: 1. user display name 2. videos text */
								$text = sprintf( esc_html__( '%1$s sent %2$s', 'buddyboss-app' ), bp_core_get_user_displayname( $secondary_item_id ), esc_html__( 'some videos', 'buddyboss-app' ) );
							} else {
								/* translators: 1. user display name 2. video text */
								$text = sprintf( esc_html__( '%1$s sent %2$s', 'buddyboss-app' ), bp_core_get_user_displayname( $secondary_item_id ), esc_html__( 'a video', 'buddyboss-app' ) );
							}
						} elseif ( empty( $gif_data ) ) {
							/* translators: 1. user display name 2. gif text */
							$text = sprintf( esc_html__( '%1$s sent %2$s', 'buddyboss-app' ), bp_core_get_user_displayname( $secondary_item_id ), esc_html__( 'a gif', 'buddyboss-app' ) );
						} else {
							/* translators: 1. user display name 2. group name */
							$text = sprintf( esc_html__( '%1$s sent a new message', 'buddyboss-app' ), bp_core_get_user_displayname( $secondary_item_id ) );
						}
					} else {
						/* translators: %1$s user display name */
						$text = sprintf( esc_html__( '%1$s sent a new message', 'buddyboss-app' ), bp_core_get_user_displayname( $secondary_item_id ) );
					}
					$content['title'] = $group_name;
					$content['text']  = $text;
				}

				break;
			case 'bb_groups_new_zoom':
				$group_id   = $item_id;
				$group      = groups_get_group( $group_id );
				$start_date = '';
				$text       = '';
				$type       = bp_notifications_get_meta( $notification_id, 'type' );
				$is_created = bp_notifications_get_meta( $notification_id, 'is_created' );

				// Check the type of zoom like is webinar or meeting.
				if ( 'meeting' === $type ) {
					$meeting = new \BP_Zoom_Meeting( $secondary_item_id );
				} else {
					$meeting = new \BP_Zoom_Webinar( $secondary_item_id );
				}

				if ( property_exists( $meeting, 'start_date' ) ) {
					$start_date = new \DateTime( $meeting->start_date );
					$start_date = $start_date->format( 'd-m-Y' );
				}

				if ( 'meeting' === $type ) {
					if ( ! empty( $start_date ) ) {
						if ( $is_created ) {
							/* translators: %s: The meeting start date */
							$text = sprintf( __( 'New meeting scheduled for %1$s', 'buddyboss-app' ), $start_date );
						} else {
							/* translators: %s: The meeting start date */
							$text = sprintf( __( 'Update meeting scheduled for %1$s', 'buddyboss-app' ), $start_date );
						}
					} else {
						if ( $is_created ) {
							$text = __( 'New meeting scheduled', 'buddyboss-app' );
						} else {
							$text = __( 'Update meeting scheduled', 'buddyboss-app' );
						}
					}
				} elseif ( 'webinar' === $type ) {
					if ( ! empty( $start_date ) ) {
						if ( $is_created ) {
							/* translators: %s: The meeting start date */
							$text = sprintf( __( 'New webinar scheduled for %1$s', 'buddyboss-app' ), $start_date );
						} else {
							/* translators: %s: The meeting start date */
							$text = sprintf( __( 'Update webinar scheduled for %1$s', 'buddyboss-app' ), $start_date );
						}
					} else {
						if ( $is_created ) {
							$text = esc_html__( 'New webinar scheduled', 'buddyboss-app' );
						} else {
							$text = esc_html__( 'Update webinar scheduled', 'buddyboss-app' );
						}
					}
				}

				$content['title'] = $group->name;
				$content['text']  = $text;
				break;
			// Legacy Notifications.
			case 'new_membership_request':
			case 'membership_request_accepted':
			case 'membership_request_rejected':
			case 'member_promoted_to_admin':
			case 'member_promoted_to_mod':
			case 'group_invite':
				$group_id         = $item_id;
				$group_name       = bp_get_group_name( groups_get_group( $group_id ) );
				$content['title'] = $group_name;
				break;
			case 'new_message':
				if ( ! empty( $secondary_item_id ) && bp_is_active( 'groups' ) && true === bp_disable_group_messages() ) {
					$group_id         = $item_id;
					$group_name       = bp_get_group_name( groups_get_group( $group_id ) );
					$content['title'] = $group_name;
				}
				break;
		}

		return $content;
	}
}
