<?php
/**
 * Holds forum push notification functionality.
 *
 * @package BuddyBossApp\Integrations\BuddyBossPlatform\Push
 */

namespace BuddyBossApp\Integrations\BuddyBossPlatform\Push;

use BuddyBossApp\Notification\IntegrationAbstract;

/**
 * Forums component push notification support.
 * Class ForumsNotification
 *
 * @package BuddyBossApp\Integrations\BuddyBossPlatform\Push
 */
class ForumsNotification extends IntegrationAbstract {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Load methods.
	 */
	public function load() {
		$this->push_types();
		$this->link_notifications();
	}

	/**
	 * Register Subscriptions Types
	 */
	public function push_types() {
		if ( bbapp_is_platform_notification_preferences_legacy_mode_enabled() ) {
			// Forum.
			$this->register_push_group( 'forum', __( 'Forums', 'buddyboss-app' ) );
			$this->register_push_type( 'bbp_new_reply', __( "A member receives a new reply to a discussion they've created", 'buddyboss-app' ), __( 'New reply on discussion you created', 'buddyboss-app' ), array( 'push_group' => 'forum' ) );
			/* translators: %s: User display name */
			$this->register_push_type( 'bbp_new_at_mention', __( 'A discussion is created in a forum a member is subscribed to', 'buddyboss-app' ), sprintf( __( 'A member mentions you in a discussion using "@%s"', 'buddyboss-app' ), bp_core_get_user_displayname( get_current_user_id() ) ), array( 'push_group' => 'forum' ) );

			if ( function_exists( 'bp_is_labs_notification_preferences_support_enabled' ) && bp_is_labs_notification_preferences_support_enabled() ) {
				$this->register_push_type( 'bb_forums_subscribed_discussion', __( 'A new discussion in a forum a member is subscribed to', 'buddyboss-app' ), __( 'New discussion in a forum you\'re subscribed to', 'buddyboss-app' ), array( 'push_group' => 'forum' ) );
				$this->register_push_type( 'bb_forums_subscribed_reply', __( 'A new reply in a discussion a member is subscribed to', 'buddyboss-app' ), __( 'New reply in a discussion you\'re subscribed to', 'buddyboss-app' ), array( 'push_group' => 'forum' ) );
			}
		}
	}

	/**
	 * Link Normal Notification to Push.
	 */
	public function link_notifications() {
		if ( bbapp_is_platform_notification_preferences_legacy_mode_enabled() ) {
			// Forum.
			$this->register_push_to_normal( 'forums', 'bbp_new_reply', 'bbp_new_reply' );
			$this->register_push_to_normal( 'forums', 'bbp_new_at_mention', 'bbp_new_at_mention' );

			// Modern notification.
			$this->register_push_to_normal( 'forums', 'bb_forums_subscribed_discussion', 'bb_forums_subscribed_discussion' );
			$this->register_push_to_normal( 'forums', 'bb_forums_subscribed_reply', 'bb_forums_subscribed_reply' );
		}
	}

	/**
	 * Function to change push notification format.
	 *
	 * @param string|array $content           Notification content.
	 * @param string       $component_name    Component name.
	 * @param string       $component_action  Component action name.
	 * @param int          $item_id           Item id.
	 * @param int          $secondary_item_id Secondary item id.
	 * @param int          $notification_id   Notification id.
	 * @param int          $total_items       Number of total items.
	 * @param string       $format            Notification format.
	 * @param string       $screen            Screen name.
	 *
	 * @since 1.4.7
	 * @return mixed|string
	 */
	public function format_notification_override( $content, $component_name, $component_action, $item_id, $secondary_item_id, $notification_id, $total_items, $format, $screen ) {
		if ( 'app_push' !== $screen ) {
			return $content;
		}

		switch ( $component_action ) {
			// Modern Notifications.
			case 'bb_forums_subscribed_discussion':
				$topic_id = bbp_get_reply_topic_id( $item_id );
				$group    = ( ( function_exists( 'bbp_is_forum_group_forum' ) && bbp_get_topic_forum_id( $topic_id ) && bbp_is_forum_group_forum( bbp_get_topic_forum_id( $topic_id ) ) && function_exists( 'groups_get_group' ) ) ? ( ! empty( bbp_get_forum_group_ids( bbp_get_topic_forum_id( $topic_id ) ) ) ? groups_get_group( current( bbp_get_forum_group_ids( bbp_get_topic_forum_id( $topic_id ) ) ) ) : '' ) : '' );

				if ( ! empty( $group ) ) {
					$group_name       = bp_get_group_name( $group );
					$content['title'] = $group_name;
				} else {
					$form_id          = bbp_get_topic_forum_id( $topic_id );
					$content['title'] = get_the_title( $form_id );
				}
				break;
			case 'bb_forums_subscribed_reply':
				$topic_id = bbp_get_topic_id( $item_id );
				$group    = ( ( function_exists( 'bbp_is_forum_group_forum' ) && bbp_get_topic_forum_id( $topic_id ) && bbp_is_forum_group_forum( bbp_get_topic_forum_id( $topic_id ) ) && function_exists( 'groups_get_group' ) ) ? ( ! empty( bbp_get_forum_group_ids( bbp_get_topic_forum_id( $topic_id ) ) ) ? groups_get_group( current( bbp_get_forum_group_ids( bbp_get_topic_forum_id( $topic_id ) ) ) ) : '' ) : '' );

				if ( ! empty( $group ) ) {
					$group_name       = bp_get_group_name( $group );
					$content['title'] = $group_name;
				} else {
					$form_id          = bbp_get_topic_forum_id( $topic_id );
					$content['title'] = get_the_title( $form_id );
				}
				break;
			case 'bbp_new_at_mention': // MENTIONS.
				if ( bbapp_is_platform_notification_preferences_legacy_mode_enabled() ) {
					$topic_id = bbp_get_reply_topic_id( $item_id );

					if ( empty( $topic_id ) ) {
						$topic_id = $item_id;
					}

					$topic_title = '"' . bp_create_excerpt( wp_strip_all_tags( bbp_get_topic_title( $topic_id ) ), 50, array( 'ending' => __( '&hellip;', 'buddyboss-app' ) ) ) . '"';
					$topic_title = str_replace( '&hellip;"', '&hellip;', $topic_title );

					if ( ! empty( $secondary_item_id ) ) {
						/* translators: %1$s: Topic title  */
						$content['text']  = sprintf( __( 'Mentioned you in %1$s', 'buddyboss-app' ), $topic_title );
						$content['title'] = bp_core_get_user_displayname( $secondary_item_id );
					} else {
						/* translators: %1$d: Number of mention, %1$s: Topic title  */
						$content['text']  = sprintf( __( 'You have %1$d new mention to %1$s', 'buddyboss-app' ), $topic_title );
						$content['title'] = '';
					}
				}
				break;
			// Legacy Notifications.
			case 'bbp_new_topic':
				$topic_id         = bbp_get_topic_id( $item_id );
				$topic_title      = bbp_get_topic_title( $topic_id );
				$content['title'] = $topic_title;
				break;
			case 'bbp_new_reply':
				$topic_id         = bbp_get_reply_topic_id( $item_id );
				$topic_title      = bbp_get_topic_title( $topic_id );
				$content['title'] = $topic_title;
				break;
		}

		return $content;
	}
}
