<?php
/**
 * Holds functionality related activity push notifications.
 *
 * @package BuddyBossApp\Integrations\BuddyBossPlatform\Push
 */

namespace BuddyBossApp\Integrations\BuddyBossPlatform\Push;

use BuddyBossApp\Notification\IntegrationAbstract;

/**
 * Activity component push notification support.
 * Class ActivityNotification
 *
 * @package BuddyBossApp\Integrations\BuddyBossPlatform\Push
 */
class ActivityNotification extends IntegrationAbstract {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Load methods.
	 */
	public function load() {
		$this->push_types();
		$this->link_notifications();

	}

	/**
	 * Register Subscriptions Types
	 */
	public function push_types() {
		if ( bbapp_is_platform_notification_preferences_legacy_mode_enabled() ) {
			// Activity.
			$this->register_push_group( 'activity', __( 'Activity Feed', 'buddyboss-app' ) );
			/* translators: %s: User display name. */
			$this->register_push_type( 'bp_activity_new_at_mention', __( 'A member is mentioned in another member’s update', 'buddyboss-app' ), sprintf( __( 'A member mentions you in an update using "@%s"', 'buddyboss-app' ), bp_core_get_user_displayname( get_current_user_id() ) ), array( 'push_group' => 'activity' ) );
			$this->register_push_type( 'bp_activity_new_reply', __( 'A member receives a reply to an update or comment they’ve posted', 'buddyboss-app' ), __( "A member replies to an update or comment you've posted", 'buddyboss-app' ), array( 'push_group' => 'activity' ) );
		}
	}

	/**
	 * Link Normal Notification to Push.
	 */
	public function link_notifications() {
		if ( bbapp_is_platform_notification_preferences_legacy_mode_enabled() ) {
			// Activity.
			$this->register_push_to_normal( 'activity', 'new_at_mention', 'bp_activity_new_at_mention' );
			$this->register_push_to_normal( 'activity', 'comment_reply', 'bp_activity_new_reply' );
			$this->register_push_to_normal( 'activity', 'update_reply', 'bp_activity_new_reply' );

			// Modern notification.
			$this->register_push_to_normal( 'activity', 'bb_new_mention', 'bp_activity_new_at_mention' );
			$this->register_push_to_normal( 'activity', 'bb_activity_comment', 'bp_activity_new_reply' );
		}
	}

	/**
	 * Function to change push notification format.
	 *
	 * @param string|array $content           Notification content.
	 * @param string       $component_name    Component name.
	 * @param string       $component_action  Component action name.
	 * @param int          $item_id           Item id.
	 * @param int          $secondary_item_id Secondary item id.
	 * @param int          $notification_id   Notification id.
	 * @param int          $total_items       Number of total items.
	 * @param string       $format            Notification format.
	 * @param string       $screen            Screen name.
	 *
	 * @since 1.4.7
	 *
	 * @return mixed|string
	 */
	public function format_notification_override( $content, $component_name, $component_action, $item_id, $secondary_item_id, $notification_id, $total_items, $format, $screen ) {
		if ( 'app_push' !== $screen ) {
			return $content;
		}

		$user_id = $secondary_item_id;
		switch ( $component_action ) {
			// Modern Notifications.
			case 'bb_activity_comment':
				$notification_type = bp_notifications_get_meta( $notification_id, 'type', true );
				$text              = '';
				if ( $notification_type ) {
					if ( 'activity_comment' === $notification_type ) {
						$notification_type_html = esc_html__( 'comment', 'buddyboss-app' );
					} elseif ( 'post_comment' === $notification_type || 'activity_post' === $notification_type ) {
						$notification_type_html = esc_html__( 'post', 'buddyboss-app' );
					}
				}

				$activity         = new \BP_Activity_Activity( $item_id );
				$activity_excerpt = bp_create_excerpt(
					wp_strip_all_tags( $activity->content ),
					50,
					array(
						'ending' => __( '&hellip;', 'buddyboss-app' ),
					)
				);
				if ( '&nbsp;' === $activity_excerpt ) {
					$activity_excerpt = '';
				}
				if ( empty( $activity_excerpt ) && function_exists( 'bp_blogs_activity_comment_content_with_read_more' ) ) {
					$activity_excerpt = bp_blogs_activity_comment_content_with_read_more( '', $activity );

					$activity_excerpt = bp_create_excerpt(
						wp_strip_all_tags( $activity_excerpt ),
						50,
						array(
							'ending' => __( '&hellip;', 'buddyboss-app' ),
						)
					);
					if ( '&nbsp;' === $activity_excerpt ) {
						$activity_excerpt = '';
					}
				}

				$activity_excerpt = '"' . $activity_excerpt . '"';
				$activity_excerpt = str_replace( '&hellip;"', '&hellip;', $activity_excerpt );
				$activity_excerpt = str_replace( '&#8203;', '', $activity_excerpt );
				$activity_excerpt = str_replace( '""', '', $activity_excerpt );
				$activity_excerpt = stripslashes_deep( $activity_excerpt );

				$media_ids    = bp_activity_get_meta( $activity->id, 'bp_media_ids', true );
				$document_ids = bp_activity_get_meta( $activity->id, 'bp_document_ids', true );
				$video_ids    = bp_activity_get_meta( $activity->id, 'bp_video_ids', true );
				$gif_data     = bp_activity_get_meta( $activity->id, '_gif_data', true );

				if ( ! empty( $notification_type_html ) ) {
					if ( ! empty( $activity_excerpt ) ) {
						$text = sprintf(
						/* translators: 1: Activity type, 2: Activity content. */
							__( 'Replied to your %1$s: %2$s', 'buddyboss-app' ),
							$notification_type_html,
							$activity_excerpt
						);
					} elseif ( $media_ids ) {
						$media_ids = array_filter( ! is_array( $media_ids ) ? explode( ',', $media_ids ) : $media_ids );
						if ( count( $media_ids ) > 1 ) {
							$text = sprintf(
							/* translators: Activity type. */
								__( 'Replied to your %s: some photos', 'buddyboss-app' ),
								$notification_type_html
							);
						} else {
							$text = sprintf(
							/* translators: Activity type. */
								__( 'Replied to your %s: a photo', 'buddyboss-app' ),
								$notification_type_html
							);
						}
					} elseif ( $document_ids ) {
						$document_ids = array_filter( ! is_array( $document_ids ) ? explode( ',', $document_ids ) : $document_ids );
						if ( count( $document_ids ) > 1 ) {
							$text = sprintf(
							/* translators: Activity type. */
								__( 'Replied to your %s: some documents', 'buddyboss-app' ),
								$notification_type_html
							);
						} else {
							$text = sprintf(
							/* translators: Activity type. */
								__( 'Replied to your %s: a document', 'buddyboss-app' ),
								$notification_type_html
							);
						}
					} elseif ( $video_ids ) {
						$video_ids = array_filter( ! is_array( $video_ids ) ? explode( ',', $video_ids ) : $video_ids );
						if ( count( $video_ids ) > 1 ) {
							$text = sprintf(
							/* translators: Activity type. */
								__( 'Replied to your %s: some videos', 'buddyboss-app' ),
								$notification_type_html
							);
						} else {
							$text = sprintf(
							/* translators: Activity type. */
								__( 'Replied to your %s: a video', 'buddyboss-app' ),
								$notification_type_html
							);
						}
					} elseif ( ! empty( $gif_data ) ) {
						$text = sprintf(
						/* translators: Activity type. */
							__( 'Replied to your %s: a gif', 'buddyboss-app' ),
							$notification_type_html
						);
					} else {
						$text = sprintf(
						/* translators: 1: Activity type. */
							__( 'Replied to your %1$s', 'buddyboss-app' ),
							$notification_type_html
						);
					}
				} else {
					if ( ! empty( $activity_excerpt ) ) {
						$text = sprintf(
						/* translators: 1: Activity content. */
							__( 'Replied: %1$s', 'buddyboss-app' ),
							$activity_excerpt
						);
					} elseif ( $media_ids ) {
						$media_ids = array_filter( ! is_array( $media_ids ) ? explode( ',', $media_ids ) : $media_ids );
						if ( count( $media_ids ) > 1 ) {
							$text = __( 'Replied: some photos', 'buddyboss-app' );
						} else {
							$text = __( 'Replied: a photo', 'buddyboss-app' );
						}
					} elseif ( $document_ids ) {
						$document_ids = array_filter( ! is_array( $document_ids ) ? explode( ',', $document_ids ) : $document_ids );
						if ( count( $document_ids ) > 1 ) {
							$text = __( 'Replied: some documents', 'buddyboss-app' );
						} else {
							$text = __( 'Replied: a document', 'buddyboss-app' );
						}
					} elseif ( $video_ids ) {
						$video_ids = array_filter( ! is_array( $video_ids ) ? explode( ',', $video_ids ) : $video_ids );
						if ( count( $video_ids ) > 1 ) {
							$text = __( 'Replied: some videos', 'buddyboss-app' );
						} else {
							$text = __( 'Replied: a video', 'buddyboss-app' );
						}
					} elseif ( ! empty( $gif_data ) ) {
						$text = __( 'Replied: a gif', 'buddyboss-app' );
					} else {
						$text = __( 'Replied', 'buddyboss-app' );
					}
				}

				$content['title'] = bp_core_get_user_displayname( $user_id );
				if ( ! empty( $text ) ) {
					$content['text'] = $text;
				}
				break;
			case 'bb_new_mention': // MENTIONS.
				if ( in_array( $component_name, array( 'activity', 'forums', 'members', 'core' ), true ) ) {
					$notification_type_html = '';
					$user_fullname          = bp_core_get_user_displayname( $user_id );
					$notification_type      = bp_notifications_get_meta( $notification_id, 'type', true );
					if ( $notification_type ) {
						if ( 'post_comment' === $notification_type ) {
							$notification_type_html = esc_html__( 'comment', 'buddyboss-app' );
						} elseif ( 'activity_comment' === $notification_type || 'activity_post' === $notification_type ) {
							$notification_type_html = esc_html__( 'post', 'buddyboss-app' );
						} elseif ( 'forum_reply' === $notification_type || 'forum_topic' === $notification_type ) {
							$notification_type_html = esc_html__( 'discussion', 'buddyboss-app' );
						}
					}

					if ( ! empty( $notification_type_html ) ) {
						$text = sprintf(
						/* translators: 1 Activity type. */
							esc_html__( 'Mentioned you in a %1$s', 'buddyboss-app' ),
							$notification_type_html
						);
					} else {
						$text = sprintf(
						/* translators: %s: User full name. */
							esc_html__( '%1$s mentioned you', 'buddyboss-app' ),
							$user_fullname
						);
					}
					$content['title'] = $user_fullname;
					$content['text']  = $text;
				}
				break;
			case 'new_at_mention':
				$user_fullname    = bp_core_get_user_displayname( $user_id );
				$content['title'] = '';
				/* translators: %1$s: User full name. */
				$content['text'] = sprintf( __( '%1$s mentioned you', 'buddyboss-app' ), $user_fullname );
				break;
			case 'update_reply':
			case 'comment_reply':
				$content['title'] = bp_core_get_user_displayname( $user_id );
				break;
		}

		return $content;
	}
}
