<?php
/**
 * Holds deepliking functionality related Platform.
 *
 * @package BuddyBossApp\Integrations\BuddyBossPlatform
 */

namespace BuddyBossApp\Integrations\BuddyBossPlatform;

use BuddyBossApp\DeepLinking\Type\TypeAbstract;

/**
 * Deeplink class for Platform.
 */
class DeepLinking extends TypeAbstract {

	/**
	 * Url parameters.
	 *
	 * @var array $url_data
	 */
	private static $url_data = array();

	/**
	 * Parse BuddyPress url
	 *
	 * @param string $url Deeplink url.
	 *
	 * @return array|mixed|null
	 */
	public function parse( $url ) {
		$data = $this->get_url_data( $url );

		if ( ! empty( $data ) ) {
			if ( bbapp_is_lite_live_app() ) {
				return array(
					'action'    => 'open_restricted',
					'namespace' => 'core',
					'item_id'   => '403',
					'url'       => $url,
				);
			}

			if ( bbapp_is_learndash_enabled() ) {
				$data['component'] = $this->bbapp_ld_permalinks( $data['component'] );
			}

			if ( function_exists( 'bp_is_active' ) && bp_is_active( 'forums' ) ) {
				$data['component']  = $this->bbapp_forum_permalinks( $data['component'] );
				$data['action']     = $this->bbapp_forum_permalinks( $data['action'] );
				$data['sub_action'] = $this->bbapp_forum_permalinks( $data['sub_action'] );
			}

			switch ( $data['component'] ) {
				case 'groups':
					$response = array(
						'action'    => 'open_' . $data['component'],
						'namespace' => 'buddypress',
						'url'       => $url,
					);

					if ( ! empty( $data['group_topic_id'] ) ) {
						// Handle Group Forum Topic link.
						$response['action']       = $response['action'] . '_topic';
						$response['item_id']      = $data['group_topic_id'];
						$response['_link_action'] = 'bbpress_topic';
					} elseif ( ! empty( $data['group_forum_id'] ) ) {
						// Handle Group Forum link.
						$response['action']       = $response['action'] . '_forum';
						$response['item_id']      = $data['group_forum_id'];
						$response['_link_action'] = 'bbpress_forum';
					} elseif ( ! empty( $data['group_id'] ) ) {
						// Handle Group link.
						$response['item_id']      = $data['group_id'];
						$response['_link_action'] = 'buddypress_group';
					}
					break;
				default:
					$response = array(
						'action'    => 'open_' . $data['component'],
						'namespace' => 'buddypress',
						'url'       => $url,
					);
			}

			if ( ! empty( $data['user_id'] ) ) {
				$response ['action']          = 'open_member_' . $data['component'];
				$response['item_id']          = $data['user_id'];
				$response['_link_action']     = 'buddypress_member';
				$response['_sub_link_action'] = 'buddypress_' . $data['component'];
				$is_component_active          = bp_is_active( $data['component'] );

				if ( function_exists( 'bp_ld_sync' ) && bp_ld_sync()->settings->get( 'course.courses_visibility' ) && 'courses' === $data['component'] ) {
					$is_component_active = true;
				}

				if ( in_array(
					$data['component'],
					array(
						'settings',
						'messages',
						'notifications',
						'courses',
					),
					true
				) && ( ! $is_component_active || get_current_user_id() !== $data['user_id'] ) ) {

					$response ['action'] = 'open_member_profile';

					unset( $response['_sub_link_action'] );
				}
			}

			if ( ! empty( $data['activity_id'] ) ) {
				$response ['action']      = 'open_' . $data['component'];
				$response['item_id']      = $data['activity_id'];
				$response['_link_action'] = 'buddypress_activity';
			}

			if ( ! empty( $data['action'] ) ) {
				$response['tab'] = $data['action'];

				if ( isset( $data['group_id'] ) ) {
					$group = groups_get_group( $data['group_id'] );

					if ( ! empty( $data['group_id'] ) && bp_get_group_permalink( $group ) === $url ) {
						$response['tab'] = 'app_home';
					}
				}
			}

			if ( ! empty( $data['sub_action'] ) ) {
				$response['sub_tab'] = $data['sub_action'];
			}

			// If message single thread deeplinking.
			if ( 'messages' === $data['component'] && ( bp_is_active( $data['component'] ) && get_current_user_id() === (int) $data['user_id'] ) ) {
				$response['action'] = ! empty( $data['action'] ) && in_array( $data['action'], array( 'view', 'compose' ), true ) ? $response['action'] . '_thread' : $response['action'];

				if ( ! empty( $data['thread_recipients'] ) ) {
					$response['thread_recipients'] = $data['thread_recipients'];
				}
			}

			/**
			 * Filter BuddyPress deep linking data
			 */
			$response = apply_filters( 'bbapp_deeplinking_buddypress_' . $data['component'], $response, $data );
			$response = apply_filters( 'bbapp_deeplinking_buddypress', $response, $data );

			return $response;
		}

		// If activity is front page that time blog post should be open post url.
		$url_components = wp_parse_url( $url );

		if ( isset( $url_components['query'] ) ) {
			parse_str( $url_components['query'], $params );

			if (
				isset( $params['document_type'] ) &&
				'document' === $params['document_type'] &&
				bbapp_is_lite_live_app()
			) {
				return array(
					'action'    => 'open_restricted',
					'namespace' => 'core',
					'item_id'   => '403',
					'url'       => $url,
				);
			}
		}

		return null;
	}

	/**
	 * Get BuddyPress data from url
	 * For Getting BuddyPress Global data we set up virtual url and then call BuddyPress function to setup globals data
	 * Once BuddyPress global data set we are using that data to determine whether given url is BuddyPress ulr or not
	 *
	 * @param string $url Deepling url.
	 *
	 * @return array
	 */
	public function get_url_data( $url ) {
		// Check whether statics data already set or not.
		if ( empty( self::$url_data[ $url ] ) ) {
			// Check whether Transient Cache set or not.
			$hash_url   = md5( $url );
			$cache_data = get_transient( '_bbapp_deeplink_bp_url_data' );

			if ( empty( $cache_data ) || ! isset( $cache_data ) ) {
				$cache_data = array();
			}

			$cache_data = array();

			if ( empty( $cache_data ) || ( is_array( $cache_data ) && ! isset( $cache_data[ $hash_url ] ) ) ) {
				$parse_url_data = wp_parse_url( $url, PHP_URL_HOST );

				/**
				 * Filter to add shorten url services.
				 *
				 * @param array $shorten_url_services Array of shorten url services.
				 *
				 * @since 2.2.80
				 */
				$shorten_url_services = apply_filters(
					'bbapp_core_parse_url_shorten_url_provider',
					array(
						'bit.ly',
						'snip.ly',
						'rb.gy',
						'tinyurl.com',
						'tiny.one',
						'rotf.lol',
						'b.link',
						'4ubr.short.gy',
						'shorturl.at',
					)
				);

				if ( in_array( $parse_url_data, $shorten_url_services, true ) ) {
					$response = wp_safe_remote_get(
						$url,
						array(
							'stream'  => true,
							'headers' => array(
								'user-agent' => 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10.15; rv:71.0) Gecko/20100101 Firefox/71.0',
							),
						),
					);

					if (
						! is_wp_error( $response ) &&
						! empty( $response['http_response']->get_response_object()->url ) &&
						$response['http_response']->get_response_object()->url !== $url
					) {
						$new_url = $response['http_response']->get_response_object()->url;

						if ( filter_var( $new_url, FILTER_VALIDATE_URL ) ) {
							$url = $new_url;
						}
					}
				}

				$url     = preg_replace( '|https?\://[^/]+/|', '/', $url );
				$tempurl = ! empty( $_SERVER['REQUEST_URI'] ) ? bbapp_input_clean( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

				$_SERVER['REQUEST_URI'] = $url;
				bp_core_set_uri_globals();
				bp_setup_globals();
				remove_action( 'bp_setup_canonical_stack', 'bp_late_include', 20 );
				bp_setup_canonical_stack();
				add_action( 'bp_setup_canonical_stack', 'bp_late_include', 20 );
				$_SERVER['REQUEST_URI'] = $tempurl;

				$component = bp_current_component();

				if ( $this->bb_component_subpage( $url ) ) {
					self::$url_data[ $url ] = array();
				} elseif ( ! empty( $component ) ) {
					if ( $this->bb_deeplink_member_profile( $url ) ) {
						$component = 'profile';
					}

					$data = array(
						'component'  => $component,
						'action'     => bp_current_action(),
						'sub_action' => bp_action_variable( 0 ),
						'user_id'    => bp_displayed_user_id(),
					);

					if ( 'groups' === $component && bp_is_single_item() ) {
						$data['group_id'] = bp_get_current_group_id();

						// Handle Group Forum and it's topic links.
						if ( isset( $data['action'] ) && ( 'forum' === $data['action'] || get_option( '_bbp_forum_slug', 'forum' ) === $data['action'] ) ) {
							if ( isset( $data['sub_action'] ) && ( 'topic' === $data['sub_action'] || get_option( '_bbp_topic_slug', 'discussion' ) === $data['sub_action'] ) ) {
								$topic                  = get_posts(
									array(
										'name'        => bp_action_variable( 1 ),
										'post_status' => 'publish',
										'post_type'   => bbp_get_topic_post_type(),
										'numberposts' => 1,
										'fields'      => 'ids',
									)
								);
								$data['group_topic_id'] = ( ! empty( $topic ) ? current( $topic ) : 0 );
							}

							$forum_ids              = bbp_get_group_forum_ids( bp_get_current_group_id() );
							$forum_id               = array_shift( $forum_ids );
							$data['group_forum_id'] = $forum_id;
						}
					}

					if ( 'activity' === $component && bp_is_single_activity() && ! bp_is_current_action( 'p' ) ) {
						$data['activity_id'] = bp_current_action();
					} elseif ( 'messages' === $component ) {
						// Extract the recipient from the URL if available.
						parse_str( wp_parse_url( $url, PHP_URL_QUERY ), $query_params );

						$recipient = ! empty( $query_params['r'] ) ? $query_params['r'] : '';

						if ( ! empty( $recipient ) ) {
							$recipient_id = bp_core_get_userid_from_nicename( $recipient );
							$thread_id    = \BP_Messages_Message::get_existing_thread( array( $recipient_id ), get_current_user_id() );

							if ( $thread_id ) {
								$data['sub_action']        = $thread_id;
								$data['action']            = 'view';
								$thread_recipients         = \BP_Messages_Thread::get_recipients_for_thread( $thread_id );
								$thread_recipients         = wp_list_pluck( $thread_recipients, 'user_id' );
								$data['thread_recipients'] = array_values( $thread_recipients );
							} else {
								$data['sub_action']        = 'new';
								$data['thread_recipients'] = array_values( array( $recipient_id, get_current_user_id() ) );
							}
						} else {
							$thread_recipients         = \BP_Messages_Thread::get_recipients_for_thread( (int) $data['sub_action'] );
							$thread_recipients         = wp_list_pluck( $thread_recipients, 'user_id' );
							$data['thread_recipients'] = array_values( $thread_recipients );

						}
					} elseif ( 'activity' === $component && is_numeric( bp_action_variable( 0 ) ) && bp_is_current_action( 'p' ) ) {
						$data['activity_id'] = bp_action_variable( 0 );
					} elseif ( get_option( 'show_on_front' ) === 'page' && get_option( 'page_on_front' ) ) {
						// If activity is front page that time blog post should be open post url.
						$url_components = wp_parse_url( $url );

						if ( isset( $url_components['query'] ) ) {
							parse_str( $url_components['query'], $params );

							if ( isset( $params['p'] ) ) {
								$data = array(); // if we add blank array then the screen take open_post feature.
							}

							if ( isset( $params['document_type'] ) && 'document' === $params['document_type'] ) {
								$data = array(); // if we add blank array then the screen take open_page feature.
							}
						}
					}
					// set transient data.
					$cache_data[ $hash_url ] = $data;
					set_transient( '_bbapp_deeplink_bp_url_data', $cache_data, 12 * HOUR_IN_SECONDS );

					// set data to static variable.
					self::$url_data[ $url ] = $data;
				} else {
					$data = $this->bb_me_screen( $url );
					if ( ! empty( $data ) ) {
						// set data to static variable.
						self::$url_data[ $url ] = $data;
					}
				}

				/* Reset buddypress data */
				buddypress()->unfiltered_uri        = '';
				buddypress()->unfiltered_uri_offset = 0;
				buddypress()->current_component     = '';
				buddypress()->displayed_user        = '';
				buddypress()->displayed_user        = new \stdClass();
				buddypress()->current_member_type   = '';
				buddypress()->current_action        = '';
				buddypress()->current_item          = '';
				buddypress()->action_variables      = '';
			} else {
				// set transient data to static variable.
				self::$url_data[ $url ] = $cache_data[ $hash_url ];
			}
		}

		// Check Data is set or empty.
		if ( ! empty( self::$url_data[ $url ] ) ) {
			return self::$url_data[ $url ];
		}

		return array();
	}

	/**
	 * BB me screens deeplinking.
	 * - /me/groups to /groups
	 * - /me to /members
	 *
	 * @param string $url Deeplink url.
	 *
	 * @return array
	 */
	public function bb_me_screen( $url = '' ) {
		$data      = array();
		$url_param = explode( '/', $url );

		foreach ( $url_param as $key => $value ) {
			if ( empty( $value ) ) {
				unset( $url_param[ $key ] );
			}
		}

		$url_param = array_values( $url_param );

		if ( ! empty( $url_param ) ) {
			if ( ( isset( $url_param[0] ) && 'me' === $url_param[0] ) && isset( $url_param[1] ) && 'groups' === $url_param[1] ) {
				$data = array(
					'component'  => 'groups',
					'action'     => '',
					'sub_action' => '',
					'user_id'    => bp_displayed_user_id(),
				);
			} elseif ( isset( $url_param[0] ) && 'me' === $url_param[0] ) {
				$data = array(
					'component'  => 'members',
					'action'     => '',
					'sub_action' => '',
					'user_id'    => bp_displayed_user_id(),
				);
			}
		}

		return $data;
	}

	/**
	 * Checks if the provided URL is a child page of a BuddyPress component.
	 *
	 * @param string $url The deeplink URL to be checked.
	 *
	 * @since 2.1.50
	 *
	 * @return bool
	 */
	public function bb_component_subpage( $url ) {
		$url_post = get_post( url_to_postid( $url ) );

		if ( empty( $url_post ) ) {
			return false;
		}

		return 'page' === $url_post->post_type && ! empty( bp_current_component() ) && $url_post->post_parent > 0;
	}

	/**
	 * Checks if the provided URL is a member profile URL.
	 *
	 * @param string $url The deeplink URL to be checked.
	 *
	 * @since 2.2.00
	 *
	 * @return bool
	 */
	public function bb_deeplink_member_profile( $url ) {
		$members_dir = preg_replace( '|https?\://[^/]+/|', '/', bp_get_members_directory_permalink() );

		if ( 0 !== strpos( $url, $members_dir ) ) {
			return;
		}

		$username = trim( str_replace( $members_dir, '', $url ), '/' );

		if ( ! username_exists( $username ) ) {
			return;
		}

		return true;
	}

	/**
	 * LearnDash permalinks.
	 *
	 * @param string $component_name Component name.
	 *
	 * @since 2.3.20
	 * @return string
	 */
	public function bbapp_ld_permalinks( $component_name ) {
		$ld_permalink_options = get_option( 'learndash_settings_permalinks', array() );
		$ld_permalink_options = array_flip( $ld_permalink_options );

		if ( ! empty( $ld_permalink_options[ $component_name ] ) ) {
			return $ld_permalink_options[ $component_name ];
		}

		return $component_name;
	}

	/**
	 * Get Permalinks settings.
	 *
	 * @param string $action_name Action name.
	 *
	 * @since 2.3.20
	 * @return string
	 */
	public function bbapp_forum_permalinks( $action_name ) {
		$results = array(
			// Single forum slugs.
			'forum'         => get_option( '_bbp_forum_slug', 'forum' ),
			'topic'         => get_option( '_bbp_topic_slug', 'discussions' ),
			'topic_tag'     => get_option( '_bbp_topic_tag_slug', 'topic-tag' ),
			'view'          => get_option( '_bbp_view_slug', 'view' ),
			'reply'         => get_option( '_bbp_reply_slug', 'reply' ),
			'search'        => get_option( '_bbp_search_slug', 'search' ),

			// Forum Profile Slugs.
			'replies'       => bbp_get_reply_archive_slug(),
			'favorites'     => bbp_get_user_favorites_slug(),
			'subscriptions' => bbp_get_user_subscriptions_slug(),
		);

		$results = array_flip( $results );

		if ( ! empty( $results[ $action_name ] ) ) {
			return $results[ $action_name ];
		}

		return $action_name;
	}
}
