<?php
/**
 * DeepLinking for BuddyPress type.
 *
 * @package BuddyBossApp\Integrations\BuddyBossPlatform
 */

namespace BuddyBossApp\Integrations\BuddyBossPlatform;

use BuddyBossApp\DeepLinking\Type\TypeAbstract;
use WP_Post;
use WP_REST_Response;

/**
 * Generate deepLink for the BuddyPress type.
 */
class BbPressDeepLinking extends TypeAbstract {

	/**
	 * List of post types.
	 *
	 * @var array $arr_post_types
	 */
	public $arr_post_types = array();

	/**
	 * Construct function for the class.
	 */
	public function __construct() {
		$this->init();
		parent::__construct();
		add_filter( 'bbapp_deeplinking_cpt_type', array( $this, 'update_cpt_object' ), 10, 3 );
		add_filter( 'bbapp_deeplinking_cpt_namespace', array( $this, 'update_cpt_namespace' ), 10, 2 );
		add_filter( 'bbapp_deeplinking_details', array( $this, 'get_mp_restricted_for_deeplinking' ), 998, 3 );
	}

	/**
	 * Init all bbPress CPT into Array.
	 */
	public function init() {
		$forum_post_type      = bbp_get_forum_post_type();
		$topic_post_type      = bbp_get_topic_post_type();
		$reply_post_type      = bbp_get_reply_post_type();
		$this->arr_post_types = array( $forum_post_type, $topic_post_type, $reply_post_type );
	}

	/**
	 * Add item embed with response
	 *
	 * @param array   $response The response data to send to the client.
	 * @param WP_Post $post     The post object.
	 * @param string  $url      The requested URL.
	 *
	 * @since 1.8.30 Added the `$url ` parameter.
	 * @return mixed
	 */
	public function update_cpt_object( $response, $post, $url ) {
		if ( bbp_get_forum_post_type() === $post->post_type ) {
			$response['_link_action'] = 'bbpress_forum';
		}

		if ( bbp_get_topic_post_type() === $post->post_type ) {
			$response['_link_action'] = 'bbpress_topic';
		}

		if ( bbp_get_reply_post_type() === $post->post_type ) {
			$response['_link_action'] = 'bbpress_reply';
		}

		$parse_url = wp_parse_url( html_entity_decode( $url ) );

		if ( ! empty( $parse_url['query'] ) ) {
			$query_params = array();

			parse_str( $parse_url['query'], $query_params );

			// Add reply id in response if topic id is present in query params and response item id is same as topic id.
			if ( ! empty( $query_params['topic_id'] ) && ! empty( $query_params['reply_id'] ) && (int) $response['item_id'] === (int) $query_params['topic_id'] ) {
				$response['reply_id'] = (int) $query_params['reply_id'];
			}
		}

		return $response;
	}

	/**
	 * Update bbPress's CPT namespace
	 *
	 * @param string $cpt_namespace Post type namespace.
	 * @param string $post_type Post type.
	 *
	 * @return string
	 */
	public function update_cpt_namespace( $cpt_namespace, $post_type ) {
		if ( isset( $post_type ) && in_array( $post_type, $this->arr_post_types, true ) ) {
			return 'bbpress';
		}

		return $cpt_namespace;
	}

	/**
	 * Parse deeplink url data.
	 *
	 * @param string $url Deeplink url.
	 */
	public function parse( $url ) {
		$url_id = url_to_postid( $url );

		if ( ! empty( $url_id ) ) {
			$post      = get_post( $url_id );
			$post_type = $post->post_type;

			if ( ! in_array( $post_type, $this->arr_post_types, true ) ) {
				return null;
			}

			if ( bbapp_is_lite_live_app() && in_array( $post_type, $this->arr_post_types, true ) ) {
				return array(
					'action'    => 'open_restricted',
					'namespace' => 'core',
					'item_id'   => '403',
					'url'       => $url,
				);
			}

			$namespace = apply_filters( 'bbapp_deeplinking_cpt_namespace', 'core', $post_type );

			$deep_link_data = array(
				'action'       => "open_{$post_type}",
				'namespace'    => $namespace,
				'url'          => $url,
				'item_id'      => $post->ID,
				'_link_action' => 'core_' . $post_type . 's',
			);

			return apply_filters( 'bbapp_deeplinking_cpt_type', $deep_link_data, $post, $url );
		}

		return null;
	}

	/**
	 * Get memberpress restriction for deeplinking.
	 *
	 * @param WP_REST_Response $response The response object.
	 * @param string           $url      The url.
	 * @param string           $version  The version.
	 *
	 * @since 2.2.80
	 *
	 * @return mixed
	 */
	public function get_mp_restricted_for_deeplinking( $response, $url, $version ) {
		if ( ! isset( $response['post_type'] ) ) {
			return $response;
		}

		if (
			bbapp_is_lite_live_app() &&
			in_array( $response['post_type'], $this->arr_post_types, true )
		) {
			return array(
				'action'    => 'open_restricted',
				'namespace' => 'core',
				'item_id'   => '403',
				'url'       => $url,
			);
		}

		return $response;
	}
}
