<?php
/**
 * Holds Access Controls functionality related to Social Group.
 *
 * @package BuddyBossApp\Integrations\BuddyBossPlatform\AccessControls
 */

namespace BuddyBossApp\Integrations\BuddyBossPlatform\AccessControls;

use BuddyBossApp\Integrations\BuddyBossPlatform\AccessControls;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Access Controls class for Social Group.
 */
class SocialGroup {

	/**
	 * Condition name.
	 *
	 * @var string $condition_name condition name.
	 */
	public $condition_name = 'bb-social-groups';

	/**
	 * Class instance.
	 *
	 * @var bool
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return SocialGroup
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Consctruct method.
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {
	}

	/**
	 * Items callback method.
	 *
	 * @param string $search Search the condition.
	 * @param int    $page   Page number.
	 * @param int    $limit  Limit the items to be fetched.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function social_group_items_callback( $search = '', $page = 1, $limit = 20 ) {
		$list = array();

		$args = array(
			'search_terms' => $search,
			'page'         => $page,
			'show_hidden'  => true,
		);

		if ( - 1 !== $limit ) {
			$args['per_page'] = $limit;
		}

		$results = \BP_Groups_Group::get( $args );

		if ( ! empty( $results ) ) {
			foreach ( $results['groups'] as $result ) {
				$list[ $result->id ] = array(
					'id'   => $result->id,
					'name' => $result->name,
					'link' => admin_url( "admin.php?page=bp-groups&gid={$result->id}&action=edit" ),
				);
			}
		}

		return $list;
	}

	/**
	 * Item callback method.
	 *
	 * @param int $item_value Item value of condition.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array|false
	 */
	public function social_group_item_callback( $item_value ) {

		$group = groups_get_group( $item_value );

		if ( empty( $group ) || is_wp_error( $group ) || empty( $group->slug ) ) {
			return false;
		}

		return array(
			'id'   => $group->id,
			'name' => $group->name,
			'link' => admin_url( "admin.php?page=bp-groups&gid={$group->id}&action=edit" ),
		);
	}

	/**
	 * Users callback method.
	 *
	 * @param array $data     condition data.
	 * @param int   $page     current page number.
	 * @param int   $per_page limit.
	 *
	 * @since 1.5.2.1
	 * @return array
	 */
	public function social_group_users_callback( $data, $page = 1, $per_page = 10 ) {
		$sub_condition = ( ! empty( $data['sub_condition'] ) ) ? $data['sub_condition'] : '';
		$item_value    = ( ! empty( $data['item_value'] ) ) ? $data['item_value'] : '';
		$group_args    = array(
			'page'     => $page,
			'per_page' => $per_page,
		);

		if ( 'specific' === $sub_condition ) {
			$group_args['group_ids'] = array( $item_value );
		}

		$group_members = function_exists( 'bb_get_all_members_for_groups' ) ? bb_get_all_members_for_groups( $group_args ) : array();
		$user_ids      = ( ! empty( $group_members ) ) ? $group_members : array();

		return AccessControls::instance()->return_users( $user_ids );
	}

	/**
	 * Function to check if user has any condition.
	 *
	 * @param int $user_id User id to check.
	 *
	 * @since 1.5.2.1
	 *
	 * @return bool
	 */
	public function has_any_items_callback( $user_id ) {
		$user_groups = bp_get_user_groups(
			$user_id,
			array(
				'is_admin' => true,
				'is_mod'   => true,
			)
		);

		return ( ! empty( $user_groups ) ) ? true : false;
	}

	/**
	 * This function add user to access control.
	 *
	 * @param int $group_id ID of the group.
	 * @param int $user_id  ID of the user joining the group.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_groups_join_group( $group_id, $user_id ) {
		if ( empty( $group_id ) || empty( $user_id ) ) {
			return;
		}
		AccessControls::instance()->condition_add_user( $user_id, $this->condition_name, $group_id );
	}

	/**
	 * This function add user to access control.
	 *
	 * @param int    $group_id ID of the group being promoted in.
	 * @param int    $user_id  ID of the user being promoted.
	 * @param string $status   New status being promoted to.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_groups_promote_member( $group_id, $user_id, $status ) {
		if ( empty( $group_id ) || empty( $user_id ) ) {
			return;
		}
		AccessControls::instance()->condition_add_user( $user_id, $this->condition_name, $group_id );
	}

	/**
	 * This function add user to access control.
	 *
	 * @param int $group_id ID of the group being promoted in.
	 * @param int $user_id  ID of the user being promoted.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_groups_unban_member( $group_id, $user_id ) {
		if ( empty( $group_id ) || empty( $user_id ) ) {
			return;
		}
		AccessControls::instance()->condition_add_user( $user_id, $this->condition_name, $group_id );
	}

	/**
	 * This function add user to access control.
	 *
	 * @param array $r result of BP_Groups_Invitation_Manager.
	 *
	 * @since 1.5.2.1
	 * @return false|void
	 */
	public function bbapp_invitations_accepted_invite( $r ) {
		if ( empty( $r ) ) {
			return;
		}
		AccessControls::instance()->condition_add_user( $r['user_id'], $this->condition_name, $r['item_id'] );
	}

	/**
	 * This function add user to access control.
	 *
	 * @param array $r Invitation arguments.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_invitations_accepted_request( $r ) {
		if ( empty( $r ) ) {
			return;
		}
		AccessControls::instance()->condition_add_user( $r['user_id'], $this->condition_name, $r['item_id'] );
	}

	/**
	 * This function remove user from access control.
	 *
	 * @param int $group_id ID of the group being promoted in.
	 * @param int $user_id  ID of the user being promoted.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_groups_ban_member( $group_id, $user_id ) {
		if ( empty( $group_id ) || empty( $user_id ) ) {
			return;
		}
		AccessControls::instance()->condition_remove_user( $user_id, $this->condition_name, $group_id );
	}

	/**
	 * This function remove user from access control.
	 *
	 * @param int $group_id ID of the group being promoted in.
	 * @param int $user_id  ID of the user being promoted.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_groups_leave_group( $group_id, $user_id ) {
		if ( empty( $group_id ) || empty( $user_id ) ) {
			return;
		}
		AccessControls::instance()->condition_remove_user( $user_id, $this->condition_name, $group_id );
	}

	/**
	 * This function remove user from access control.
	 *
	 * @param int $user_id    ID of the user being promoted.
	 * @param int $group_id   ID of the group being promoted in.
	 * @param int $inviter_id Inviter user id.
	 *
	 * @return void
	 */
	public function bbapp_groups_reject_invite( $user_id, $group_id, $inviter_id ) {
		if ( empty( $user_id ) || empty( $group_id ) ) {
			return;
		}
		AccessControls::instance()->condition_remove_user( $user_id, $this->condition_name, $group_id );
	}

	/**
	 * This function remove user from access control.
	 *
	 * @param int $user_id    ID of the user being promoted.
	 * @param int $group_id   ID of the group being promoted in.
	 * @param int $inviter_id Inviter user id.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function bbapp_groups_delete_invite( $user_id, $group_id, $inviter_id ) {
		if ( empty( $user_id ) || empty( $group_id ) ) {
			return;
		}
		AccessControls::instance()->condition_remove_user( $user_id, $this->condition_name, $group_id );
	}

	/**
	 * This function remove user from access control.
	 *
	 * @param int $user_id    ID of the user being promoted.
	 * @param int $group_id   ID of the group being promoted in.
	 * @param int $inviter_id Inviter user id.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function bbapp_groups_membership_rejected( $user_id, $group_id, $inviter_id ) {
		if ( empty( $user_id ) || empty( $group_id ) ) {
			return;
		}
		AccessControls::instance()->condition_remove_user( $user_id, $this->condition_name, $group_id );
	}

	/**
	 * This function remove user from access control.
	 *
	 * @param object $group_obj Group object.
	 * @param array  $user_ids  User ids.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_groups_delete_group( $group_obj, $user_ids ) {
		if ( empty( $group_obj ) || empty( $user_ids ) ) {
			return;
		}
		AccessControls::instance()->condition_remove_users( $user_ids, $this->condition_name, $group_obj->id );
	}

	/**
	 * This function remove user from access control.
	 *
	 * @param object $group_member_obj Group object.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_groups_member_after_remove( $group_member_obj ) {
		if ( empty( $group_member_obj ) ) {
			return;
		}
		AccessControls::instance()->condition_remove_user( $group_member_obj->user_id, $this->condition_name, $group_member_obj->group_id );
	}

	/**
	 * This function remove user from access control.
	 *
	 * @param int $user_id  ID of the user being promoted.
	 * @param int $group_id ID of the group being promoted in.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_groups_member_after_delete( $user_id, $group_id ) {
		if ( empty( $user_id ) || empty( $group_id ) ) {
			return;
		}
		AccessControls::instance()->condition_remove_user( $user_id, $this->condition_name, $group_id );
	}

	/**
	 * This function will remove member/group from access control when group will delete from admin side.
	 *
	 * @param int $group_id Group Id.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_groups_delete_group_members( $group_id ) {
		AccessControls::instance()->condition_item_deleted( $this->condition_name, $group_id );
		AccessControls::instance()->condition_remove_all_users( $this->condition_name, $group_id );
		bb_access_recalculate_member_for_has_any_membership_group( $this->condition_name );
	}
}
