<?php
/**
 * Holds Access Controls functionality related Profile Type.
 *
 * @package BuddyBossApp\Integrations\BuddyBossPlatform\AccessControls
 */

namespace BuddyBossApp\Integrations\BuddyBossPlatform\AccessControls;

use BuddyBossApp\Integrations\BuddyBossPlatform\AccessControls;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Profile Type.
 */
class ProfileType {

	/**
	 * Condition name.
	 *
	 * @var string $condition_name condition name.
	 */
	public $condition_name = 'bb-profile-types';

	/**
	 * Class instance.
	 *
	 * @var bool
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return ProfileType
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Class construct function.
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {}

	/**
	 * Items callback method.
	 *
	 * @param string $search Search the condition.
	 * @param int    $page   Page number.
	 * @param int    $limit  Limit the items to be fetched.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function profile_type_items_callback( $search = '', $page = 1, $limit = 20 ) {
		$list          = array();
		$profile_types = bp_get_active_member_types(
			array(
				'posts_per_page' => ! empty( $limit ) ? $limit : 10,
				'per_page'       => ! empty( $page ) ? $page : 1,
				's'              => ! empty( $search ) ? $search : false,
				'fields'         => 'all',
			)
		);

		if ( ! empty( $profile_types ) ) {
			foreach ( $profile_types as $profile_type ) {
				$list[ $profile_type->ID ] = array(
					'id'   => $profile_type->ID,
					'name' => $profile_type->post_title,
					'link' => admin_url( "post.php?post={$profile_type->ID}&action=edit" ),
				);
			}
		}

		return $list;
	}

	/**
	 * Item callback method.
	 *
	 * @param int $item_value Item value of condition.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array|false
	 */
	public function profile_type_item_callback( $item_value ) {
		$member_type = bp_get_active_member_types(
			array(
				'post__in' => array( $item_value ),
				'fields'   => 'all',
			)
		);

		if ( empty( $member_type[0] ) ) {
			return false;
		}

		return array(
			'id'   => $member_type[0]->ID,
			'name' => $member_type[0]->post_title,
			'link' => admin_url( "post.php?post={$member_type[0]->ID}&action=edit" ),
		);
	}

	/**
	 * Users callback method.
	 *
	 * @param array $data     condition data.
	 * @param int   $page     current page number.
	 * @param int   $per_page limit.
	 *
	 * @since 1.5.2.1
	 * @return array
	 */
	public function profile_type_users_callback( $data, $page = 1, $per_page = 10 ) {
		$user_ids   = array();
		$item_value = ( ! empty( $data['item_value'] ) ) ? $data['item_value'] : '';

		if ( empty( $item_value ) ) {
			return $user_ids;
		}

		$name      = bp_get_member_type_key( $item_value );
		$type_id   = bp_member_type_term_taxonomy_id( $name );
		$users_ids = bp_member_type_by_type( $type_id );

		if ( ! empty( $users_ids ) ) {
			$user_ids = AccessControls::instance()->paginate_users_list( array_unique( $users_ids ), $page, $per_page );
		}

		return AccessControls::instance()->return_users( $user_ids );
	}

	/**
	 * Function to add user role and update member access.
	 *
	 * @param int    $user_id The user ID.
	 * @param string $role    The new role.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_add_user_role( $user_id, $role ) {
		if ( empty( $user_id ) ) {
			return;
		}

		$member_type = bp_get_member_type( $user_id );
		$post_id     = bp_member_type_post_by_type( $member_type );
		AccessControls::instance()->condition_add_user( $user_id, $this->condition_name, (string) $post_id );
	}

	/**
	 * Function to remove user role and update member access.
	 *
	 * @param int    $user_id The user ID.
	 * @param string $role    The new role.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_remove_user_role( $user_id, $role ) {
		if ( empty( $user_id ) ) {
			return;
		}
		$member_type = bp_get_member_type( $user_id );
		$post_id     = bp_member_type_post_by_type( $member_type );
		AccessControls::instance()->condition_add_user( $user_id, $this->condition_name, (string) $post_id );
	}

	/**
	 * Function to add user role and update member access.
	 *
	 * @param int    $user_id     ID of the user whose profile type has been updated.
	 * @param string $member_type profile type.
	 * @param bool   $append      Whether the type is being appended to existing types.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_set_member_type( $user_id, $member_type, $append ) {
		if ( empty( $user_id ) || empty( $member_type ) ) {
			return;
		}
		$post_id = bp_member_type_post_by_type( $member_type );
		AccessControls::instance()->condition_add_user( $user_id, $this->condition_name, (string) $post_id );
	}

	/**
	 * Function to remove user role and update member access.
	 *
	 * @param int    $user_id     ID of the user whose profile type has been updated.
	 * @param string $member_type profile type.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_remove_member_type( $user_id, $member_type ) {
		if ( empty( $user_id ) || empty( $member_type ) ) {
			return;
		}
		$post_id = bp_member_type_post_by_type( $member_type );
		AccessControls::instance()->condition_remove_user( $user_id, $this->condition_name, (string) $post_id );
	}

	/**
	 * Function to add/remove user role and update member access.
	 *
	 * @param int   $displayed_user_id Displayed user ID.
	 * @param array $posted_field_ids  Array of field IDs that were edited.
	 * @param bool  $errors            Whether or not any errors occurred.
	 * @param array $old_values        Array of original values before updated.
	 * @param array $new_values        Array of newly saved values after update.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_xprofile_updated_profile( $displayed_user_id, $posted_field_ids, $errors, $old_values, $new_values ) {
		$field_member_type_id = bp_get_xprofile_member_type_field_id();
		if ( isset( $old_values[ $field_member_type_id ] ) ) {
			$old_member_type_data = $old_values[ $field_member_type_id ];
			if ( isset( $old_member_type_data['value'] ) ) {
				AccessControls::instance()->condition_remove_user( $displayed_user_id, $this->condition_name, (string) $old_member_type_data['value'] );
			}
		}

		if ( isset( $new_values[ $field_member_type_id ] ) ) {
			$new_member_type_data = $new_values[ $field_member_type_id ];
			if ( isset( $new_member_type_data['value'] ) ) {
				AccessControls::instance()->condition_add_user( $displayed_user_id, $this->condition_name, (string) $new_member_type_data['value'] );
			}
		}
	}
	/**
	 * Fires immediately before an object-term relationship is deleted.
	 *
	 * @param int    $object_id Object ID.
	 * @param array  $tt_ids    An array of term taxonomy IDs.
	 * @param string $taxonomy  Taxonomy slug.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_delete_term_relationships( $object_id, $tt_ids, $taxonomy ) {
		if ( bp_get_member_type_tax_name() === $taxonomy ) {
			$member_type = bp_get_member_type( $object_id );
			if ( ! empty( $member_type ) ) {
				$old_post_id = bp_member_type_post_by_type( $member_type );
				AccessControls::instance()->condition_remove_user( $object_id, $this->condition_name, (string) $old_post_id );
			}
		}
	}

	/**
	 * Function to remove users role and update member access.
	 *
	 * @param string   $new_status New post status.
	 * @param string   $old_status Old post status.
	 * @param \WP_Post $post       Post object.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_trashed_post( $new_status, $old_status, $post ) {
		if ( 'publish' !== $new_status && 'bp-member-type' === $post->post_type ) {
			AccessControls::instance()->condition_item_deleted( $this->condition_name, $post->ID );
			AccessControls::instance()->condition_remove_all_users( $this->condition_name, $post->ID );
		}
	}

	/**
	 * Function to add users role and update member access.
	 *
	 * @param int    $post_id         Post ID.
	 * @param string $previous_status The status of the post at the point where it was trashed.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_untrashed_post( $post_id, $previous_status ) {
		$profile_type_post = get_post( $post_id );
		if ( $profile_type_post instanceof \WP_Post && 'bp-member-type' === $profile_type_post->post_type && 'publish' === $profile_type_post->post_status ) {
			$name = bp_get_member_type_key( $post_id );
			if ( ! empty( $name ) ) {
				$type_id              = bp_member_type_term_taxonomy_id( $name );
				$member_type_user_ids = bp_member_type_by_type( $type_id );
				if ( ! empty( $member_type_user_ids ) ) {
					AccessControls::instance()->add_users_to_condition( $member_type_user_ids, $this->condition_name, (string) $post_id );
				}
			}
		}
	}
}
