<?php

namespace BuddyBossApp\Integrations\BadgeOs;

use \WP_Error as WP_Error;

class RestAPIRank extends \WP_REST_Controller
{

    protected $namespace = "buddyboss-app/badgeos/v1";
    protected $rest_base = 'ranks';
    protected static $instance;

	/**
	 * RestAPIRank constructor.
	 */
    public function __construct()
    {
        /** Nothing here */
    }

	/**
	 * @return mixed
	 */
    public static function instance()
    {
        if (!isset(self::$instance)) {
            $class = __CLASS__;
            self::$instance = new $class;
            self::$instance->hooks(); // run the hooks.
        }
        return self::$instance;
    }

	/**
	 *
	 */
    public function hooks()
    {
        add_action('rest_api_init', array($this, "registerRoutes"), 99);
    }

	/**
	 *
	 */
    public function registerRoutes()
    {
        register_rest_route($this->namespace, '/' . $this->rest_base, array(
            array(
                'methods' => \WP_REST_Server::READABLE,
                'callback' => array($this, 'get_items'),
                'permission_callback' => array($this, 'get_items_permissions_check'),
                'args' => $this->get_collection_params(),
            ),
            'schema' => array($this, 'get_public_item_schema'),
        ));
    }

    /**
     * @param $request
     * @return array
     * @apiPrivate
     * @api {GET} /wp-json/buddyboss-app/badgeos/v1/ranks Learner ranks
     * @apiName GetGamipressRanks
     * @apiGroup Gamipress
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Get Gamipress ranks
     * @apiHeader {String} accessToken Auth token
     * @apiParam {String=view,edit,embed} [context=view]
     * @apiParam {Number} user_id Limit result set to specific user.
     */
    public function get_items($request)
    {

        if (!empty($request['user_id'])) {
            $user_id = (int)$request['user_id'];
        } else {
            $user_id = get_current_user_id();
        }
        $ranks = array();

        $rank_types = badgeos_get_rank_types_slugs_detailed();
        foreach ($rank_types as $rank_type => $rank) {

            $user_ranks = badgeos_get_user_ranks(array(
                'user_id' => absint($user_id),
                'rank_type' => $rank_type,
            ));

            if (!empty($user_ranks)) {
                foreach ($user_ranks as $user_rank) {
                    $rank = get_post($user_rank->rank_id);
                    $obj_rank = new \stdClass();
                    $obj_rank->ID = $user_rank->id;
                    $obj_rank->title = get_the_title($rank->ID);
                    $obj_rank->link = get_permalink($rank->ID);
                    $obj_rank->content = !empty($rank->post_excerpt) ? $rank->post_excerpt : $rank->post_content;
                    $points = get_post_meta($rank->ID, '_ranks_points_to_unlock', true);
                    $obj_rank->points = $points['_ranks_points_to_unlock'];
                    $obj_rank->points_type = get_the_title($points['_ranks_points_to_unlock_type']);
                    $obj_rank->image = RestAPI::badgeos_get_achievement_post_thumbnail_url($rank->ID);
                    $data = $this->prepare_item_for_response($obj_rank, $request);
                    $ranks[] = $this->prepare_response_for_collection($data);
                }
            }
        }

        $response = rest_ensure_response($ranks);

        return $response;
    }

	/**
	 * @param $request
	 *
	 * @return mixed
	 */
    public function get_items_permissions_check($request)
    {
        return apply_filters('bbapp_badgeos_ranks_items_permissions', true, $request);
    }

    /**
     * Get the Post's schema, conforming to JSON Schema.
     *
     * @return array
     */
    public function get_item_schema()
    {

        $schema = array(
            '$schema' => 'http://json-schema.org/draft-04/schema#',
            'title' => 'badgeos_rank',
            'type' => 'object',
            /*
                 * Base properties for every Post.
            */
            'properties' => array(
                'id' => array(
                    'description' => __('Unique identifier for the object.', 'buddyboss-app'),
                    'type' => 'integer',
                    'context' => array('view', 'edit', 'embed'),
                    'readonly' => true,
                ),
                'title' => array(
                    'description' => __('The title for the object.', 'buddyboss-app'),
                    'type' => 'object',
                    'context' => array('view', 'edit', 'embed'),
                    'properties' => array(
                        'raw' => array(
                            'description' => __('Title for the object, as it exists in the database.', 'buddyboss-app'),
                            'type' => 'string',
                            'context' => array('edit'),
                        ),
                        'rendered' => array(
                            'description' => __('HTML title for the object, transformed for display.', 'buddyboss-app'),
                            'type' => 'string',
                            'context' => array('view', 'edit', 'embed'),
                        ),
                    ),
                ),
                'content' => array(
                    'description' => __('The content for the object.', 'buddyboss-app'),
                    'type' => 'object',
                    'context' => array('view', 'edit'),
                    'properties' => array(
                        'raw' => array(
                            'description' => __('Content for the object, as it exists in the database.', 'buddyboss-app'),
                            'type' => 'string',
                            'context' => array('edit'),
                        ),
                        'rendered' => array(
                            'description' => __('HTML content for the object, transformed for display.', 'buddyboss-app'),
                            'type' => 'string',
                            'context' => array('view', 'edit'),
                        ),
                    ),
                ),
                'link' => array(
                    'description' => __('URL to the object.', 'buddyboss-app'),
                    'type' => 'string',
                    'format' => 'uri',
                    'context' => array('view', 'edit', 'embed'),
                    'readonly' => true,
                ),
                'image' => array(
                    'description' => __('URL to the image object.', 'buddyboss-app'),
                    'type' => 'image',
                    'format' => 'uri',
                    'context' => array('view', 'edit', 'embed'),
                    'readonly' => true,
                ),
            ),
        );

        $schema['properties']['points'] = array(
            'description' => __('Point for the object', 'buddyboss-app'),
            'type' => 'integer',
            'context' => array('view', 'edit'),
        );

        $schema['properties']['points_type'] = array(
            'description' => __('Point type for the object', 'buddyboss-app'),
            'type' => 'string',
            'context' => array('view', 'edit'),
        );

        return $this->add_additional_fields_schema($schema);
    }

    /**
     * Get the query params for collections of attachments.
     *
     * @return array
     */
    public function get_collection_params()
    {
        $params = parent::get_collection_params();

        $params['context']['default'] = 'view';

        $params['user_id'] = array(
            'description' => __('Limit result set to specific user.', 'buddyboss-app'),
            'type' => 'integer',
            'sanitize_callback' => 'absint',
            'validate_callback' => 'rest_validate_request_arg',
        );

        return $params;
    }

    /**
     * Prepare a single post output for response.
     *
     * @param WP_Post $post Post object.
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response $data
     */
    public function prepare_item_for_response($post, $request)
    {
        $data = array();

        if (isset($post->ID)) {
            $data['id'] = $post->ID;
        } else {
            return new WP_Error('badgeos_json_internal_error', __('Required field "ID" missing by add-on plugin', 'buddyboss-app'), array('status' => 400));
        }

        $schema = $this->get_item_schema();

        if (!empty($schema['properties']['title'])) {
            if (isset($post->title)) {
                $data['title'] = array(
                    'raw' => $post->title,
                    'rendered' => $post->title,
                );
            } else {
                return new WP_Error('badgeos_json_internal_error', __('Required field "Title" missing by add-on plugin', 'buddyboss-app'), array('status' => 400));
            }
        }

        if (!empty($schema['properties']['content'])) {
            if (isset($post->content)) {
                $data['content'] = array(
                    'raw' => $post->content,
                    /** This filter is documented in wp-includes/post-template.php */
                    'rendered' => apply_filters('the_content', $post->content),
                );
            } else {
                return new WP_Error('badgeos_json_internal_error', __('Required field "Content" missing by add-on plugin', 'buddyboss-app'), array('status' => 400));
            }
        }

        if (!empty($schema['properties']['points'])) {
            $data['points'] = (int)$post->points;
        }

        if (!empty($schema['properties']['points_type'])) {
            $data['points_type'] = $post->points_type;
        }

        if (!empty($schema['properties']['link'])) {
            $data['points'] = (int)$post->points;
        }

        if (!empty($schema['properties']['image'])) {
            $data['image'] = $post->image;
        }

        $context = !empty($request['context']) ? $request['context'] : 'view';
        $data = $this->add_additional_fields_to_object($data, $request);
        $data = $this->filter_response_by_context($data, $context);

        // Wrap the data in a response object.
        $response = rest_ensure_response($data);

        /**
         * Filter the post data for a response.
         *
         * The dynamic portion of the hook name, $this->post_type, refers to post_type of the post being
         * prepared for the response.
         *
         * @param WP_REST_Response $response The response object.
         * @param WP_Post $post Post object.
         * @param WP_REST_Request $request Request object.
         */
        return apply_filters("rest_prepare_badgeos_ranks", $response, $post, $request);
    }


}