<?php

namespace BuddyBossApp\Integrations\BadgeOs;

class RestAPIPoint extends \WP_REST_Controller
{

    protected $namespace = "buddyboss-app/badgeos/v1";
    protected $rest_base = 'points';
    protected static $instance;

	/**
	 * RestAPIPoint constructor.
	 */
    public function __construct()
    {
        /** Nothing here */
    }

	/**
	 * @return mixed
	 */
    public static function instance()
    {
        if (!isset(self::$instance)) {
            $class = __CLASS__;
            self::$instance = new $class;
            self::$instance->hooks(); // run the hooks.
        }
        return self::$instance;
    }

	/**
	 *
	 */
    public function hooks()
    {
        add_action('rest_api_init', array($this, "registerRoutes"), 99);
    }

	/**
	 *
	 */
    public function registerRoutes()
    {
        register_rest_route($this->namespace, '/' . $this->rest_base, array(
            array(
                'methods' => \WP_REST_Server::READABLE,
                'callback' => array($this, 'get_items'),
                'permission_callback' => array($this, 'get_items_permissions_check'),
                'args' => $this->get_collection_params(),
            ),
            'schema' => array($this, 'get_public_item_schema'),
        ));

        $this->register_learner_fields();
    }

    /**
     * Register Rest field
     */
    public function register_learner_fields(){
        register_rest_field('user',
            'points',
            array(
                'get_callback' => array($this, "get_points"),
                'update_callback' => null,
                'schema' => null,
            )
        );

        register_rest_field('user',
            'user_points',
            array(
                'get_callback' => array($this, "get_user_points"),
                'update_callback' => null,
                'schema' => null,
            )
        );

        register_rest_field('bp_members',
            'points',
            array(
                'get_callback' => array($this, "get_points"),
                'update_callback' => null,
                'schema' => null,
            )
        );

        register_rest_field('bp_members',
            'user_points',
            array(
                'get_callback' => array($this, "get_user_points"),
                'update_callback' => null,
                'schema' => null,
            )
        );
    }

    /**
     * Get user total points
     *
     * @param $object
     * @param $field_name
     * @param $request
     * @return array
     */
    public function get_points( $object, $field_name, $request ){
        $points = 0;
        if (function_exists('badgeos_get_point_types')) {
            $credit_types = badgeos_get_point_types();
            if (is_array($credit_types) && !empty($credit_types)) {
                if ( count( $credit_types ) == 1 ) {
                    $credit_type = $credit_types[0];
                    $_point = badgeos_get_points_by_type($credit_type->ID, $object["id"]);
                    return array(
                        'point' => $_point,
                        'label' => ( $_point >1 ) ? get_post_meta($credit_type->ID,'_point_plural_name', true ) :  $credit_type->post_title,
                        'singular_name' => $credit_type->post_title,
                        'plural_name' => get_post_meta($credit_type->ID,'_point_plural_name', true ),
                    );
                } else{
                    foreach ($credit_types as $credit_type) {
                        $_point = (int) badgeos_get_points_by_type($credit_type->ID, $object["id"]);
                        $points = $points + $_point;
                    }
                }
            }
        } else {
            $points =  badgeos_get_users_points($object["id"]);
        }
        return array(
            'point' => $points,
            'label' => ( $points >1 ) ? __( 'Points', 'buddyboss-app' ) :  __( 'Point', 'buddyboss-app' ),
            'singular_name' => __( 'Point', 'buddyboss-app' ),
            'plural_name' => __( 'Points', 'buddyboss-app' ),
        );
    }

    /**
     * Get user all type of points
     * @param $object
     * @param $field_name
     * @param $request
     * @return array
     */
    public function get_user_points( $object, $field_name, $request ){
        if (function_exists('badgeos_get_point_types')) {
            $points = array();
            $credit_types = badgeos_get_point_types();
            if (is_array($credit_types) && !empty($credit_types)) {
                foreach ($credit_types as $credit_type) {
                    $_point = (int) badgeos_get_points_by_type($credit_type->ID, $object["id"]);
                    $points[] = array(
                        'point' => $_point,
                        'label' => ( $_point >1 ) ? get_post_meta($credit_type->ID,'_point_plural_name', true ) :  $credit_type->post_title,
                        'singular_name' => $credit_type->post_title,
                        'plural_name' => get_post_meta($credit_type->ID,'_point_plural_name', true ),
                        'image' => RestAPI::badgeos_get_achievement_post_thumbnail_url($credit_type->ID, 'badgeos-achievement', false),
                    );
                }
            }
            return $points;
        } else {
            $points = badgeos_get_users_points($object["id"]);
            return array(
                array(
                    'point' => $points,
                    'label' => ( $points >1 ) ? __( 'Points', 'buddyboss-app' ) :  __( 'Point', 'buddyboss-app' ),
                    'singular_name' => __( 'Point', 'buddyboss-app' ),
                    'plural_name' => __( 'Points', 'buddyboss-app' ),
                    'image' => '', // Todo: This else part, how we can handle or we have to remove this else part?
                ),
            );
        }
    }

    /**
     * @param $request
     * @return array
     * @apiPrivate
     * @api {GET} /wp-json/buddyboss-app/badgeos/v1/points Learner points
     * @apiName GetGamipressPoints
     * @apiGroup Gamipress
     * @apiVersion 1.0.0
     * @apiPermission LoggedInUser
     * @apiDescription Get Gamipress points
     * @apiHeader {String} accessToken Auth token
     * @apiParam {String=view,edit,embed} [context=view]
     * @apiParam {Number} user_id Limit result set to specific user.
     */
    public function get_items($request)
    {

        if (!empty($request['user_id'])) {
            $user_id = (int)$request['user_id'];
        } else {
            $user_id = get_current_user_id();
        }

        $points = array();
        $points_types = badgeos_get_point_types();
        foreach ($points_types as $points_type) {

            $_point = badgeos_get_points_by_type($points_type->ID, $user_id);

            $obj_point = new \stdClass();
            $obj_point->title = ( $_point >1 ) ? get_post_meta($points_type->ID,'_point_plural_name', true ) :  $points_type->post_title;
            $obj_point->singular_name = $points_type->post_title;
            $obj_point->plural_name = get_post_meta($points_type->ID,'_point_plural_name', true );
            $obj_point->points = $_point;
            $obj_point->image = RestAPI::badgeos_get_achievement_post_thumbnail_url($points_type->ID, 'badgeos-achievement', false);

            $data = $this->prepare_item_for_response($obj_point, $request);
            $points[] = $this->prepare_response_for_collection($data);
        }

        $response = rest_ensure_response($points);

        return $response;
    }

	/**
	 * @param $request
	 *
	 * @return mixed
	 */
    public function get_items_permissions_check($request)
    {
        return apply_filters('bbapp_badgeos_points_items_permissions', true, $request);
    }

    /**
     * Get the Post's schema, conforming to JSON Schema.
     *
     * @return array
     */
    public function get_item_schema()
    {

        $schema = array(
            '$schema' => 'http://json-schema.org/draft-04/schema#',
            'title' => 'badgeos_point',
            'type' => 'object',
            /*
                 * Base properties for every Post.
            */
            'properties' => array(
                'title' => array(
                    'description' => __('The title for the object.', 'buddyboss-app'),
                    'type' => 'object',
                    'context' => array('view', 'edit', 'embed'),
                    'properties' => array(
                        'raw' => array(
                            'description' => __('Title for the object, as it exists in the database.', 'buddyboss-app'),
                            'type' => 'string',
                            'context' => array('edit'),
                        ),
                        'rendered' => array(
                            'description' => __('HTML title for the object, transformed for display.', 'buddyboss-app'),
                            'type' => 'string',
                            'context' => array('view', 'edit', 'embed'),
                        ),
                    ),
                ),
                'image' => array(
                    'description' => __('URL to the image object.', 'buddyboss-app'),
                    'type' => 'image',
                    'format' => 'uri',
                    'context' => array('view', 'edit', 'embed'),
                    'readonly' => true,
                ),
            ),
        );

        $schema['properties']['singular_name'] = array(
            'description' => __('Point name for the object', 'buddyboss-app'),
            'type' => 'string',
            'context' => array('view', 'edit'),
        );

        $schema['properties']['plural_name'] = array(
            'description' => __('Point name for the object', 'buddyboss-app'),
            'type' => 'string',
            'context' => array('view', 'edit'),
        );

        $schema['properties']['points'] = array(
            'description' => __('Point for the object', 'buddyboss-app'),
            'type' => 'integer',
            'context' => array('view', 'edit'),
        );

        return $this->add_additional_fields_schema($schema);
    }

    /**
     * Get the query params for collections of attachments.
     *
     * @return array
     */
    public function get_collection_params()
    {
        $params = parent::get_collection_params();

        $params['context']['default'] = 'view';

        $params['user_id'] = array(
            'description' => __('Limit result set to specific user.', 'buddyboss-app'),
            'type' => 'integer',
            'sanitize_callback' => 'absint',
            'validate_callback' => 'rest_validate_request_arg',
        );

        return $params;
    }

    /**
     * Prepare a single post output for response.
     *
     * @param WP_Post $post Post object.
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response $data
     */
    public function prepare_item_for_response($post, $request)
    {
        $data = array();

        $schema = $this->get_item_schema();

        if (!empty($schema['properties']['title'])) {
            if (isset($post->title)) {
                $data['title'] = array(
                    'raw' => $post->title,
                    'rendered' => $post->title,
                );
            } else {
                return new \WP_Error('badgeos_json_internal_error', __('Required field "Title" missing by add-on plugin', 'buddyboss-app'), array('status' => 400));
            }
        }

        if (!empty($schema['properties']['singular_name'])) {
            $data['singular_name'] = $post->singular_name;
        }

        if (!empty($schema['properties']['plural_name'])) {
            $data['plural_name'] = $post->plural_name;
        }

        if (!empty($schema['properties']['image'])) {
            $data['image'] = $post->image;
        }

        if (!empty($schema['properties']['points'])) {
            $data['points'] = (int)$post->points;
        }

        $context = !empty($request['context']) ? $request['context'] : 'view';
        $data = $this->add_additional_fields_to_object($data, $request);
        $data = $this->filter_response_by_context($data, $context);

        // Wrap the data in a response object.
        $response = rest_ensure_response($data);

        /**
         * Filter the post data for a response.
         *
         * The dynamic portion of the hook name, $this->post_type, refers to post_type of the post being
         * prepared for the response.
         *
         * @param WP_REST_Response $response The response object.
         * @param WP_Post $post Post object.
         * @param WP_REST_Request $request Request object.
         */
        return apply_filters("rest_prepare_badgeos_points", $response, $post, $request);
    }


}