<?php
/**
 * Holds IAP related functions.
 *
 * @package BuddyBossApp\InAppPurchases
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\InAppPurchases\Controller;
use BuddyBossApp\Tools\Logger;
use BuddyBossApp\Helpers\BBAPP_File;

/**
 * Get the main component object.
 * Note - This function is not available when component is disabled, instead of class on that time.
 *
 * @return \BuddyBossApp\InAppPurchases\Controller the singleton object
 */
function bbapp_iap() {
	return Controller::instance();
}

/**
 * Return IAP Registered Integrations.
 *
 * @return mixed|void
 */
function bbapp_iap_get_integrations() {
	/**
	 * Filters registered conditions.
	 *
	 * @param array $conditions Registered conditions.
	 */
	return apply_filters( 'bbapp_iap_get_registered_integrations', array() );
}

/**
 * Return the IAP types.
 */
function bbapp_iap_get_types() {

	/**
	 * Filters registered IAP.
	 *
	 * @param array $registered_iap Registered IAP.
	 */
	return apply_filters( 'bbapp_iap_get_registered_iap', array() );
}

/**
 * Function to prepare product item.
 *
 * @param array $product direct object or array from db.
 *
 * @return array|direct
 */
function bbapp_iap_prepare_product_item( $product ) {
	$product = (array) $product;

	return $product;
}

/**
 * Process & format integration_data of iap product.
 *
 * @param array $integration_data Integration data.
 *
 * @return mixed
 */
function bbapp_iap_prepare_product_integration_data( $integration_data ) {

	$integration_data = maybe_unserialize( $integration_data );

	if ( ! is_array( $integration_data ) ) {
		$integration_data = array();
	}

	$integration_data['linked'] = ( ! isset( $integration_data['linked'] ) ) ? false : $integration_data['linked'];

	return $integration_data;
}

/**
 * Return the iap upload directory with relative path.
 *
 * @param string $fullpath Relative path of IAP.
 *
 * @return mixed
 */
function bbapp_iap_get_upload_relative_path( $fullpath ) {
	$upload_dir = wp_upload_dir();

	return str_replace( $upload_dir['basedir'], '', $fullpath );
}

/**
 * Return count of all products.
 *
 * @param string $status eg.published,trash.
 *
 * @return null|string
 */
function bbapp_iap_get_products_count( $status = '' ) {
	global $wpdb;

	$global_prefix = bbapp_iap()->get_global_dbprefix();
	$sql           = "SELECT COUNT(*) FROM {$global_prefix}bbapp_iap_products";

	if ( ! empty( $status ) ) {
		$sql .= $wpdb->prepare( ' where status=%s', $status );
	} else {
		$sql .= $wpdb->prepare( ' where status !=%s', 'trash' );
	}

	return $wpdb->get_var( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.NotPrepared
}

/**
 * Returns iap products.
 *
 * @param array $args Array of arguments.
 *
 * @return array
 */
function bbapp_iap_get_products( $args ) {
	global $wpdb;

	$default_args = array(
		'iap_type'     => 'any',
		'per_page'     => false,
		'current_page' => 1,
		'orderby'      => 'id',
		'order'        => 'desc',
		'status'       => 'all',
	);
	$args         = array_merge( $default_args, $args );
	$where_clause = array();

	if ( 'any' !== $args['iap_type'] ) {
		$where_clause[] = $wpdb->prepare( 'iap_type=%s', $args['iap_type'] );
	}

	if ( 'all' !== $args['status'] ) {
		$where_clause[] = $wpdb->prepare( 'status=%s', $args['status'] );
	} else {
		$where_clause[] = $wpdb->prepare( 'status!=%s', 'trash' );
	}

	if ( ! empty( $where_clause ) ) {
		$where_clause = 'WHERE ' . implode( 'AND', $where_clause );
	} else {
		$where_clause = '';
	}

	$limit_clause = '';

	if ( $args['per_page'] ) {
		$args['per_page']     = (int) $args['per_page'];
		$args['current_page'] = (int) $args['current_page'];
		$limit_clause         = " LIMIT {$args["per_page"]} ";
		$limit_clause        .= ' OFFSET ' . ( $args['current_page'] - 1 ) * $args['per_page'];
	}

	$order_by_clause = '';

	if ( ! empty( $args['orderby'] ) ) {
		$order_by_clause .= ' ORDER BY menu_order,' . esc_sql( $args['orderby'] );
		$order_by_clause .= ! empty( $args['order'] ) ? ' ' . esc_sql( $args['order'] ) : ' ASC';
	}

	$get_results = $wpdb->get_results( "SELECT * FROM {$wpdb->base_prefix}bbapp_iap_products {$where_clause} {$order_by_clause} {$limit_clause}" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	$products    = array();

	foreach ( $get_results as $product ) {
		$products[] = bbapp_iap_prepare_product_item( $product );
	}

	return $products;
}

/**
 * Returns single iap product.
 *
 * @param int $product_id IAP product id.
 *
 * @return array|bool|direct|null|object|void
 */
function bbapp_iap_get_product( $product_id ) {
	global $wpdb;

	$product_id = (int) $product_id;
	$product    = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$wpdb->base_prefix}bbapp_iap_products WHERE id=%d", $product_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

	if ( ! $product ) {
		return false;
	}

	$product = bbapp_iap_prepare_product_item( $product );

	return $product;
}

/**
 * Create iap product.
 *
 * @param array $product Product data.
 *
 * @return array|direct|WP_Error
 */
function bbapp_iap_create_product( $product ) {
	global $wpdb;

	if ( IAP_LOG ) {
		Logger::instance()->add( 'iap_log', 'include/InAppPurchases/functions.php->bbapp_iap_create_product()' );
	}

	$global_prefix = bbapp_iap()->get_global_dbprefix();
	$default       = array(
		'blog_id'           => get_current_blog_id(),
		'product_author_id' => get_current_user_id(),
		'name'              => '',
		'tagline'           => '',
		'description'       => '',
		'store_data'        => '',
		'misc_settings'     => array(),
		'integration_data'  => '',
		'date_created'      => current_time( 'mysql', 1 ),
		'date_updated'      => current_time( 'mysql', 1 ),
		'iap_group'         => 0,
	);
	$product       = array_merge( $default, $product );

	if ( empty( $product['name'] ) ) {
		return new WP_Error( 'name_required', __( 'Product Name is missing.', 'buddyboss-app' ) );
	}

	$add = $wpdb->insert( "{$global_prefix}bbapp_iap_products", $product ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery

	if ( ! $add ) {
		return new WP_Error(
			'error_inserting',
			__( 'There is a database error while adding record.', 'buddyboss-app' )
		);
	}

	$product       = (array) $product;
	$product['id'] = $wpdb->insert_id;

	/**
	 * Create iap product.
	 *
	 * @peram int   $product_id  Product id.
	 * @peram array $product  Product ids.
	 *
	 * @since 1.6.7
	 */
	do_action( 'bbapp_after_iap_created_product', $product['id'], $product );

	return bbapp_iap_prepare_product_item( $product );
}

/**
 * Update iap product.
 *
 * @param int   $product_id IAP product id.
 * @param array $data       Product data.
 *
 * @return bool
 */
function bbapp_iap_update_product( $product_id, $data ) {
	global $wpdb;

	$global_prefix = bbapp_iap()->get_global_dbprefix();
	$columns       = array(
		'name'             => $data['name'],
		'tagline'          => $data['tagline'],
		'description'      => $data['description'],
		'store_data'       => $data['store_data'],
		'misc_settings'    => $data['misc_settings'],
		'integration_data' => $data['integration_data'],
		'date_updated'     => current_time( 'mysql', 1 ),
		'iap_group'        => $data['iap_group'],
		'status'           => $data['status'],
	);
	$new_vals      = array();

	foreach ( $data as $k => $v ) {
		if ( isset( $columns[ $k ] ) ) {
			$new_vals[ $k ] = $v;
		}
	}

	$update = $wpdb->update( "{$global_prefix}bbapp_iap_products", $new_vals, array( 'id' => $product_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

	/**
	 * Update iap product.
	 *
	 * @peram int   $product_id  Product id.
	 * @peram array $product  Product ids.
	 *
	 * @since 1.6.7
	 */
	do_action( 'bbapp_after_iap_updated_product', $product_id, $new_vals );

	return ( false !== $update ) ? true : false;
}

/**
 * Add logs into database.
 *
 * @param string $type Log type.
 * @param string $text Log text.
 * @param bool   $ref  Reference provide.
 *
 * @return bool
 */
function bbapp_iap_add_logs( $type, $text, $ref = false ) {
	if ( ! class_exists( 'Logger' ) ) {
		return false;
	}

	return Logger::instance()->add( $type, $text, $ref );
}

/**
 * Function to get store products based on platform.
 *
 * @param string $device_platform Device paltform.
 *
 * @return false|mixed|void
 */
function bbapp_iap_get_store_products( $device_platform ) {
	if ( empty( $device_platform ) ) {
		return array();
	}

	return get_option( "bbapp_{$device_platform}_store_product", array() );
}

/**
 * Function to get store product.
 *
 * @param string $store_product_id Store product id.
 * @param string $device_platform  Platform name.
 * @param bool   $field            Get product field.
 *
 * @return mixed|string
 */
function bbapp_iap_get_store_product( $store_product_id, $device_platform, $field = false ) {
	if ( ! empty( $store_product_id ) && ! empty( $device_platform ) ) {
		$products = bbapp_iap_get_store_products( $device_platform );
		if ( ! empty( $products ) ) {
			$key     = array_search( $store_product_id, array_column( $products, 'id' ), true );
			$product = $products[ $key ];
			if ( ! empty( $field ) ) {
				return ! empty( $product[ $field ] ) ? $product[ $field ] : '';
			}

			return $product;
		}
	}

	return '';
}

/**
 * Get metadata for a given iap item.
 *
 * @param int    $iap_id        ID of the iap item whose metadata is being requested.
 * @param string $meta_key      Optional. If present, only the metadata matching
 *                              that meta key will be returned. Otherwise, all metadata for the
 *                              iap item will be fetched.
 * @param bool   $single        Optional. If true, return only the first value of the
 *                              specified meta_key. This parameter has no effect if meta_key is not
 *                              specified. Default: true.
 *
 * @since 1.4.0
 *
 * @return mixed The meta value(s) being requested.
 */
function bbapp_iap_get_meta( $iap_id = 0, $meta_key = '', $single = true ) {
	global $wpdb;

	// WP_Meta_Query expects the table name at
	// $wpdb->iapmeta.
	$wpdb->iapmeta = \bbapp_iap()->get_global_dbprefix() . 'bbapp_iap_productmeta';

	$retval = get_metadata( 'iap', $iap_id, $meta_key, $single );

	/**
	 * Filters the metadata for a specified iap item.
	 *
	 * @param mixed  $retval   The meta values for the iap item.
	 * @param int    $iap_id   ID of the iap item.
	 * @param string $meta_key Meta key for the value being requested.
	 * @param bool   $single   Whether to return one matched meta key row or all.
	 *
	 * @since 1.4.0
	 */
	return apply_filters( 'bbapp_iap_get_meta', $retval, $iap_id, $meta_key, $single );
}

/**
 * Add a piece of iap metadata.
 *
 * @param int    $iap_id        ID of the iap item.
 * @param string $meta_key      Metadata key.
 * @param mixed  $meta_value    Metadata value.
 * @param bool   $unique        Optional. Whether to enforce a single metadata value for the
 *                              given key. If true, and the object already has a value for
 *                              the key, no change will be made. Default: false.
 *
 * @since 1.4.0
 *
 * @return int|bool The meta ID on successful update, false on failure.
 */
function bbapp_iap_add_meta( $iap_id, $meta_key, $meta_value, $unique = false ) {
	global $wpdb;

	// WP_Meta_Query expects the table name at
	// $wpdb->iapmeta.
	$wpdb->iapmeta = \bbapp_iap()->get_global_dbprefix() . 'bbapp_iap_productmeta';

	$retval = add_metadata( 'iap', $iap_id, $meta_key, $meta_value, $unique );

	return $retval;
}

/**
 * Update a piece of iap meta.
 *
 * @param int    $iap_id        ID of the iap item whose metadata is being updated.
 * @param string $meta_key      Key of the metadata being updated.
 * @param mixed  $meta_value    Value to be set.
 * @param mixed  $prev_value    Optional. If specified, only update existing metadata entries
 *                              with the specified value. Otherwise, update all entries.
 *
 * @since 1.4.0
 *
 * @return bool|int Returns false on failure. On successful update of existing
 *                  metadata, returns true. On successful creation of new metadata,
 *                  returns the integer ID of the new metadata row.
 */
function bbapp_iap_update_meta( $iap_id, $meta_key, $meta_value, $prev_value = '' ) {
	global $wpdb;

	// WP_Meta_Query expects the table name at
	// $wpdb->iapmeta.
	$wpdb->iapmeta = \bbapp_iap()->get_global_dbprefix() . 'bbapp_iap_productmeta';

	$retval = update_metadata( 'iap', $iap_id, $meta_key, $meta_value, $prev_value );

	return $retval;
}

/**
 * Delete a meta entry from the DB for an iap item.
 *
 * @param int    $iap_id        ID of the iap item whose metadata is being deleted.
 * @param string $meta_key      Optional. The key of the metadata being deleted. If
 *                              omitted, all metadata associated with the iap
 *                              item will be deleted.
 * @param string $meta_value    Optional. If present, the metadata will only be
 *                              deleted if the meta_value matches this parameter.
 * @param bool   $delete_all    Optional. If true, delete matching metadata entries
 *                              for all objects, ignoring the specified object_id. Otherwise,
 *                              only delete matching metadata entries for the specified
 *                              iap item. Default: false.
 *
 * @since 1.4.0
 *
 * @return bool True on success, false on failure.
 * @global wpdb  $wpdb          WordPress database abstraction object.
 */
function bbapp_iap_delete_meta( $iap_id, $meta_key = '', $meta_value = '', $delete_all = false ) {
	global $wpdb;

	// Legacy - if no meta_key is passed, delete all for the item.
	if ( empty( $meta_key ) ) {
		$all_meta = bbapp_iap_get_meta( $iap_id );
		$keys     = ! empty( $all_meta ) ? array_keys( $all_meta ) : array();

		// With no meta_key, leave $delete_all.
		$delete_all = false;
	} else {
		$keys = array( $meta_key );
	}

	$retval = true;
	// WP_Meta_Query expects the table name at
	// $wpdb->iapmeta.
	$wpdb->iapmeta = \bbapp_iap()->get_global_dbprefix() . 'bbapp_iap_productmeta';

	foreach ( $keys as $key ) {
		$retval = delete_metadata( 'iap', $iap_id, $key, $meta_value, $delete_all );
	}

	return $retval;
}

/**
 * Returns the iap upload dir.
 *
 * @since 1.4.4
 *
 * @return string
 */
function get_iap_upload_dir() {
	$upload_dir = wp_upload_dir();
	$dir        = $upload_dir['basedir'] . '/bbapp/iap';
	BBAPP_File::create_dir( $dir );
	$dir = $dir . '/' . get_current_blog_id();
	BBAPP_File::create_dir( $dir );
	$dir = str_replace( '/', DIRECTORY_SEPARATOR, $dir );

	return $dir;
}

/**
 * Returns the iap upload url.
 *
 * @since 1.4.4
 *
 * @return string
 */
function get_iap_upload_url() {
	$upload_dir = wp_upload_dir();
	$dir        = $upload_dir['baseurl'] . '/bbapp/iap';
	$dir        = $dir . '/' . get_current_blog_id();

	return $dir;
}

/**
 * Returns the iap files.
 *
 * @param string $filename  Name of the file.
 * @param string $file_hash hash of the file i.e - time.
 *
 * @since 1.4.4
 *
 * @return string
 */
function get_iap_file( $filename, $file_hash = '' ) {
	$files_dir = get_iap_upload_dir();
	$files_url = get_iap_upload_url();
	$value     = ( file_exists( $files_dir . '/' . $filename ) ) ? $files_url . '/' . $filename : '';
	if ( ! empty( $file_hash ) ) {
		$value = ( file_exists( $files_dir . '/' . $filename ) ) ? $files_url . '/' . $filename . '?' . $file_hash : '';
	}

	return $value;
}

/**
 * Returns the iap files.
 *
 * @param int $product_id Id of the IAP product.
 *
 * @since 1.4.4
 *
 * @return string $iap_featured_image
 */
function get_iap_feature_image( $product_id ) {
	$iap_featured_image = '';
	if ( isset( $product_id ) ) {
		$iap_product_image = bbapp_iap_get_meta( $product_id, 'iap_product_image', true );
		if ( ! empty( $iap_product_image ) && is_array( $iap_product_image ) ) {
			$file_name          = key( $iap_product_image );
			$file_hash          = $iap_product_image[ $file_name ];
			$iap_featured_image = get_iap_file( $file_name, $file_hash );
		}
	}

	return $iap_featured_image;
}
